import { a as assert } from './shared.rollup.js';

// Copyright 2013 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Persistent cache storing images in an indexed database on the hard disk.
 */
class ImageCache {
    /**
     * IndexedDB database handle.
     */
    db_ = null;
    /**
     * Initializes the cache database.
     * @param callback Completion callback.
     */
    initialize(callback) {
        // Establish a connection to the database or (re)create it if not available
        // or not up to date. After changing the database's schema, increment
        // DB_VERSION to force database recreating.
        const openRequest = indexedDB.open(DB_NAME, DB_VERSION);
        openRequest.onsuccess = () => {
            this.db_ = openRequest.result;
            callback();
        };
        openRequest.onerror = callback;
        openRequest.onupgradeneeded = () => {
            console.info('Cache database creating or upgrading.');
            const db = openRequest.result;
            if (db.objectStoreNames.contains('metadata')) {
                db.deleteObjectStore('metadata');
            }
            if (db.objectStoreNames.contains('data')) {
                db.deleteObjectStore('data');
            }
            if (db.objectStoreNames.contains('settings')) {
                db.deleteObjectStore('settings');
            }
            db.createObjectStore('metadata', { keyPath: 'key' });
            db.createObjectStore('data', { keyPath: 'key' });
            db.createObjectStore('settings', { keyPath: 'key' });
        };
    }
    /**
     * Sets size of the cache.
     *
     * @param size Size in bytes.
     * @param transaction Transaction to be reused. If not provided, then a new
     *     one is created.
     */
    setCacheSize_(size, transaction) {
        transaction =
            transaction || this.db_.transaction(['settings'], 'readwrite');
        const settingsStore = transaction.objectStore('settings');
        settingsStore.put({ key: 'size', value: size }); // Update asynchronously.
    }
    /**
     * Fetches current size of the cache.
     *
     * @param onSuccess Callback to return the size.
     * @param onFailure Failure callback.
     * @param transaction Transaction to be reused. If not
     *     provided, then a new one is created.
     */
    fetchCacheSize_(onSuccess, onFailure, transaction) {
        transaction = transaction ||
            this.db_.transaction(['settings', 'metadata', 'data'], 'readwrite');
        const settingsStore = transaction.objectStore('settings');
        const sizeRequest = settingsStore.get('size');
        sizeRequest.onsuccess = () => {
            const result = sizeRequest.result;
            if (result) {
                onSuccess(result.value);
            }
            else {
                onSuccess(0);
            }
        };
        sizeRequest.onerror = () => {
            console.warn('Failed to fetch size from the database.');
            onFailure();
        };
    }
    /**
     * Evicts the least used elements in cache to make space for a new image and
     * updates size of the cache taking into account the upcoming item.
     *
     * @param size Requested size.
     * @param onSuccess Success callback.
     * @param onFailure Failure callback.
     * @param dbTransaction Transaction to be reused. If not provided, then a new
     *     one is created.
     */
    evictCache_(size, onSuccess, onFailure, dbTransaction) {
        const transaction = dbTransaction ||
            this.db_.transaction(['settings', 'metadata', 'data'], 'readwrite');
        // Check if the requested size is smaller than the cache size.
        if (size > MEMORY_LIMIT) {
            onFailure();
            return;
        }
        const onCacheSize = (cacheSize) => {
            if (size < MEMORY_LIMIT - cacheSize) {
                // Enough space, no need to evict.
                this.setCacheSize_(cacheSize + size, transaction);
                onSuccess();
                return;
            }
            let bytesToEvict = Math.max(size, EVICTION_CHUNK_SIZE);
            // Fetch all metadata.
            const metadataEntries = [];
            const metadataStore = transaction.objectStore('metadata');
            const dataStore = transaction.objectStore('data');
            const onEntriesFetched = () => {
                metadataEntries.sort((a, b) => {
                    return b.lastLoadTimestamp - a.lastLoadTimestamp;
                });
                let totalEvicted = 0;
                while (bytesToEvict > 0) {
                    const entry = metadataEntries.pop();
                    totalEvicted += entry.size;
                    bytesToEvict -= entry.size;
                    metadataStore.delete(entry.key); // Remove asynchronously.
                    dataStore.delete(entry.key); // Remove asynchronously.
                }
                this.setCacheSize_(cacheSize - totalEvicted + size, transaction);
            };
            const cursor = metadataStore.openCursor();
            cursor.onsuccess = () => {
                const result = cursor.result;
                if (result) {
                    metadataEntries.push(result.value);
                    result.continue();
                }
                else {
                    onEntriesFetched();
                }
            };
        };
        this.fetchCacheSize_(onCacheSize, onFailure, transaction);
    }
    /**
     * Saves an image in the cache.
     *
     * @param key Cache key.
     * @param timestamp Last modification timestamp. Used to detect if the image
     *     cache entry is out of date.
     * @param width Image width.
     * @param height Image height.
     * @param ifd Image ifd, null if none.
     * @param data Image data.
     */
    saveImage(key, timestamp, width, height, ifd, data) {
        if (!this.db_) {
            console.warn('Cache database not available.');
            return;
        }
        const onNotFoundInCache = () => {
            const metadataEntry = {
                key: key,
                timestamp: timestamp,
                width: width,
                height: height,
                ifd: ifd,
                size: data.length,
                lastLoadTimestamp: Date.now(),
            };
            const dataEntry = { key: key, data: data };
            const transaction = this.db_.transaction(['settings', 'metadata', 'data'], 'readwrite');
            const metadataStore = transaction.objectStore('metadata');
            const dataStore = transaction.objectStore('data');
            const onCacheEvicted = () => {
                metadataStore.put(metadataEntry); // Add asynchronously.
                dataStore.put(dataEntry); // Add asynchronously.
            };
            // Make sure there is enough space in the cache.
            this.evictCache_(data.length, onCacheEvicted, () => { }, transaction);
        };
        // Check if the image is already in cache. If not, then save it to
        // cache.
        this.loadImage(key, timestamp, () => { }, onNotFoundInCache);
    }
    /**
     * Loads an image from the cache.
     *
     * @param key Cache key.
     * @param timestamp Last modification timestamp. If different than the one in
     *     cache, then the entry will be invalidated.
     * @param onSuccess Success callback.
     * @param onFailure Failure callback.
     */
    loadImage(key, timestamp, onSuccess, onFailure) {
        if (!this.db_) {
            console.warn('Cache database not available.');
            onFailure();
            return;
        }
        const transaction = this.db_.transaction(['settings', 'metadata', 'data'], 'readwrite');
        const metadataStore = transaction.objectStore('metadata');
        const dataStore = transaction.objectStore('data');
        const metadataRequest = metadataStore.get(key);
        const dataRequest = dataStore.get(key);
        let metadataEntry = null;
        let metadataReceived = false;
        let dataEntry = null;
        let dataReceived = false;
        const onPartialSuccess = () => {
            // Check if all sub-requests have finished.
            if (!metadataReceived || !dataReceived) {
                return;
            }
            // Check if both entries are available or both unavailable.
            if (!!metadataEntry !== !!dataEntry) {
                console.warn('Inconsistent cache database.');
                onFailure();
                return;
            }
            // Process the responses.
            if (!metadataEntry) {
                // The image not found.
                onFailure();
            }
            else if (metadataEntry.timestamp !== timestamp) {
                // The image is not up to date, so remove it.
                this.removeImage(key, () => { }, () => { }, transaction);
                onFailure();
            }
            else {
                // The image is available. Update the last load time and return the
                // image data.
                metadataEntry.lastLoadTimestamp = Date.now();
                metadataStore.put(metadataEntry); // Added asynchronously.
                onSuccess(metadataEntry.width, metadataEntry.height, metadataEntry.ifd, dataEntry.data);
            }
        };
        metadataRequest.onsuccess = () => {
            if (metadataRequest.result) {
                metadataEntry = metadataRequest.result;
            }
            metadataReceived = true;
            onPartialSuccess();
        };
        dataRequest.onsuccess = () => {
            if (dataRequest.result) {
                dataEntry = dataRequest.result;
            }
            dataReceived = true;
            onPartialSuccess();
        };
        metadataRequest.onerror = () => {
            console.warn('Failed to fetch metadata from the database.');
            metadataReceived = true;
            onPartialSuccess();
        };
        dataRequest.onerror = () => {
            console.warn('Failed to fetch image data from the database.');
            dataReceived = true;
            onPartialSuccess();
        };
    }
    /**
     * Removes the image from the cache.
     *
     * @param key Cache key.
     * @param onSuccess Success callback.
     * @param onFailure Failure callback.
     * @param transaction Transaction to be reused. If not provided, then a new
     *     one is created.
     */
    removeImage(key, onSuccess, onFailure, transaction) {
        if (!this.db_) {
            console.warn('Cache database not available.');
            return;
        }
        transaction = transaction ||
            this.db_.transaction(['settings', 'metadata', 'data'], 'readwrite');
        const metadataStore = transaction.objectStore('metadata');
        const dataStore = transaction.objectStore('data');
        let cacheSize = null;
        let cacheSizeReceived = false;
        let metadataEntry = null;
        let metadataReceived = false;
        const onPartialSuccess = () => {
            if (!cacheSizeReceived || !metadataReceived) {
                return;
            }
            // If either cache size or metadata entry is not available, then it is an
            // error.
            if (cacheSize === null || !metadataEntry) {
                if (onFailure) {
                    onFailure();
                }
                return;
            }
            if (onSuccess) {
                onSuccess();
            }
            this.setCacheSize_(cacheSize - metadataEntry.size, transaction);
            metadataStore.delete(key); // Delete asynchronously.
            dataStore.delete(key); // Delete asynchronously.
        };
        const onCacheSizeFailure = () => {
            cacheSizeReceived = true;
        };
        const onCacheSizeSuccess = (result) => {
            cacheSize = result;
            cacheSizeReceived = true;
            onPartialSuccess();
        };
        // Fetch the current cache size.
        this.fetchCacheSize_(onCacheSizeSuccess, onCacheSizeFailure, transaction);
        // Receive image's metadata.
        const metadataRequest = metadataStore.get(key);
        metadataRequest.onsuccess = () => {
            if (metadataRequest.result) {
                metadataEntry = metadataRequest.result;
            }
            metadataReceived = true;
            onPartialSuccess();
        };
        metadataRequest.onerror = () => {
            console.warn('Failed to remove an image.');
            metadataReceived = true;
            onPartialSuccess();
        };
    }
}
/**
 * Cache database name.
 */
const DB_NAME = 'image-loader';
/**
 * Cache database version.
 */
const DB_VERSION = 16;
/**
 * Memory limit for images data in bytes.
 */
const MEMORY_LIMIT = 250 * 1024 * 1024; // 250 MB.
/**
 * Minimal amount of memory freed per eviction. Used to limit number
 * of evictions which are expensive.
 */
const EVICTION_CHUNK_SIZE = 50 * 1024 * 1024; // 50 MB.

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Class representing image orientation.
 */
class ImageOrientation {
    a;
    b;
    c;
    d;
    /**
     * The constructor takes 2x2 matrix value that cancels the image orientation:
     * |a, c|
     * |b, d|
     */
    constructor(a, b, c, d) {
        this.a = a;
        this.b = b;
        this.c = c;
        this.d = d;
    }
    /**
     * @param orientation 1-based orientation number defined by EXIF.
     */
    static fromExifOrientation(orientation) {
        switch (~~orientation) {
            case 1:
                return new ImageOrientation(1, 0, 0, 1);
            case 2:
                return new ImageOrientation(-1, 0, 0, 1);
            case 3:
                return new ImageOrientation(-1, 0, 0, -1);
            case 4:
                return new ImageOrientation(1, 0, 0, -1);
            case 5:
                return new ImageOrientation(0, 1, 1, 0);
            case 6:
                return new ImageOrientation(0, 1, -1, 0);
            case 7:
                return new ImageOrientation(0, -1, -1, 0);
            case 8:
                return new ImageOrientation(0, -1, 1, 0);
            default:
                console.error(`Invalid orientation number: ${orientation}`);
                return new ImageOrientation(1, 0, 0, 1);
        }
    }
    /**
     * @param rotation90 Clockwise degrees / 90.
     */
    static fromClockwiseRotation(rotation90) {
        switch (~~(rotation90 % 4)) {
            case 0:
                return new ImageOrientation(1, 0, 0, 1);
            case 1:
            case -3:
                return new ImageOrientation(0, 1, -1, 0);
            case 2:
            case -2:
                return new ImageOrientation(-1, 0, 0, -1);
            case 3:
            case -1:
                return new ImageOrientation(0, -1, 1, 0);
            default:
                console.error(`Invalid rotation number: ${rotation90}`);
                return new ImageOrientation(1, 0, 0, 1);
        }
    }
    /** Builds a transformation matrix from the image transform parameters. */
    static fromRotationAndScale(transform) {
        const scaleX = transform.scaleX;
        const scaleY = transform.scaleY;
        const rotate90 = transform.rotate90;
        const orientation = ImageOrientation.fromClockwiseRotation(rotate90);
        // Flip X and Y.
        // In the Files app., CSS transformations are applied like
        // "transform: rotate(90deg) scaleX(-1)".
        // Since the image is scaled based on the X,Y axes pinned to the original,
        // it is equivalent to scale first and then rotate.
        // |a c| |s_x 0 | |x|   |a*s_x c*s_y| |x|
        // |b d| | 0 s_y| |y| = |b*s_x d*s_y| |y|
        return new ImageOrientation(orientation.a * scaleX, orientation.b * scaleX, orientation.c * scaleY, orientation.d * scaleY);
    }
    /** Obtains the image size after cancelling its orientation. */
    getSizeAfterCancelling(imageWidth, imageHeight) {
        const projectedX = this.a * imageWidth + this.c * imageHeight;
        const projectedY = this.b * imageWidth + this.d * imageHeight;
        return {
            width: Math.abs(projectedX),
            height: Math.abs(projectedY),
        };
    }
    /**
     * Applies the transformation that cancels the image orientation to the given
     * context.
     */
    cancelImageOrientation(context, imageWidth, imageHeight) {
        // Calculate where to project the point of (imageWidth, imageHeight).
        const projectedX = this.a * imageWidth + this.c * imageHeight;
        const projectedY = this.b * imageWidth + this.d * imageHeight;
        // If the projected point coordinates are negative, add offset to cancel it.
        const offsetX = projectedX < 0 ? -projectedX : 0;
        const offsetY = projectedY < 0 ? -projectedY : 0;
        // Apply the transform.
        context.setTransform(this.a, this.b, this.c, this.d, offsetX, offsetY);
    }
    /**
     * Checks if the orientation represents identity transformation or not.
     */
    isIdentity() {
        return this.a === 1 && this.b === 0 && this.c === 0 && this.d === 1;
    }
}
function isImageOrientation(orientation) {
    return !!orientation && 'a' in orientation && 'b' in orientation &&
        'c' in orientation && 'd' in orientation;
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/* This file is generated from:
 *  ../../ui/file_manager/base/gn/file_types.json5
 */
/**
 * @typedef {{
 *   translationKey: !string,
 *   type: !string,
 *   icon: (string|undefined),
 *   subtype: !string,
 *   extensions: (!Array<!string>|undefined),
 *   mime: (string|undefined),
 *   encrypted: (boolean|undefined),
 *   originalMimeType: (string|undefined)
 * }}
 */
// @ts-ignore:  error TS7005: Variable 'FileExtensionType' implicitly has an 'any' type.
/**
 * Maps a file extension to a FileExtensionType.
 * Note: If an extension can match multiple types, in this map contains
 * only the first occurrence.
 *
 * @const Map<string, !FileExtensionType>
 */
const EXTENSION_TO_TYPE = new Map([
    [".jpeg", {
            "extensions": [
                ".jpeg",
                ".jpg",
                ".jfif",
                ".pjpeg",
                ".pjp"
            ],
            "mime": "image/jpeg",
            "subtype": "JPEG",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "image"
        }],
    [".jpg", {
            "extensions": [
                ".jpeg",
                ".jpg",
                ".jfif",
                ".pjpeg",
                ".pjp"
            ],
            "mime": "image/jpeg",
            "subtype": "JPEG",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "image"
        }],
    [".jfif", {
            "extensions": [
                ".jpeg",
                ".jpg",
                ".jfif",
                ".pjpeg",
                ".pjp"
            ],
            "mime": "image/jpeg",
            "subtype": "JPEG",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "image"
        }],
    [".pjpeg", {
            "extensions": [
                ".jpeg",
                ".jpg",
                ".jfif",
                ".pjpeg",
                ".pjp"
            ],
            "mime": "image/jpeg",
            "subtype": "JPEG",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "image"
        }],
    [".pjp", {
            "extensions": [
                ".jpeg",
                ".jpg",
                ".jfif",
                ".pjpeg",
                ".pjp"
            ],
            "mime": "image/jpeg",
            "subtype": "JPEG",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "image"
        }],
    [".bmp", {
            "extensions": [
                ".bmp"
            ],
            "mime": "image/bmp",
            "subtype": "BMP",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "image"
        }],
    [".gif", {
            "extensions": [
                ".gif"
            ],
            "mime": "image/gif",
            "subtype": "GIF",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "image"
        }],
    [".ico", {
            "extensions": [
                ".ico"
            ],
            "mime": "image/x-icon",
            "subtype": "ICO",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "image"
        }],
    [".png", {
            "extensions": [
                ".png"
            ],
            "mime": "image/png",
            "subtype": "PNG",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "image"
        }],
    [".webp", {
            "extensions": [
                ".webp"
            ],
            "mime": "image/webp",
            "subtype": "WebP",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "image"
        }],
    [".tif", {
            "extensions": [
                ".tif",
                ".tiff"
            ],
            "mime": "image/tiff",
            "subtype": "TIFF",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "image"
        }],
    [".tiff", {
            "extensions": [
                ".tif",
                ".tiff"
            ],
            "mime": "image/tiff",
            "subtype": "TIFF",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "image"
        }],
    [".svg", {
            "extensions": [
                ".svg",
                ".svgz"
            ],
            "mime": "image/svg+xml",
            "subtype": "SVG",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "image"
        }],
    [".svgz", {
            "extensions": [
                ".svg",
                ".svgz"
            ],
            "mime": "image/svg+xml",
            "subtype": "SVG",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "image"
        }],
    [".avif", {
            "extensions": [
                ".avif"
            ],
            "mime": "image/avif",
            "subtype": "AVIF",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "image"
        }],
    [".jxl", {
            "extensions": [
                ".jxl"
            ],
            "mime": "image/jxl",
            "subtype": "JXL",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "image"
        }],
    [".xbm", {
            "extensions": [
                ".xbm"
            ],
            "mime": "image/x-xbitmap",
            "subtype": "XBM",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "image"
        }],
    [".arw", {
            "extensions": [
                ".arw"
            ],
            "icon": "image",
            "mime": "image/x-sony-arw",
            "subtype": "ARW",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "raw"
        }],
    [".cr2", {
            "extensions": [
                ".cr2"
            ],
            "icon": "image",
            "mime": "image/x-canon-cr2",
            "subtype": "CR2",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "raw"
        }],
    [".dng", {
            "extensions": [
                ".dng"
            ],
            "icon": "image",
            "mime": "image/x-adobe-dng",
            "subtype": "DNG",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "raw"
        }],
    [".nef", {
            "extensions": [
                ".nef"
            ],
            "icon": "image",
            "mime": "image/x-nikon-nef",
            "subtype": "NEF",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "raw"
        }],
    [".nrw", {
            "extensions": [
                ".nrw"
            ],
            "icon": "image",
            "mime": "image/x-nikon-nrw",
            "subtype": "NRW",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "raw"
        }],
    [".orf", {
            "extensions": [
                ".orf"
            ],
            "icon": "image",
            "mime": "image/x-olympus-orf",
            "subtype": "ORF",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "raw"
        }],
    [".raf", {
            "extensions": [
                ".raf"
            ],
            "icon": "image",
            "mime": "image/x-fuji-raf",
            "subtype": "RAF",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "raw"
        }],
    [".rw2", {
            "extensions": [
                ".rw2"
            ],
            "icon": "image",
            "mime": "image/x-panasonic-rw2",
            "subtype": "RW2",
            "translationKey": "IMAGE_FILE_TYPE",
            "type": "raw"
        }],
    [".3gp", {
            "extensions": [
                ".3gp",
                ".3gpp"
            ],
            "mime": "video/3gpp",
            "subtype": "3GP",
            "translationKey": "VIDEO_FILE_TYPE",
            "type": "video"
        }],
    [".3gpp", {
            "extensions": [
                ".3gp",
                ".3gpp"
            ],
            "mime": "video/3gpp",
            "subtype": "3GP",
            "translationKey": "VIDEO_FILE_TYPE",
            "type": "video"
        }],
    [".avi", {
            "extensions": [
                ".avi"
            ],
            "mime": "video/x-msvideo",
            "subtype": "AVI",
            "translationKey": "VIDEO_FILE_TYPE",
            "type": "video"
        }],
    [".mov", {
            "extensions": [
                ".mov"
            ],
            "mime": "video/quicktime",
            "subtype": "QuickTime",
            "translationKey": "VIDEO_FILE_TYPE",
            "type": "video"
        }],
    [".mkv", {
            "extensions": [
                ".mkv"
            ],
            "mime": "video/x-matroska",
            "subtype": "MKV",
            "translationKey": "VIDEO_FILE_TYPE",
            "type": "video"
        }],
    [".mp4", {
            "extensions": [
                ".mp4",
                ".m4v",
                ".mpg4",
                ".mpeg4"
            ],
            "mime": "video/mp4",
            "subtype": "MPEG",
            "translationKey": "VIDEO_FILE_TYPE",
            "type": "video"
        }],
    [".m4v", {
            "extensions": [
                ".mp4",
                ".m4v",
                ".mpg4",
                ".mpeg4"
            ],
            "mime": "video/mp4",
            "subtype": "MPEG",
            "translationKey": "VIDEO_FILE_TYPE",
            "type": "video"
        }],
    [".mpg4", {
            "extensions": [
                ".mp4",
                ".m4v",
                ".mpg4",
                ".mpeg4"
            ],
            "mime": "video/mp4",
            "subtype": "MPEG",
            "translationKey": "VIDEO_FILE_TYPE",
            "type": "video"
        }],
    [".mpeg4", {
            "extensions": [
                ".mp4",
                ".m4v",
                ".mpg4",
                ".mpeg4"
            ],
            "mime": "video/mp4",
            "subtype": "MPEG",
            "translationKey": "VIDEO_FILE_TYPE",
            "type": "video"
        }],
    [".mpg", {
            "extensions": [
                ".mpg",
                ".mpeg"
            ],
            "mime": "video/mpeg",
            "subtype": "MPEG",
            "translationKey": "VIDEO_FILE_TYPE",
            "type": "video"
        }],
    [".mpeg", {
            "extensions": [
                ".mpg",
                ".mpeg"
            ],
            "mime": "video/mpeg",
            "subtype": "MPEG",
            "translationKey": "VIDEO_FILE_TYPE",
            "type": "video"
        }],
    [".ogm", {
            "extensions": [
                ".ogm",
                ".ogv",
                ".ogx"
            ],
            "mime": "video/ogg",
            "subtype": "OGG",
            "translationKey": "VIDEO_FILE_TYPE",
            "type": "video"
        }],
    [".ogv", {
            "extensions": [
                ".ogm",
                ".ogv",
                ".ogx"
            ],
            "mime": "video/ogg",
            "subtype": "OGG",
            "translationKey": "VIDEO_FILE_TYPE",
            "type": "video"
        }],
    [".ogx", {
            "extensions": [
                ".ogm",
                ".ogv",
                ".ogx"
            ],
            "mime": "video/ogg",
            "subtype": "OGG",
            "translationKey": "VIDEO_FILE_TYPE",
            "type": "video"
        }],
    [".webm", {
            "extensions": [
                ".webm"
            ],
            "mime": "video/webm",
            "subtype": "WebM",
            "translationKey": "VIDEO_FILE_TYPE",
            "type": "video"
        }],
    [".amr", {
            "extensions": [
                ".amr"
            ],
            "mime": "audio/amr",
            "subtype": "AMR",
            "translationKey": "AUDIO_FILE_TYPE",
            "type": "audio"
        }],
    [".flac", {
            "extensions": [
                ".flac"
            ],
            "mime": "audio/flac",
            "subtype": "FLAC",
            "translationKey": "AUDIO_FILE_TYPE",
            "type": "audio"
        }],
    [".mp3", {
            "extensions": [
                ".mp3"
            ],
            "mime": "audio/mpeg",
            "subtype": "MP3",
            "translationKey": "AUDIO_FILE_TYPE",
            "type": "audio"
        }],
    [".m4a", {
            "extensions": [
                ".m4a"
            ],
            "mime": "audio/mp4a-latm",
            "subtype": "MPEG",
            "translationKey": "AUDIO_FILE_TYPE",
            "type": "audio"
        }],
    [".oga", {
            "extensions": [
                ".oga",
                ".ogg",
                ".opus"
            ],
            "mime": "audio/ogg",
            "subtype": "OGG",
            "translationKey": "AUDIO_FILE_TYPE",
            "type": "audio"
        }],
    [".ogg", {
            "extensions": [
                ".oga",
                ".ogg",
                ".opus"
            ],
            "mime": "audio/ogg",
            "subtype": "OGG",
            "translationKey": "AUDIO_FILE_TYPE",
            "type": "audio"
        }],
    [".opus", {
            "extensions": [
                ".oga",
                ".ogg",
                ".opus"
            ],
            "mime": "audio/ogg",
            "subtype": "OGG",
            "translationKey": "AUDIO_FILE_TYPE",
            "type": "audio"
        }],
    [".wav", {
            "extensions": [
                ".wav"
            ],
            "mime": "audio/x-wav",
            "subtype": "WAV",
            "translationKey": "AUDIO_FILE_TYPE",
            "type": "audio"
        }],
    [".weba", {
            "extensions": [
                ".weba"
            ],
            "mime": "audio/webm",
            "subtype": "WEBA",
            "translationKey": "AUDIO_FILE_TYPE",
            "type": "audio"
        }],
    [".txt", {
            "extensions": [
                ".txt",
                ".text"
            ],
            "mime": "text/plain",
            "subtype": "TXT",
            "translationKey": "PLAIN_TEXT_FILE_TYPE",
            "type": "text"
        }],
    [".text", {
            "extensions": [
                ".txt",
                ".text"
            ],
            "mime": "text/plain",
            "subtype": "TXT",
            "translationKey": "PLAIN_TEXT_FILE_TYPE",
            "type": "text"
        }],
    [".csv", {
            "extensions": [
                ".csv"
            ],
            "mime": "text/csv",
            "subtype": "CSV",
            "translationKey": "CSV_TEXT_FILE_TYPE",
            "type": "text"
        }],
    [".zip", {
            "extensions": [
                ".zip"
            ],
            "mime": "application/zip",
            "subtype": "ZIP",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".rar", {
            "extensions": [
                ".rar"
            ],
            "mime": "application/x-rar-compressed",
            "subtype": "RAR",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".iso", {
            "extensions": [
                ".iso"
            ],
            "mime": "application/x-iso9660-image",
            "subtype": "ISO",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".7z", {
            "extensions": [
                ".7z"
            ],
            "mime": "application/x-7z-compressed",
            "subtype": "7-Zip",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".crx", {
            "extensions": [
                ".crx"
            ],
            "mime": "application/x-chrome-extension",
            "subtype": "CRX",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".tar", {
            "extensions": [
                ".tar"
            ],
            "mime": "application/x-tar",
            "subtype": "TAR",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".bz2", {
            "extensions": [
                ".bz2",
                ".bz",
                ".tbz",
                ".tbz2",
                ".tz2",
                ".tb2"
            ],
            "mime": "application/x-bzip2",
            "subtype": "BZIP2",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".bz", {
            "extensions": [
                ".bz2",
                ".bz",
                ".tbz",
                ".tbz2",
                ".tz2",
                ".tb2"
            ],
            "mime": "application/x-bzip2",
            "subtype": "BZIP2",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".tbz", {
            "extensions": [
                ".bz2",
                ".bz",
                ".tbz",
                ".tbz2",
                ".tz2",
                ".tb2"
            ],
            "mime": "application/x-bzip2",
            "subtype": "BZIP2",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".tbz2", {
            "extensions": [
                ".bz2",
                ".bz",
                ".tbz",
                ".tbz2",
                ".tz2",
                ".tb2"
            ],
            "mime": "application/x-bzip2",
            "subtype": "BZIP2",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".tz2", {
            "extensions": [
                ".bz2",
                ".bz",
                ".tbz",
                ".tbz2",
                ".tz2",
                ".tb2"
            ],
            "mime": "application/x-bzip2",
            "subtype": "BZIP2",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".tb2", {
            "extensions": [
                ".bz2",
                ".bz",
                ".tbz",
                ".tbz2",
                ".tz2",
                ".tb2"
            ],
            "mime": "application/x-bzip2",
            "subtype": "BZIP2",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".gz", {
            "extensions": [
                ".gz",
                ".tgz"
            ],
            "mime": "application/x-gzip",
            "subtype": "GZIP",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".tgz", {
            "extensions": [
                ".gz",
                ".tgz"
            ],
            "mime": "application/x-gzip",
            "subtype": "GZIP",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".lz", {
            "extensions": [
                ".lz"
            ],
            "mime": "application/x-lzip",
            "subtype": "LZIP",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".lzo", {
            "extensions": [
                ".lzo"
            ],
            "mime": "application/x-lzop",
            "subtype": "LZOP",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".lzma", {
            "extensions": [
                ".lzma",
                ".tlzma",
                ".tlz"
            ],
            "mime": "application/x-lzma",
            "subtype": "LZMA",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".tlzma", {
            "extensions": [
                ".lzma",
                ".tlzma",
                ".tlz"
            ],
            "mime": "application/x-lzma",
            "subtype": "LZMA",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".tlz", {
            "extensions": [
                ".lzma",
                ".tlzma",
                ".tlz"
            ],
            "mime": "application/x-lzma",
            "subtype": "LZMA",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".xz", {
            "extensions": [
                ".xz",
                ".txz"
            ],
            "mime": "application/x-xz",
            "subtype": "XZ",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".txz", {
            "extensions": [
                ".xz",
                ".txz"
            ],
            "mime": "application/x-xz",
            "subtype": "XZ",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".z", {
            "extensions": [
                ".z",
                ".taz",
                ".tz"
            ],
            "mime": "application/x-compress",
            "subtype": "Z",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".taz", {
            "extensions": [
                ".z",
                ".taz",
                ".tz"
            ],
            "mime": "application/x-compress",
            "subtype": "Z",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".tz", {
            "extensions": [
                ".z",
                ".taz",
                ".tz"
            ],
            "mime": "application/x-compress",
            "subtype": "Z",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".zst", {
            "extensions": [
                ".zst",
                ".tzst"
            ],
            "mime": "application/zstd",
            "subtype": "Zstandard",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".tzst", {
            "extensions": [
                ".zst",
                ".tzst"
            ],
            "mime": "application/zstd",
            "subtype": "Zstandard",
            "translationKey": "ARCHIVE_FILE_TYPE",
            "type": "archive"
        }],
    [".gdoc", {
            "extensions": [
                ".gdoc"
            ],
            "icon": "gdoc",
            "mime": "application/vnd.google-apps.document",
            "subtype": "doc",
            "translationKey": "GDOC_DOCUMENT_FILE_TYPE",
            "type": "hosted"
        }],
    [".gsheet", {
            "extensions": [
                ".gsheet"
            ],
            "icon": "gsheet",
            "mime": "application/vnd.google-apps.spreadsheet",
            "subtype": "sheet",
            "translationKey": "GSHEET_DOCUMENT_FILE_TYPE",
            "type": "hosted"
        }],
    [".gslides", {
            "extensions": [
                ".gslides"
            ],
            "icon": "gslides",
            "mime": "application/vnd.google-apps.presentation",
            "subtype": "slides",
            "translationKey": "GSLIDES_DOCUMENT_FILE_TYPE",
            "type": "hosted"
        }],
    [".gdraw", {
            "extensions": [
                ".gdraw"
            ],
            "icon": "gdraw",
            "mime": "application/vnd.google-apps.drawing",
            "subtype": "draw",
            "translationKey": "GDRAW_DOCUMENT_FILE_TYPE",
            "type": "hosted"
        }],
    [".gtable", {
            "extensions": [
                ".gtable"
            ],
            "icon": "gtable",
            "mime": "application/vnd.google-apps.fusiontable",
            "subtype": "table",
            "translationKey": "GTABLE_DOCUMENT_FILE_TYPE",
            "type": "hosted"
        }],
    [".glink", {
            "extensions": [
                ".glink"
            ],
            "icon": "glink",
            "mime": "application/vnd.google-apps.shortcut",
            "subtype": "glink",
            "translationKey": "GLINK_DOCUMENT_FILE_TYPE",
            "type": "hosted"
        }],
    [".gform", {
            "extensions": [
                ".gform"
            ],
            "icon": "gform",
            "mime": "application/vnd.google-apps.form",
            "subtype": "form",
            "translationKey": "GFORM_DOCUMENT_FILE_TYPE",
            "type": "hosted"
        }],
    [".gmap", {
            "extensions": [
                ".gmap"
            ],
            "icon": "gmap",
            "mime": "application/vnd.google-apps.map",
            "subtype": "map",
            "translationKey": "GMAP_DOCUMENT_FILE_TYPE",
            "type": "hosted"
        }],
    [".gsite", {
            "extensions": [
                ".gsite"
            ],
            "icon": "gsite",
            "mime": "application/vnd.google-apps.site",
            "subtype": "site",
            "translationKey": "GSITE_DOCUMENT_FILE_TYPE",
            "type": "hosted"
        }],
    [".gmaillayout", {
            "extensions": [
                ".gmaillayout"
            ],
            "icon": "gmaillayout",
            "mime": "application/vnd.google-apps.mail-layout",
            "subtype": "emaillayouts",
            "translationKey": "EMAIL_LAYOUTS_DOCUMENT_FILE_TYPE",
            "type": "hosted"
        }],
    [".pdf", {
            "extensions": [
                ".pdf"
            ],
            "icon": "pdf",
            "mime": "application/pdf",
            "subtype": "PDF",
            "translationKey": "PDF_DOCUMENT_FILE_TYPE",
            "type": "document"
        }],
    [".htm", {
            "extensions": [
                ".htm",
                ".html",
                ".mht",
                ".mhtml",
                ".shtml",
                ".xht",
                ".xhtml"
            ],
            "mime": "text/html",
            "subtype": "HTML",
            "translationKey": "HTML_DOCUMENT_FILE_TYPE",
            "type": "document"
        }],
    [".html", {
            "extensions": [
                ".htm",
                ".html",
                ".mht",
                ".mhtml",
                ".shtml",
                ".xht",
                ".xhtml"
            ],
            "mime": "text/html",
            "subtype": "HTML",
            "translationKey": "HTML_DOCUMENT_FILE_TYPE",
            "type": "document"
        }],
    [".mht", {
            "extensions": [
                ".htm",
                ".html",
                ".mht",
                ".mhtml",
                ".shtml",
                ".xht",
                ".xhtml"
            ],
            "mime": "text/html",
            "subtype": "HTML",
            "translationKey": "HTML_DOCUMENT_FILE_TYPE",
            "type": "document"
        }],
    [".mhtml", {
            "extensions": [
                ".htm",
                ".html",
                ".mht",
                ".mhtml",
                ".shtml",
                ".xht",
                ".xhtml"
            ],
            "mime": "text/html",
            "subtype": "HTML",
            "translationKey": "HTML_DOCUMENT_FILE_TYPE",
            "type": "document"
        }],
    [".shtml", {
            "extensions": [
                ".htm",
                ".html",
                ".mht",
                ".mhtml",
                ".shtml",
                ".xht",
                ".xhtml"
            ],
            "mime": "text/html",
            "subtype": "HTML",
            "translationKey": "HTML_DOCUMENT_FILE_TYPE",
            "type": "document"
        }],
    [".xht", {
            "extensions": [
                ".htm",
                ".html",
                ".mht",
                ".mhtml",
                ".shtml",
                ".xht",
                ".xhtml"
            ],
            "mime": "text/html",
            "subtype": "HTML",
            "translationKey": "HTML_DOCUMENT_FILE_TYPE",
            "type": "document"
        }],
    [".xhtml", {
            "extensions": [
                ".htm",
                ".html",
                ".mht",
                ".mhtml",
                ".shtml",
                ".xht",
                ".xhtml"
            ],
            "mime": "text/html",
            "subtype": "HTML",
            "translationKey": "HTML_DOCUMENT_FILE_TYPE",
            "type": "document"
        }],
    [".doc", {
            "extensions": [
                ".doc"
            ],
            "icon": "word",
            "mime": "application/msword",
            "subtype": "Word",
            "translationKey": "WORD_DOCUMENT_FILE_TYPE",
            "type": "document"
        }],
    [".docx", {
            "extensions": [
                ".docx"
            ],
            "icon": "word",
            "mime": "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
            "subtype": "Word",
            "translationKey": "WORD_DOCUMENT_FILE_TYPE",
            "type": "document"
        }],
    [".ppt", {
            "extensions": [
                ".ppt"
            ],
            "icon": "ppt",
            "mime": "application/vnd.ms-powerpoint",
            "subtype": "PPT",
            "translationKey": "POWERPOINT_PRESENTATION_FILE_TYPE",
            "type": "document"
        }],
    [".pptx", {
            "extensions": [
                ".pptx"
            ],
            "icon": "ppt",
            "mime": "application/vnd.openxmlformats-officedocument.presentationml.presentation",
            "subtype": "PPT",
            "translationKey": "POWERPOINT_PRESENTATION_FILE_TYPE",
            "type": "document"
        }],
    [".xls", {
            "extensions": [
                ".xls"
            ],
            "icon": "excel",
            "mime": "application/vnd.ms-excel",
            "subtype": "Excel",
            "translationKey": "EXCEL_FILE_TYPE",
            "type": "document"
        }],
    [".xlsx", {
            "extensions": [
                ".xlsx"
            ],
            "icon": "excel",
            "mime": "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
            "subtype": "Excel",
            "translationKey": "EXCEL_FILE_TYPE",
            "type": "document"
        }],
    [".xlsm", {
            "extensions": [
                ".xlsm"
            ],
            "icon": "excel",
            "mime": "application/vnd.ms-excel.sheet.macroEnabled.12",
            "subtype": "Excel",
            "translationKey": "EXCEL_FILE_TYPE",
            "type": "document"
        }],
    [".tini", {
            "extensions": [
                ".tini"
            ],
            "icon": "tini",
            "subtype": "TGZ",
            "translationKey": "TINI_FILE_TYPE",
            "type": "archive"
        }],
]);

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * A special placeholder for unknown types with no extension.
 */
const PLACEHOLDER = {
    translationKey: 'NO_EXTENSION_FILE_TYPE',
    type: 'UNKNOWN',
    icon: '',
    subtype: '',
    extensions: undefined,
    mime: undefined,
    encrypted: undefined,
    originalMimeType: undefined,
};
/**
 * Returns the final extension of a file name, check for the last two dots
 * to distinguish extensions like ".tar.gz" and ".gz".
 */
function getFinalExtension(fileName) {
    if (!fileName) {
        return '';
    }
    const lowerCaseFileName = fileName.toLowerCase();
    const parts = lowerCaseFileName.split('.');
    // No dot, so no extension.
    if (parts.length === 1) {
        return '';
    }
    // Only one dot, so only 1 extension.
    if (parts.length === 2) {
        return `.${parts.pop()}`;
    }
    // More than 1 dot/extension: e.g. ".tar.gz".
    const last = `.${parts.pop()}`;
    const secondLast = `.${parts.pop()}`;
    const doubleExtension = `${secondLast}${last}`;
    if (EXTENSION_TO_TYPE.has(doubleExtension)) {
        return doubleExtension;
    }
    // Double extension doesn't exist in the map, return the single one.
    return last;
}
/**
 * Gets the file type object for a given file name (base name). Use getType()
 * if possible, since this method can't recognize directories.
 */
function getFileTypeForName(name) {
    const extension = getFinalExtension(name);
    if (EXTENSION_TO_TYPE.has(extension)) {
        return EXTENSION_TO_TYPE.get(extension);
    }
    // Unknown file type.
    if (extension === '') {
        return PLACEHOLDER;
    }
    // subtype is the extension excluding the first dot.
    return {
        translationKey: 'GENERIC_FILE_TYPE',
        type: 'UNKNOWN',
        subtype: extension.substr(1).toUpperCase(),
        icon: '',
        extensions: undefined,
        mime: undefined,
        encrypted: undefined,
        originalMimeType: undefined,
    };
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Checks if the options on the request contain any image processing.
 *
 * @param width Source width.
 * @param height Source height.
 * @param request The request, containing resizing options.
 * @return True if yes, false if not.
 */
function shouldProcess(width, height, request) {
    const targetDimensions = resizeDimensions(width, height, request);
    // Dimensions has to be adjusted.
    if (targetDimensions.width !== width || targetDimensions.height !== height) {
        return true;
    }
    // Orientation has to be adjusted.
    if (isImageOrientation(request.orientation) &&
        !request.orientation.isIdentity()) {
        return true;
    }
    // No changes required.
    return false;
}
/**
 * Calculates dimensions taking into account resize options, such as:
 * - scale: for scaling,
 * - maxWidth, maxHeight: for maximum dimensions,
 * - width, height: for exact requested size.
 * Returns the target size as hash array with width, height properties.
 *
 * @param width Source width.
 * @param height Source height.
 * @param request The request, containing resizing options.
 * @return Dimensions.
 */
function resizeDimensions(width, height, request) {
    const scale = request.scale || 1;
    assert(isImageOrientation(request.orientation));
    const targetDimensions = request.orientation.getSizeAfterCancelling(width * scale, height * scale);
    let targetWidth = targetDimensions.width;
    let targetHeight = targetDimensions.height;
    if (request.maxWidth && targetWidth > request.maxWidth) {
        const scale = request.maxWidth / targetWidth;
        targetWidth *= scale;
        targetHeight *= scale;
    }
    if (request.maxHeight && targetHeight > request.maxHeight) {
        const scale = request.maxHeight / targetHeight;
        targetWidth *= scale;
        targetHeight *= scale;
    }
    if (request.width) {
        targetWidth = request.width;
    }
    if (request.height) {
        targetHeight = request.height;
    }
    targetWidth = Math.round(targetWidth);
    targetHeight = Math.round(targetHeight);
    return { width: targetWidth, height: targetHeight };
}
/**
 * Performs resizing and cropping of the source image into the target canvas.
 *
 * @param source Source image or canvas.
 * @param target Target canvas.
 * @param request The request, containing resizing options.
 */
function resizeAndCrop(source, target, request) {
    // Calculates copy parameters.
    const copyParameters = calculateCopyParameters(source, request);
    target.width = copyParameters.canvas.width;
    target.height = copyParameters.canvas.height;
    // Apply.
    const targetContext = target.getContext('2d');
    targetContext.save();
    assert(isImageOrientation(request.orientation));
    request.orientation.cancelImageOrientation(targetContext, copyParameters.target.width, copyParameters.target.height);
    targetContext.drawImage(source, copyParameters.source.x, copyParameters.source.y, copyParameters.source.width, copyParameters.source.height, copyParameters.target.x, copyParameters.target.y, copyParameters.target.width, copyParameters.target.height);
    targetContext.restore();
}
/**
 * Calculates copy parameters.
 *
 * @param source Source image or canvas.
 * @param request The request, containing resizing options.
 * @return Calculated copy parameters.
 */
function calculateCopyParameters(source, request) {
    if (request.crop) {
        // When an image is cropped, target should be a fixed size square.
        assert(request.width);
        assert(request.height);
        assert(request.width === request.height);
        // The length of shorter edge becomes dimension of cropped area in the
        // source.
        const cropSourceDimension = Math.min(source.width, source.height);
        return {
            source: {
                x: Math.floor((source.width / 2) - (cropSourceDimension / 2)),
                y: Math.floor((source.height / 2) - (cropSourceDimension / 2)),
                width: cropSourceDimension,
                height: cropSourceDimension,
            },
            target: {
                x: 0,
                y: 0,
                width: request.width,
                height: request.height,
            },
            canvas: {
                width: request.width,
                height: request.height,
            },
        };
    }
    // Target dimension is calculated in the rotated(transformed) coordinate.
    const targetCanvasDimensions = resizeDimensions(source.width, source.height, request);
    assert(isImageOrientation(request.orientation));
    const targetDimensions = request.orientation.getSizeAfterCancelling(targetCanvasDimensions.width, targetCanvasDimensions.height);
    return {
        source: {
            x: 0,
            y: 0,
            width: source.width,
            height: source.height,
        },
        target: {
            x: 0,
            y: 0,
            width: targetDimensions.width,
            height: targetDimensions.height,
        },
        canvas: {
            width: targetCanvasDimensions.width,
            height: targetCanvasDimensions.height,
        },
    };
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Response status.
 */
var LoadImageResponseStatus;
(function (LoadImageResponseStatus) {
    LoadImageResponseStatus["SUCCESS"] = "success";
    LoadImageResponseStatus["ERROR"] = "error";
})(LoadImageResponseStatus || (LoadImageResponseStatus = {}));
/**
 * Structure of the response object passed to the LoadImageRequest callback.
 * All methods must be static since this is passed between isolated contexts.
 */
class LoadImageResponse {
    status;
    taskId;
    width;
    height;
    ifd;
    /** The (compressed) image data as a data URL.  */
    data;
    /** @param taskId or null if fulfilled by the client-side cache.  */
    constructor(status, taskId, result) {
        this.status = status;
        this.taskId = taskId;
        if (status === LoadImageResponseStatus.ERROR) {
            return;
        }
        // Response result defined only when status === SUCCESS.
        assert(result);
        this.width = result.width;
        this.height = result.height;
        this.ifd = result.ifd;
        this.data = result.data;
    }
    /**
     * Returns the cacheable result value for |response|, or null for an error.
     *
     * @param response Response data from the ImageLoader.
     * @param timestamp The request timestamp. If undefined, then null is used.
     *     Currently this disables any caching in the ImageLoader, but disables
     *     only *expiration* in the client unless a timestamp is presented on a
     *     later request.
     */
    static cacheValue(response, timestamp) {
        if (!response || response.status === LoadImageResponseStatus.ERROR) {
            return null;
        }
        // Response result defined only when status === SUCCESS.
        assert(response.width);
        assert(response.height);
        assert(response.data);
        return {
            timestamp: timestamp || null,
            width: response.width,
            height: response.height,
            ifd: response.ifd,
            data: response.data,
        };
    }
}
/**
 * Creates a cache key.
 *
 * @return Cache key. It may be null if the cache does not support the request.
 *     e.g. Data URI.
 */
function cacheKey(request) {
    if (/^data:/i.test(request.url ?? '')) {
        return null;
    }
    return JSON.stringify({
        url: request.url,
        orientation: request.orientation,
        scale: request.scale,
        width: request.width,
        height: request.height,
        maxWidth: request.maxWidth,
        maxHeight: request.maxHeight,
    });
}

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Module defined by 'piex.js.wasm' script upon initialization.
 */
let PiexModule;
/**
 * Module constructor defined by 'piex.js.wasm' script.
 */
const initPiexModule = globalThis.createPiexModule;
console.info(`[PiexLoader] available [init=${typeof initPiexModule}]`);
/**
 * Set true if the Module.onAbort() handler is called.
 */
let piexFailed = false;
const MODULE_SETTINGS = {
    /**
     * Installs an (Emscripten) Module.onAbort handler. Record that the
     * Module has failed in piexFailed and re-throw the error.
     *
     * @throws {!Error|string}
     */
    onAbort: (error) => {
        piexFailed = true;
        throw error;
    },
};
let initPiexModulePromise = null;
/**
 * Returns a promise that resolves once initialization is complete. PiexModule
 * may be undefined before this promise resolves.
 */
function piexModuleInitialized() {
    if (!initPiexModulePromise) {
        initPiexModulePromise = new Promise(resolve => {
            initPiexModule(MODULE_SETTINGS).then(module => {
                PiexModule = module;
                console.info(`[PiexLoader] loaded [module=${typeof module}]`);
                resolve();
            });
        });
    }
    return initPiexModulePromise;
}
/**
 * Module failure recovery: if piexFailed is set via onAbort due to OOM in
 * the C++ for example, or the Module failed to load or call run, then the
 * Module is in a broken, non-functional state.
 *
 * Loading the entire page is the only reliable way to recover from broken
 * Module state. Log the error, and return true to tell caller to initiate
 * failure recovery steps.
 *
 */
function piexModuleFailed() {
    if (piexFailed || !PiexModule.calledRun) {
        console.error('[PiexLoader] piex wasm module failed');
        return true;
    }
    return false;
}
class PiexLoaderResponse {
    thumbnail;
    mimeType;
    /** JEITA EXIF image orientation being an integer in [1..8].  */
    orientation;
    /** JEITA EXIF image color space: 'sRgb' or 'adobeRgb'.  */
    colorSpace;
    /** JSON encoded RAW image photographic details.  */
    ifd;
    /**
     * @param data The extracted preview image data.
     */
    constructor(data) {
        this.thumbnail = data.thumbnail;
        this.mimeType = data.mimeType || 'image/jpeg';
        this.orientation = data.orientation;
        this.colorSpace = data.colorSpace;
        this.ifd = data.ifd || null;
    }
}
/** JFIF APP2 ICC_PROFILE segment containing an AdobeRGB1998 Color Profile. */
const adobeProfile = new Uint8Array([
    // clang-format off
    // APP2 ICC_PROFILE\0 segment header.
    0xff, 0xe2, 0x02, 0x40, 0x49, 0x43, 0x43, 0x5f, 0x50, 0x52, 0x4f, 0x46,
    0x49, 0x4c, 0x45, 0x00, 0x01, 0x01,
    // AdobeRGB1998 ICC Color Profile data.
    0x00, 0x00, 0x02, 0x30, 0x41, 0x44, 0x42, 0x45, 0x02, 0x10, 0x00, 0x00,
    0x6d, 0x6e, 0x74, 0x72, 0x52, 0x47, 0x42, 0x20, 0x58, 0x59, 0x5a, 0x20,
    0x07, 0xd0, 0x00, 0x08, 0x00, 0x0b, 0x00, 0x13, 0x00, 0x33, 0x00, 0x3b,
    0x61, 0x63, 0x73, 0x70, 0x41, 0x50, 0x50, 0x4c, 0x00, 0x00, 0x00, 0x00,
    0x6e, 0x6f, 0x6e, 0x65, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf6, 0xd6,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0xd3, 0x2d, 0x41, 0x44, 0x42, 0x45,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a,
    0x63, 0x70, 0x72, 0x74, 0x00, 0x00, 0x00, 0xfc, 0x00, 0x00, 0x00, 0x32,
    0x64, 0x65, 0x73, 0x63, 0x00, 0x00, 0x01, 0x30, 0x00, 0x00, 0x00, 0x6b,
    0x77, 0x74, 0x70, 0x74, 0x00, 0x00, 0x01, 0x9c, 0x00, 0x00, 0x00, 0x14,
    0x62, 0x6b, 0x70, 0x74, 0x00, 0x00, 0x01, 0xb0, 0x00, 0x00, 0x00, 0x14,
    0x72, 0x54, 0x52, 0x43, 0x00, 0x00, 0x01, 0xc4, 0x00, 0x00, 0x00, 0x0e,
    0x67, 0x54, 0x52, 0x43, 0x00, 0x00, 0x01, 0xd4, 0x00, 0x00, 0x00, 0x0e,
    0x62, 0x54, 0x52, 0x43, 0x00, 0x00, 0x01, 0xe4, 0x00, 0x00, 0x00, 0x0e,
    0x72, 0x58, 0x59, 0x5a, 0x00, 0x00, 0x01, 0xf4, 0x00, 0x00, 0x00, 0x14,
    0x67, 0x58, 0x59, 0x5a, 0x00, 0x00, 0x02, 0x08, 0x00, 0x00, 0x00, 0x14,
    0x62, 0x58, 0x59, 0x5a, 0x00, 0x00, 0x02, 0x1c, 0x00, 0x00, 0x00, 0x14,
    0x74, 0x65, 0x78, 0x74, 0x00, 0x00, 0x00, 0x00, 0x43, 0x6f, 0x70, 0x79,
    0x72, 0x69, 0x67, 0x68, 0x74, 0x20, 0x32, 0x30, 0x30, 0x30, 0x20, 0x41,
    0x64, 0x6f, 0x62, 0x65, 0x20, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6d, 0x73,
    0x20, 0x49, 0x6e, 0x63, 0x6f, 0x72, 0x70, 0x6f, 0x72, 0x61, 0x74, 0x65,
    0x64, 0x00, 0x00, 0x00, 0x64, 0x65, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x11, 0x41, 0x64, 0x6f, 0x62, 0x65, 0x20, 0x52, 0x47,
    0x42, 0x20, 0x28, 0x31, 0x39, 0x39, 0x38, 0x29, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x58, 0x59, 0x5a, 0x20, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0xf3, 0x51, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x16, 0xcc,
    0x58, 0x59, 0x5a, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x63, 0x75, 0x72, 0x76,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x33, 0x00, 0x00,
    0x63, 0x75, 0x72, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x02, 0x33, 0x00, 0x00, 0x63, 0x75, 0x72, 0x76, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x02, 0x33, 0x00, 0x00, 0x58, 0x59, 0x5a, 0x20,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9c, 0x18, 0x00, 0x00, 0x4f, 0xa5,
    0x00, 0x00, 0x04, 0xfc, 0x58, 0x59, 0x5a, 0x20, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x34, 0x8d, 0x00, 0x00, 0xa0, 0x2c, 0x00, 0x00, 0x0f, 0x95,
    0x58, 0x59, 0x5a, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x26, 0x31,
    0x00, 0x00, 0x10, 0x2f, 0x00, 0x00, 0xbe, 0x9c,
    // clang-format on
]);
/**
 * Preview Image EXtractor (PIEX).
 */
class ImageBuffer {
    source;
    length;
    memory = 0;
    /**
     * @param buffer - RAW image source data.
     */
    constructor(buffer) {
        this.source = new Uint8Array(buffer);
        this.length = buffer.byteLength;
    }
    /**
     * Calls Module.image() to process |this.source| and return the result.
     *
     * @throws {!Error} Memory allocation error.
     */
    process() {
        this.memory = PiexModule._malloc(this.length);
        if (!this.memory) {
            throw new Error('Image malloc failed: ' + this.length + ' bytes');
        }
        PiexModule.HEAP8.set(this.source, this.memory);
        const result = PiexModule.image(this.memory, this.length);
        if (result.error) {
            throw new Error(result.error);
        }
        return result;
    }
    /**
     * Returns the preview image data. If no preview image was found, returns
     * the thumbnail image.
     *
     * @throws {!Error} Data access security error.
     */
    preview(result) {
        const preview = result.preview;
        if (!preview) {
            return this.thumbnail_(result);
        }
        const offset = preview.offset;
        const length = preview.length;
        if (offset > this.length || (this.length - offset) < length) {
            throw new Error('Preview image access failed');
        }
        const view = new Uint8Array(this.source.buffer, offset, length);
        return {
            thumbnail: this.createImageDataArray_(view, preview).buffer,
            mimeType: 'image/jpeg',
            ifd: this.details_(result, preview.orientation),
            orientation: preview.orientation,
            colorSpace: preview.colorSpace,
        };
    }
    /**
     * Returns the thumbnail image. If no thumbnail image was found, returns
     * an empty thumbnail image.
     *
     * @throws {!Error} Data access security error.
     */
    thumbnail_(result) {
        const thumbnail = result.thumbnail;
        if (!thumbnail) {
            return {
                thumbnail: new ArrayBuffer(0),
                colorSpace: 'sRgb',
                orientation: 1,
                ifd: null,
            };
        }
        if (thumbnail.format) {
            return this.rgb_(result);
        }
        const offset = thumbnail.offset;
        const length = thumbnail.length;
        if (offset > this.length || (this.length - offset) < length) {
            throw new Error('Thumbnail image access failed');
        }
        const view = new Uint8Array(this.source.buffer, offset, length);
        return {
            thumbnail: this.createImageDataArray_(view, thumbnail).buffer,
            mimeType: 'image/jpeg',
            ifd: this.details_(result, thumbnail.orientation),
            orientation: thumbnail.orientation,
            colorSpace: thumbnail.colorSpace,
        };
    }
    /**
     * Returns the RGB thumbnail. If no RGB thumbnail was found, returns
     * an empty thumbnail image.
     *
     * @throws {!Error} Data access security error.
     */
    rgb_(result) {
        const thumbnail = result.thumbnail;
        if (!thumbnail || thumbnail.format !== 1) {
            return {
                thumbnail: new ArrayBuffer(0),
                colorSpace: 'sRgb',
                orientation: 1,
                ifd: null,
            };
        }
        // Expect a width and height.
        if (!thumbnail.width || !thumbnail.height) {
            throw new Error('invalid image width or height');
        }
        const offset = thumbnail.offset;
        const length = thumbnail.length;
        if (offset > this.length || (this.length - offset) < length) {
            throw new Error('Thumbnail image access failed');
        }
        const view = new Uint8Array(this.source.buffer, offset, length);
        // Compute output image width and height.
        const usesWidthAsHeight = thumbnail.orientation >= 5;
        const height = usesWidthAsHeight ? thumbnail.width : thumbnail.height;
        const width = usesWidthAsHeight ? thumbnail.height : thumbnail.width;
        // Compute pixel row stride.
        const rowPad = width & 3;
        const rowStride = 3 * width + rowPad;
        // Create bitmap image.
        const pixelDataOffset = 14 + 108;
        const fileSize = pixelDataOffset + rowStride * height;
        const bitmap = new DataView(new ArrayBuffer(fileSize));
        // BITMAPFILEHEADER 14 bytes.
        bitmap.setUint8(0, 'B'.charCodeAt(0));
        bitmap.setUint8(1, 'M'.charCodeAt(0));
        bitmap.setUint32(2, fileSize /* bytes */, true);
        bitmap.setUint32(6, /* Reserved */ 0, true);
        bitmap.setUint32(10, pixelDataOffset, true);
        // DIB BITMAPV4HEADER 108 bytes.
        bitmap.setUint32(14, /* HeaderSize */ 108, true);
        bitmap.setInt32(18, width, true);
        bitmap.setInt32(22, -height /* top-down DIB */, true);
        bitmap.setInt16(26, /* ColorPlanes */ 1, true);
        bitmap.setInt16(28, /* BitsPerPixel BI_RGB */ 24, true);
        bitmap.setUint32(30, /* Compression: BI_RGB none */ 0, true);
        bitmap.setUint32(34, /* ImageSize: 0 not compressed */ 0, true);
        bitmap.setInt32(38, /* XPixelsPerMeter */ 0, true);
        bitmap.setInt32(42, /* YPixelPerMeter */ 0, true);
        bitmap.setUint32(46, /* TotalPalletColors */ 0, true);
        bitmap.setUint32(50, /* ImportantColors */ 0, true);
        bitmap.setUint32(54, /* RedMask */ 0, true);
        bitmap.setUint32(58, /* GreenMask */ 0, true);
        bitmap.setUint32(62, /* BlueMask */ 0, true);
        bitmap.setUint32(66, /* AlphaMask */ 0, true);
        let rx = 0;
        let ry = 0;
        let gx = 0;
        let gy = 0;
        let bx = 0;
        let by = 0;
        let zz = 0;
        let gg = 0;
        if (thumbnail.colorSpace !== 'adobeRgb') {
            bitmap.setUint8(70, 's'.charCodeAt(0));
            bitmap.setUint8(71, 'R'.charCodeAt(0));
            bitmap.setUint8(72, 'G'.charCodeAt(0));
            bitmap.setUint8(73, 'B'.charCodeAt(0));
        }
        else {
            bitmap.setUint32(70, /* adobeRgb LCS_CALIBRATED_RGB */ 0);
            rx = Math.round(0.6400 * (1 << 30));
            ry = Math.round(0.3300 * (1 << 30));
            gx = Math.round(0.2100 * (1 << 30));
            gy = Math.round(0.7100 * (1 << 30));
            bx = Math.round(0.1500 * (1 << 30));
            by = Math.round(0.0600 * (1 << 30));
            zz = Math.round(1.0000 * (1 << 30));
            gg = Math.round(2.1992187 * (1 << 16));
        }
        // RGB CIEXYZ.
        bitmap.setUint32(74, /* R CIEXYZ x */ rx, true);
        bitmap.setUint32(78, /* R CIEXYZ y */ ry, true);
        bitmap.setUint32(82, /* R CIEXYZ z */ zz, true);
        bitmap.setUint32(86, /* G CIEXYZ x */ gx, true);
        bitmap.setUint32(90, /* G CIEXYZ y */ gy, true);
        bitmap.setUint32(94, /* G CIEXYZ z */ zz, true);
        bitmap.setUint32(98, /* B CIEXYZ x */ bx, true);
        bitmap.setUint32(102, /* B CIEXYZ y */ by, true);
        bitmap.setUint32(106, /* B CIEXYZ z */ zz, true);
        // RGB gamma.
        bitmap.setUint32(110, /* R Gamma */ gg, true);
        bitmap.setUint32(114, /* G Gamma */ gg, true);
        bitmap.setUint32(118, /* B Gamma */ gg, true);
        // Write RGB row pixels in top-down DIB order.
        const h = thumbnail.height - 1;
        const w = thumbnail.width - 1;
        let dx = 0;
        for (let input = 0, y = 0; y <= h; ++y) {
            let output = pixelDataOffset;
            /**
             * Compute affine(a,b,c,d,tx,ty) transform of pixel (x,y)
             *   { x': a * x + c * y + tx, y': d * y + b * x + ty }
             * a,b,c,d in [-1,0,1], to apply the image orientation at
             * (0,y) to find the output location of the input row.
             * The transform derivative in x is used to calculate the
             * relative output location of adjacent input row pixels.
             */
            switch (thumbnail.orientation) {
                case 1: // affine(+1, 0, 0, +1, 0, 0)
                    output += y * rowStride;
                    dx = 3;
                    break;
                case 2: // affine(-1, 0, 0, +1, w, 0)
                    output += y * rowStride + 3 * w;
                    dx = -3;
                    break;
                case 3: // affine(-1, 0, 0, -1, w, h)
                    output += (h - y) * rowStride + 3 * w;
                    dx = -3;
                    break;
                case 4: // affine(+1, 0, 0, -1, 0, h)
                    output += (h - y) * rowStride;
                    dx = 3;
                    break;
                case 5: // affine(0, +1, +1, 0, 0, 0)
                    output += 3 * y;
                    dx = rowStride;
                    break;
                case 6: // affine(0, +1, -1, 0, h, 0)
                    output += 3 * (h - y);
                    dx = rowStride;
                    break;
                case 7: // affine(0, -1, -1, 0, h, w)
                    output += w * rowStride + 3 * (h - y);
                    dx = -rowStride;
                    break;
                case 8: // affine(0, -1, +1, 0, 0, w)
                    output += w * rowStride + 3 * y;
                    dx = -rowStride;
                    break;
            }
            for (let x = 0; x <= w; ++x, input += 3, output += dx) {
                bitmap.setUint8(output + 0, view[input + 2]); // B
                bitmap.setUint8(output + 1, view[input + 1]); // G
                bitmap.setUint8(output + 2, view[input + 0]); // R
            }
        }
        // Write pixel row padding bytes if needed.
        if (rowPad) {
            let paddingOffset = pixelDataOffset + 3 * width;
            for (let y = 0; y < height; ++y) {
                let output = paddingOffset;
                switch (rowPad) {
                    case 1:
                        bitmap.setUint8(output++, 0);
                        break;
                    case 2:
                        bitmap.setUint8(output++, 0);
                        bitmap.setUint8(output++, 0);
                        break;
                    case 3:
                        bitmap.setUint8(output++, 0);
                        bitmap.setUint8(output++, 0);
                        bitmap.setUint8(output++, 0);
                        break;
                }
                paddingOffset += rowStride;
            }
        }
        return {
            thumbnail: bitmap.buffer,
            mimeType: 'image/bmp',
            ifd: this.details_(result, thumbnail.orientation),
            colorSpace: thumbnail.colorSpace,
            orientation: 1,
        };
    }
    /**
     * Converts a |view| of the "preview image" to Uint8Array data. Embeds an
     * AdobeRGB1998 ICC Color Profile in that data if the preview is JPEG and
     * it has 'adodeRgb' color space.
     *
     */
    createImageDataArray_(view, preview) {
        const jpeg = view.byteLength > 2 && view[0] === 0xff && view[1] === 0xd8;
        if (jpeg && preview.colorSpace === 'adobeRgb') {
            const data = new Uint8Array(view.byteLength + adobeProfile.byteLength);
            data.set(view.subarray(2), 2 + adobeProfile.byteLength);
            data.set(adobeProfile, 2);
            data.set([0xff, 0xd8], 0);
            return data;
        }
        return new Uint8Array(view);
    }
    /**
     * Returns the RAW image photographic |details| in a JSON-encoded string.
     * Only number and string values are retained, and they are formatted for
     * presentation to the user.
     *
     * @param orientation - image EXIF orientation
     */
    details_(result, orientation) {
        const details = result.details;
        if (!details) {
            return null;
        }
        const format = {};
        for (const [key, value] of Object.entries(details)) {
            if (typeof value === 'string') {
                format[key] = value.replace(/\0+$/, '').trim();
            }
            else if (typeof value === 'number') {
                if (!Number.isInteger(value)) {
                    format[key] = Number(value.toFixed(3).replace(/0+$/, ''));
                }
                else {
                    format[key] = value;
                }
            }
        }
        const usesWidthAsHeight = orientation >= 5;
        if (usesWidthAsHeight) {
            const width = format['width'];
            format['width'] = format['height'];
            format['height'] = width;
        }
        return JSON.stringify(format);
    }
    /**
     * Release resources.
     */
    close() {
        const memory = this.memory;
        if (memory) {
            PiexModule._free(memory);
            this.memory = 0;
        }
    }
}
/**
 * PiexLoader: is a namespace.
 */
const PiexLoader = {
    /**
     * Loads a RAW image. Returns the image metadata and the image thumbnail in a
     * PiexLoaderResponse.
     *
     * piexModuleFailed() returns true if the Module is in an unrecoverable error
     * state. This is rare, but possible, and the only reliable way to recover is
     * to reload the page. Callback |onPiexModuleFailed| is used to indicate that
     * the caller should initiate failure recovery steps.
     *
     */
    load(buffer, onPiexModuleFailed) {
        let imageBuffer;
        return piexModuleInitialized()
            .then(() => {
            if (piexModuleFailed()) {
                throw new Error('piex wasm module failed');
            }
            imageBuffer = new ImageBuffer(buffer);
            return imageBuffer.process();
        })
            .then((result) => {
            return new PiexLoaderResponse(imageBuffer.preview(result));
        })
            .catch((error) => {
            if (piexModuleFailed()) {
                setTimeout(onPiexModuleFailed, 0);
                return Promise.reject('piex wasm module failed');
            }
            console.warn('[PiexLoader] ' + error);
            return Promise.reject(error);
        })
            .finally(() => {
            imageBuffer && imageBuffer.close();
        });
    },
};

// Copyright 2013 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const ExtensionContentTypeMap = new Map([
    ['gif', 'image/gif'],
    ['png', 'image/png'],
    ['svg', 'image/svg'],
    ['bmp', 'image/bmp'],
    ['jpg', 'image/jpeg'],
    ['jpeg', 'image/jpeg'],
]);
const adpRegExp = RegExp('^filesystem:chrome-extension://[a-z]+/external/arc-documents-provider/');
/**
 * Calls the imageLoaderPrivate API with the given message.
 *
 * @param msg The imageLoaderPrivate call arguments.
 * @return A promise for the thumbnailDataUrl.
 */
function callImageLoaderPrivate(msg) {
    return new Promise((resolve, reject) => {
        chrome.runtime.sendMessage(null, msg, undefined, (thumbnailDataUrl) => {
            if (chrome.runtime.lastError) {
                console.warn(chrome.runtime.lastError.message);
                reject(chrome.runtime.lastError);
            }
            else if (thumbnailDataUrl) {
                resolve(thumbnailDataUrl);
            }
            else {
                reject();
            }
        });
    });
}
/**
 * Creates and starts downloading and then resizing of the image. Finally,
 * returns the image using the callback.
 */
class ImageRequestTask {
    id_;
    cache_;
    request_;
    sendResponse_;
    /**
     * The maximum milliseconds to load video. If loading video exceeds the limit,
     * we give up generating video thumbnail and free the consumed memory.
     */
    static MAX_MILLISECONDS_TO_LOAD_VIDEO = 10000;
    /**
     * The default width of a non-square thumbnail. The value is set to match the
     * behavior of drivefs thumbnail generation.
     * See chromeos/ash/components/drivefs/mojom/drivefs.mojom
     */
    static DEFAULT_THUMBNAIL_SQUARE_SIZE = 360;
    /**
     * The default width of a non-square thumbnail. The value is set to match the
     * behavior of drivefs thumbnail generation.
     * See chromeos/ash/components/drivefs/mojom/drivefs.mojom
     */
    static DEFAULT_THUMBNAIL_WIDTH = 500;
    /**
     * The default height of a non-square thumbnail. The value is set to match the
     * behavior of drivefs thumbnail generation.
     * See chromeos/ash/components/drivefs/mojom/drivefs.mojom
     */
    static DEFAULT_THUMBNAIL_HEIGHT = 500;
    /**
     * Temporary image used to download images.
     */
    image_ = new Image();
    /**
     * MIME type of the fetched image.
     */
    contentType_;
    /**
     * IFD data of the fetched image. Only RAW images provide a non-null
     * ifd at this time. Drive images might provide an ifd in future.
     */
    ifd_;
    /**
     * Used to download remote images using http:// or https:// protocols.
     */
    xhr_ = null;
    /**
     * Temporary canvas used to resize and compress the image.
     */
    canvas_;
    renderOrientation_ = null;
    /**
     * Callback to be called once downloading is finished.
     */
    downloadCallback_ = null;
    aborted_ = false;
    /**
     * @param id Request ID.
     * @param cache Cache object.
     * @param request Request message as a hash array.
     * @param callback Response handler.
     */
    constructor(id_, cache_, request_, sendResponse_) {
        this.id_ = id_;
        this.cache_ = cache_;
        this.request_ = request_;
        this.sendResponse_ = sendResponse_;
        this.canvas_ = document.createElement('canvas');
    }
    /**
     * Extracts MIME type of a data URL.
     * @param dataUrl Data URL.
     * @return MIME type string, or null if the URL is invalid.
     */
    static getDataUrlMimeType(dataUrl) {
        const dataUrlMatches = (dataUrl || '').match(/^data:([^,;]*)[,;]/);
        return dataUrlMatches ? dataUrlMatches[1] : undefined;
    }
    /**
     * Returns ID of the request.
     * @return Request ID.
     */
    getId() {
        return this.id_;
    }
    getClientTaskId() {
        // Every incoming request should have been given a taskId.
        assert(this.request_.taskId);
        return this.request_.taskId;
    }
    /**
     * Returns priority of the request. The higher priority, the faster it will
     * be handled. The highest priority is 0. The default one is 2.
     *
     * @return Priority.
     */
    getPriority() {
        return this.request_.priority !== undefined ? this.request_.priority : 2;
    }
    /**
     * Tries to load the image from cache, if it exists in the cache, and sends
     * the response. Fails if the image is not found in the cache.
     *
     * @param onSuccess Success callback.
     * @param onFailure Failure callback.
     */
    loadFromCacheAndProcess(onSuccess, onFailure) {
        this.loadFromCache_((width, height, ifd, data) => {
            // Found in cache.
            this.ifd_ = ifd;
            this.sendImageData_(width, height, data);
            onSuccess();
        }, onFailure); // Not found in cache.
    }
    /**
     * Tries to download the image, resizes and sends the response.
     *
     * @param callback Completion callback.
     */
    downloadAndProcess(callback) {
        if (this.downloadCallback_) {
            throw new Error('Downloading already started.');
        }
        this.downloadCallback_ = callback;
        this.downloadThumbnail_(this.onImageLoad_.bind(this), this.onImageError_.bind(this));
    }
    /**
     * Fetches the image from the persistent cache.
     *
     * @param onSuccess callback with the image width, height, ?ifd, and data.
     * @param onFailure Failure callback.
     */
    loadFromCache_(onSuccess, onFailure) {
        const key = cacheKey(this.request_);
        if (!key) {
            // Cache key is not provided for the request.
            onFailure();
            return;
        }
        if (!this.request_.cache) {
            // Cache is disabled for this request; therefore, remove it from cache
            // if existed.
            this.cache_.removeImage(key);
            onFailure();
            return;
        }
        const timestamp = this.request_.timestamp;
        if (!timestamp) {
            // Persistent cache is available only when a timestamp is provided.
            onFailure();
            return;
        }
        this.cache_.loadImage(key, timestamp, onSuccess, onFailure);
    }
    /**
     * Saves the image to the persistent cache.
     *
     * @param width Image width.
     * @param height Image height.
     * @param data Image data.
     */
    saveToCache_(width, height, data) {
        const timestamp = this.request_.timestamp;
        if (!this.request_.cache || !timestamp) {
            // Persistent cache is available only when a timestamp is provided.
            return;
        }
        const key = cacheKey(this.request_);
        if (!key) {
            // Cache key is not provided for the request.
            return;
        }
        this.cache_.saveImage(key, timestamp, width, height, this.ifd_, data);
    }
    /**
     * Gets the target image size for external thumbnails, where supported.
     * The defaults replicate drivefs thumbnailer behavior.
     */
    targetThumbnailSize_() {
        const crop = !!this.request_.crop;
        const defaultWidth = crop ? ImageRequestTask.DEFAULT_THUMBNAIL_SQUARE_SIZE :
            ImageRequestTask.DEFAULT_THUMBNAIL_WIDTH;
        const defaultHeight = crop ?
            ImageRequestTask.DEFAULT_THUMBNAIL_SQUARE_SIZE :
            ImageRequestTask.DEFAULT_THUMBNAIL_HEIGHT;
        return {
            width: this.request_.width || defaultWidth,
            height: this.request_.height || defaultHeight,
        };
    }
    /**
     * Loads |this.image_| with the |this.request_.url| source or the thumbnail
     * image of the source.
     *
     * @param onSuccess Success callback.
     * @param onFailure Failure callback.
     */
    async downloadThumbnail_(onSuccess, onFailure) {
        // Load methods below set |this.image_.src|. Call revokeObjectURL(src) to
        // release resources if the image src was created with createObjectURL().
        this.image_.onload = () => {
            URL.revokeObjectURL(this.image_.src);
            onSuccess();
        };
        this.image_.onerror = () => {
            URL.revokeObjectURL(this.image_.src);
            onFailure();
        };
        // Load dataURL sources directly.
        const dataUrlMimeType = ImageRequestTask.getDataUrlMimeType(this.request_.url);
        const requestUrl = this.request_.url ?? '';
        if (dataUrlMimeType) {
            this.image_.src = requestUrl;
            this.contentType_ = dataUrlMimeType;
            return;
        }
        const onExternalThumbnail = (dataUrl) => {
            this.image_.src = dataUrl;
            this.contentType_ = ImageRequestTask.getDataUrlMimeType(dataUrl);
        };
        // Load Drive source thumbnail.
        const drivefsUrlMatches = requestUrl.match(/^drivefs:(.*)/);
        if (drivefsUrlMatches) {
            callImageLoaderPrivate({
                apiMethod: 'getDriveThumbnail',
                params: {
                    url: drivefsUrlMatches[1] || '',
                    cropToSquare: !!this.request_.crop,
                },
            })
                .then(onExternalThumbnail)
                .catch(onFailure);
            return;
        }
        // Load PDF source thumbnail.
        if (requestUrl.endsWith('.pdf')) {
            const { width, height } = this.targetThumbnailSize_();
            callImageLoaderPrivate({
                apiMethod: 'getPdfThumbnail',
                params: {
                    url: requestUrl,
                    width,
                    height,
                },
            })
                .then(onExternalThumbnail)
                .catch(onFailure);
            return;
        }
        // Load ARC DocumentsProvider thumbnail, if supported.
        if (requestUrl.match(adpRegExp)) {
            const { width, height } = this.targetThumbnailSize_();
            callImageLoaderPrivate({
                apiMethod: 'getArcDocumentsProviderThumbnail',
                params: {
                    url: requestUrl,
                    widthHint: width,
                    heightHint: height,
                },
            })
                .then(onExternalThumbnail)
                .catch(onFailure);
            return;
        }
        const fileType = getFileTypeForName(requestUrl);
        // Load video source thumbnail.
        if (fileType.type === 'video') {
            this.createVideoThumbnailUrl_(requestUrl)
                .then((url) => {
                this.image_.src = url;
            })
                .catch((error) => {
                console.warn('Video thumbnail error: ', error);
                onFailure();
            });
            return;
        }
        // Load the source directly.
        this.load(requestUrl, (contentType, blob) => {
            // Load RAW image source thumbnail.
            if (fileType.type === 'raw') {
                blob.arrayBuffer()
                    .then((buffer) => PiexLoader.load(buffer, chrome.runtime.reload))
                    .then((data) => {
                    this.renderOrientation_ =
                        ImageOrientation.fromExifOrientation(data.orientation);
                    this.ifd_ = data.ifd ?? undefined;
                    this.contentType_ = data.mimeType;
                    const blob = new Blob([data.thumbnail], { type: data.mimeType });
                    this.image_.src = URL.createObjectURL(blob);
                })
                    .catch(onFailure);
                return;
            }
            this.image_.src = blob ? URL.createObjectURL(blob) : '!';
            this.contentType_ = contentType || undefined;
            if (this.contentType_ === 'image/jpeg') {
                this.renderOrientation_ = ImageOrientation.fromExifOrientation(1);
            }
        }, onFailure);
    }
    /**
     * Creates a video thumbnail data url from video file.
     *
     * @param url Video URL.
     * @return Promise that resolves with the data url of video
     *    thumbnail.
     */
    createVideoThumbnailUrl_(url) {
        const video = document.createElement('video');
        return Promise
            .race([
            new Promise((resolve, reject) => {
                video.addEventListener('loadedmetadata', () => {
                    video.addEventListener('seeked', () => {
                        if (video.readyState >= video.HAVE_CURRENT_DATA) {
                            resolve();
                        }
                        else {
                            video.addEventListener('loadeddata', () => resolve());
                        }
                    });
                    // For videos with longer duration (>= 6 seconds), consider the
                    // frame at 3rd second, or use the frame at midpoint otherwise.
                    // This ensures the target position is always close to the
                    // beginning of the video. Seek operations may be costly if the
                    // video doesn't contain keyframes for referencing.
                    const thumbnailPosition = Math.min(video.duration / 2, 3);
                    video.currentTime = thumbnailPosition;
                });
                video.addEventListener('error', reject);
                video.preload = 'metadata';
                video.src = url;
                video.load();
            }),
            new Promise((resolve) => {
                setTimeout(resolve, ImageRequestTask.MAX_MILLISECONDS_TO_LOAD_VIDEO);
            }).then(() => {
                // If we can't get the frame at the midpoint of the video after 3
                // seconds have passed for some reason (e.g. unseekable video), we
                // give up generating thumbnail.
                // Make sure to stop loading remaining part of the video.
                video.src = '';
                throw new Error('Seeking video failed.');
            }),
        ])
            .then(() => {
            const canvas = document.createElement('canvas');
            canvas.width = video.videoWidth;
            canvas.height = video.videoHeight;
            canvas.getContext('2d').drawImage(video, 0, 0);
            // Clearing the `src` helps the decoder to dispose its memory earlier.
            video.src = '';
            return canvas.toDataURL();
        });
    }
    /**
     * Loads an image.
     *
     * @param url URL to the resource to be fetched.
     * @param onSuccess Success callback with the content type and the fetched
     *     data.
     * @param onFailure Failure callback.
     */
    load(url, onSuccess, onFailure) {
        this.aborted_ = false;
        // Do not call any callbacks when aborting.
        const onMaybeSuccess = (contentType, response) => {
            // When content type is not available, try to estimate it from url.
            if (!contentType) {
                contentType = ExtensionContentTypeMap.get(this.extractExtension_(url));
            }
            if (!this.aborted_) {
                onSuccess(contentType, response);
            }
        };
        const onMaybeFailure = () => {
            if (!this.aborted_) {
                onFailure();
            }
        };
        // The query parameter is workaround for crbug.com/379678, which forces the
        // browser to obtain the latest contents of the image.
        const noCacheUrl = url + '?nocache=' + Date.now();
        this.xhr_ = ImageRequestTask.load_(noCacheUrl, onMaybeSuccess, onMaybeFailure);
    }
    /**
     * Extracts extension from url.
     * @param url Url.
     * @return Extracted extension, e.g. png.
     */
    extractExtension_(url) {
        const result = /\.([a-zA-Z]+)$/i.exec(url);
        return result ? result[1] ?? '' : '';
    }
    /**
     * Fetches data using XmlHttpRequest.
     *
     * @param url URL to the resource to be fetched.
     * @param onSuccess Success callback with the content type and the fetched
     *     data.
     * @param onFailure Failure callback with the error code if available.
     * @return XHR instance.
     */
    static load_(url, onSuccess, onFailure) {
        const xhr = new XMLHttpRequest();
        xhr.responseType = 'blob';
        xhr.onreadystatechange = () => {
            if (xhr.readyState !== 4) {
                return;
            }
            if (xhr.status !== 200) {
                onFailure(xhr.status);
                return;
            }
            const response = xhr.response;
            const contentType = xhr.getResponseHeader('Content-Type') || response.type;
            onSuccess(contentType, response);
        };
        // Perform a xhr request.
        try {
            xhr.open('GET', url, true);
            xhr.send();
        }
        catch (e) {
            onFailure();
        }
        return xhr;
    }
    /**
     * Sends the resized image via the callback. If the image has been changed,
     * then packs the canvas contents, otherwise sends the raw image data.
     *
     * @param imageChanged Whether the image has been changed.
     */
    sendImage_(imageChanged) {
        let width;
        let height;
        let data;
        if (!imageChanged) {
            // The image hasn't been processed, so the raw data can be directly
            // forwarded for speed (no need to encode the image again).
            width = this.image_.width;
            height = this.image_.height;
            data = this.image_.src;
        }
        else {
            // The image has been resized or rotated, therefore the canvas has to be
            // encoded to get the correct compressed image data.
            width = this.canvas_.width;
            height = this.canvas_.height;
            switch (this.contentType_) {
                case 'image/jpeg':
                    data = this.canvas_.toDataURL('image/jpeg', 0.9);
                    break;
                default:
                    data = this.canvas_.toDataURL('image/png');
                    break;
            }
        }
        // Send the image data and also save it in the persistent cache.
        this.sendImageData_(width, height, data);
        this.saveToCache_(width, height, data);
    }
    /**
     * Sends the resized image via the callback.
     *
     * @param width Image width.
     * @param height Image height.
     * @param data Image data.
     */
    sendImageData_(width, height, data) {
        const result = { width, height, ifd: this.ifd_, data };
        this.sendResponse_(new LoadImageResponse(LoadImageResponseStatus.SUCCESS, this.getClientTaskId(), result));
    }
    /**
     * Handler, when contents are loaded into the image element. Performs image
     * processing operations if needed, and finalizes the request process.
     */
    onImageLoad_() {
        const requestOrientation = this.request_.orientation;
        // Override the request orientation before processing if needed.
        if (this.renderOrientation_) {
            this.request_.orientation = this.renderOrientation_;
        }
        // Perform processing if the url is not a data url, or if there are some
        // operations requested.
        let imageChanged = false;
        const reqUrl = this.request_.url ?? '';
        if (!(reqUrl.match(/^data/) || reqUrl.match(/^drivefs:/)) ||
            shouldProcess(this.image_.width, this.image_.height, this.request_)) {
            resizeAndCrop(this.image_, this.canvas_, this.request_);
            imageChanged = true; // The image is now on the <canvas>.
        }
        // Restore the request orientation after processing.
        if (this.renderOrientation_) {
            this.request_.orientation = requestOrientation;
        }
        // Finalize the request.
        this.sendImage_(imageChanged);
        this.cleanup_();
        if (this.downloadCallback_) {
            this.downloadCallback_();
        }
    }
    /**
     * Handler, when loading of the image fails. Sends a failure response and
     * finalizes the request process.
     */
    onImageError_() {
        this.sendResponse_(new LoadImageResponse(LoadImageResponseStatus.ERROR, this.getClientTaskId()));
        this.cleanup_();
        if (this.downloadCallback_) {
            this.downloadCallback_();
        }
    }
    /**
     * Cancels the request.
     */
    cancel() {
        this.cleanup_();
        // If downloading has started, then call the callback.
        if (this.downloadCallback_) {
            this.downloadCallback_();
        }
    }
    /**
     * Cleans up memory used by this request.
     */
    cleanup_() {
        this.image_.onerror = () => { };
        this.image_.onload = () => { };
        // Transparent 1x1 pixel gif, to force garbage collecting.
        this.image_.src =
            'data:image/gif;base64,R0lGODlhAQABAAAAACH5BAEKAAEALAAAAA' +
                'ABAAEAAAICTAEAOw==';
        this.aborted_ = true;
        if (this.xhr_) {
            this.xhr_.abort();
        }
        // Dispose memory allocated by Canvas.
        this.canvas_.width = 0;
        this.canvas_.height = 0;
    }
}

// Copyright 2013 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// `deviceMemory` might be 0.5 or 0.25, so we normalize to minimum of 2.
const memory = Math.max(2, navigator.deviceMemory);
// For low end devices `hardwareCount` can be low like 4 some other devies are
// low in memory, it will have the value 4 as in 4GB.
const resourceCount = Math.min(navigator.hardwareConcurrency, memory, 10);
/**
 * Maximum download tasks to be run in parallel, for low end devices we expect
 * the result to be 2, higher end devices we expect to be at least 4, but no
 * more than 5.
 */
const MAXIMUM_IN_PARALLEL = resourceCount / 2;
console.warn(`Image Loader maximum parallel tasks: ${MAXIMUM_IN_PARALLEL}`);
/**
 * Scheduler for ImageRequestTask objects. Fetches tasks from a queue and
 * processes them synchronously, taking into account priorities. The highest
 * priority is 0.
 */
class Scheduler {
    /**
     * List of tasks waiting to be checked. If these items are available in
     * cache, then they are processed immediately after starting the scheduler.
     * However, if they have to be downloaded, then these tasks are moved to
     * pendingTasks_.
     */
    newTasks_ = [];
    /** List of pending tasks for images to be downloaded. */
    pendingTasks_ = [];
    /** List of tasks being processed. */
    activeTasks_ = [];
    /**
     * Map of tasks being added to the queue, but not finalized yet. Keyed by
     * the ImageRequestTask id.
     */
    tasks_ = {};
    /** If the scheduler has been started. */
    started_ = false;
    /**
     * Adds a task to the internal priority queue and executes it when tasks
     * with higher priorities are finished. If the result is cached, then it is
     * processed immediately once the scheduler is started.
     */
    add(task) {
        if (!this.started_) {
            this.newTasks_.push(task);
            this.tasks_[task.getId()] = task;
            return;
        }
        // Enqueue the tasks, since already started.
        this.pendingTasks_.push(task);
        this.sortPendingTasks_();
        this.continue_();
    }
    /** Removes a task from the scheduler (if exists). */
    remove(taskId) {
        const task = this.tasks_[taskId];
        if (!task) {
            return;
        }
        // Remove from the internal queues with pending tasks.
        const newIndex = this.newTasks_.indexOf(task);
        if (newIndex !== -1) {
            this.newTasks_.splice(newIndex, 1);
        }
        const pendingIndex = this.pendingTasks_.indexOf(task);
        if (pendingIndex !== -1) {
            this.pendingTasks_.splice(pendingIndex, 1);
        }
        // Cancel the task.
        task.cancel();
        delete this.tasks_[taskId];
    }
    /** Starts handling tasks. */
    start() {
        this.started_ = true;
        // Process tasks added before scheduler has been started.
        this.pendingTasks_ = this.newTasks_;
        this.sortPendingTasks_();
        this.newTasks_ = [];
        // Start serving enqueued tasks.
        this.continue_();
    }
    /** Sorts pending tasks by priorities. */
    sortPendingTasks_() {
        this.pendingTasks_.sort((a, b) => {
            return a.getPriority() - b.getPriority();
        });
    }
    /**
     * Processes pending tasks from the queue. There is no guarantee that
     * all of the tasks will be processed at once.
     */
    continue_() {
        // Run only up to MAXIMUM_IN_PARALLEL in the same time.
        while (this.pendingTasks_.length > 0 &&
            this.activeTasks_.length < MAXIMUM_IN_PARALLEL) {
            const task = this.pendingTasks_.shift();
            this.activeTasks_.push(task);
            // Try to load from cache. If doesn't exist, then download.
            task.loadFromCacheAndProcess(() => this.finish_(task), () => task.downloadAndProcess(() => this.finish_(task)));
        }
    }
    /** Handles a finished task. */
    finish_(task) {
        const index = this.activeTasks_.indexOf(task);
        if (index < 0) {
            console.warn('ImageRequestTask not found.');
        }
        this.activeTasks_.splice(index, 1);
        delete this.tasks_[task.getId()];
        // Continue handling the most important tasks (if started).
        if (this.started_) {
            this.continue_();
        }
    }
}

// Copyright 2013 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
let instance = null;
/**
 * Loads and resizes an image.
 */
class ImageLoader {
    /**
     * Persistent cache object.
     */
    cache_ = new ImageCache();
    /**
     * Manages pending requests and runs them in order of priorities.
     */
    scheduler_ = new Scheduler();
    constructor() {
        // Initialize the cache and then start the scheduler.
        this.cache_.initialize(() => this.scheduler_.start());
    }
    /**
     * Handles a request. Depending on type of the request, starts or stops
     * an image task.
     * @return True if the message channel should stay alive until the
     *     sendResponse callback is called.
     */
    handle(request, sendResponse) {
        assert(request.imageLoaderRequestId);
        if (request.cancel) {
            this.scheduler_.remove(request.imageLoaderRequestId);
            return false;
        }
        // When manually debugging the Image Loader extension, you can reply with
        // a placeholder image here by patching in https://crrev.com/c/5796592
        // Sending a response may fail if the receiver already went offline.
        // This is not an error, but a normal and quite common situation.
        const failSafeSendResponse = function (response) {
            try {
                sendResponse(response);
            }
            catch (e) {
                // Ignore the error.
            }
        };
        // Incoming requests won't have the full type.
        assert(!(request.orientation instanceof ImageOrientation));
        assert(typeof request.orientation !== 'number');
        if (request.orientation) {
            request.orientation = ImageOrientation.fromRotationAndScale(request.orientation);
        }
        else {
            request.orientation = new ImageOrientation(1, 0, 0, 1);
        }
        // Add a new request task to the scheduler (queue).
        this.scheduler_.add(new ImageRequestTask(request.imageLoaderRequestId, this.cache_, request, failSafeSendResponse));
        return true;
    }
    /**
     * Returns a singleton instance.
     */
    static getInstance() {
        if (!instance) {
            instance = new ImageLoader();
        }
        return instance;
    }
}

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Entry point for the Image Loader's offscreen document.
 */
const EXTENSION_ID = 'pmfjbimdmchhbnneeidfognadeopoehp';
chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
    if ((sender.id !== EXTENSION_ID) || !msg.imageLoaderRequestId) {
        return false;
    }
    return ImageLoader.getInstance().handle(msg, sendResponse);
});
//# sourceMappingURL=offscreen.rollup.js.map
