// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Calls the `fn` function which should expect the callback as last argument.
 *
 * Resolves with the result of the `fn`.
 *
 * Rejects if there is `chrome.runtime.lastError`.
 */
export async function promisify(fn, ...args) {
    return new Promise((resolve, reject) => {
        const callback = (result) => {
            if (chrome.runtime.lastError) {
                reject(chrome.runtime.lastError.message);
            }
            else {
                resolve(result);
            }
        };
        fn(...args, callback);
    });
}
export function iconSetToCSSBackgroundImageValue(iconSet) {
    let lowDpiPart = null;
    let highDpiPart = null;
    if (iconSet.icon16x16Url) {
        lowDpiPart = 'url(' + iconSet.icon16x16Url + ') 1x';
    }
    if (iconSet.icon32x32Url) {
        highDpiPart = 'url(' + iconSet.icon32x32Url + ') 2x';
    }
    if (lowDpiPart && highDpiPart) {
        return 'image-set(' + lowDpiPart + ', ' + highDpiPart + ')';
    }
    else if (lowDpiPart) {
        return 'image-set(' + lowDpiPart + ')';
    }
    else if (highDpiPart) {
        return 'image-set(' + highDpiPart + ')';
    }
    return 'none';
}
/**
 * Mapping table for FileError.code style enum to DOMError.name string.
 */
export var FileErrorToDomError;
(function (FileErrorToDomError) {
    FileErrorToDomError["ABORT_ERR"] = "AbortError";
    FileErrorToDomError["INVALID_MODIFICATION_ERR"] = "InvalidModificationError";
    FileErrorToDomError["INVALID_STATE_ERR"] = "InvalidStateError";
    FileErrorToDomError["NO_MODIFICATION_ALLOWED_ERR"] = "NoModificationAllowedError";
    FileErrorToDomError["NOT_FOUND_ERR"] = "NotFoundError";
    FileErrorToDomError["NOT_READABLE_ERR"] = "NotReadable";
    FileErrorToDomError["PATH_EXISTS_ERR"] = "PathExistsError";
    FileErrorToDomError["QUOTA_EXCEEDED_ERR"] = "QuotaExceededError";
    FileErrorToDomError["TYPE_MISMATCH_ERR"] = "TypeMismatchError";
    FileErrorToDomError["ENCODING_ERR"] = "EncodingError";
})(FileErrorToDomError || (FileErrorToDomError = {}));
/**
 * Extracts path from filesystem: URL.
 * @return The path if it can be parsed, null if it cannot.
 */
export function extractFilePath(url) {
    const match = /^filesystem:[\w-]*:\/\/[\w-]*\/(external|persistent|temporary)(\/.*)$/
        .exec(url || '');
    const path = match && match[2];
    if (!path) {
        return null;
    }
    return decodeURIComponent(path);
}
/**
 * @return True if the Files app is running as an open files or a
 *     select folder dialog. False otherwise.
 */
export function runningInBrowser() {
    return !window.appID;
}
/**
 * The last URL with visitURL().
 */
let lastVisitedURL;
/**
 * Visit the URL.
 *
 * If the browser is opening, the url is opened in a new tab, otherwise the url
 * is opened in a new window.
 */
export function visitURL(url) {
    lastVisitedURL = url;
    window.open(url);
}
/**
 * Return the last URL visited with visitURL().
 */
export function getLastVisitedURL() {
    return lastVisitedURL;
}
/**
 * Returns whether the window is teleported or not.
 */
export function isTeleported() {
    return new Promise(onFulfilled => {
        chrome.fileManagerPrivate.getProfiles((response) => {
            onFulfilled(response.currentProfileId !== response.displayedProfileId);
        });
    });
}
/**
 * Runs chrome.test.sendMessage in test environment. Does nothing if running
 * in production environment.
 */
export function testSendMessage(message) {
    if (chrome.test) {
        chrome.test.sendMessage(message);
    }
}
/**
 * Extracts the extension of the path.
 *
 * Examples:
 * splitExtension('abc.ext') -> ['abc', '.ext']
 * splitExtension('a/b/abc.ext') -> ['a/b/abc', '.ext']
 * splitExtension('a/b') -> ['a/b', '']
 * splitExtension('.cshrc') -> ['', '.cshrc']
 * splitExtension('a/b.backup/hoge') -> ['a/b.backup/hoge', '']
 */
export function splitExtension(path) {
    let dotPosition = path.lastIndexOf('.');
    if (dotPosition <= path.lastIndexOf('/')) {
        dotPosition = -1;
    }
    const filename = dotPosition !== -1 ? path.substr(0, dotPosition) : path;
    const extension = dotPosition !== -1 ? path.substr(dotPosition) : '';
    return [filename, extension];
}
/**
 * Checks if an API call returned an error, and if yes then prints it.
 */
export function checkAPIError() {
    if (chrome.runtime.lastError) {
        console.warn(chrome.runtime.lastError.message);
    }
}
/**
 * Makes a promise which will be fulfilled |ms| milliseconds later.
 */
export function delay(ms) {
    return new Promise(resolve => {
        setTimeout(resolve, ms);
    });
}
/**
 * Makes a promise which will be rejected if the given |promise| is not resolved
 * or rejected for |ms| milliseconds.
 */
export function timeoutPromise(promise, ms, message) {
    return Promise.race([
        promise,
        delay(ms).then(() => {
            throw new Error(message || 'Operation timed out.');
        }),
    ]);
}
/**
 * Returns the Files app modal dialog used to embed any files app dialog
 * that derives from cr.ui.dialogs.
 */
export function getFilesAppModalDialogInstance() {
    let dialogElement = document.querySelector('#files-app-modal-dialog');
    if (!dialogElement) { // Lazily create the files app dialog instance.
        dialogElement = document.createElement('dialog');
        dialogElement.id = 'files-app-modal-dialog';
        document.body.appendChild(dialogElement);
    }
    return dialogElement;
}
export function descriptorEqual(left, right) {
    return left.appId === right.appId && left.taskType === right.taskType &&
        left.actionId === right.actionId;
}
/**
 * Create a taskID which is a string unique-ID for a task. This is temporary
 * and will be removed once we use task.descriptor everywhere instead.
 */
export function makeTaskID({ appId, taskType, actionId }) {
    return `${appId}|${taskType}|${actionId}`;
}
/**
 * Returns a new promise which, when fulfilled carries a boolean indicating
 * whether the app is in the guest mode. Typical use:
 *
 * isInGuestMode().then(
 *     (guest) => { if (guest) { ... in guest mode } }
 */
export async function isInGuestMode() {
    const response = await promisify(chrome.fileManagerPrivate.getProfiles);
    const profiles = response.profiles;
    return profiles.length > 0 && profiles[0]?.profileId === '$guest';
}
/**
 * A kind of error that represents user electing to cancel an operation. We use
 * this specialization to differentiate between system errors and errors
 * generated through legitimate user actions.
 */
export class UserCanceledError extends Error {
}
/**
 * Returns whether the given value is null or undefined.
 */
export const isNullOrUndefined = (value) => value === null || value === undefined;
/**
 * Bulk pinning should only show visible UI elements when in progress or
 * continuing to sync.
 */
export function canBulkPinningCloudPanelShow(stage, enabled) {
    const BulkPinStage = chrome.fileManagerPrivate.BulkPinStage;
    // If the stage is in progress and the bulk pinning preference is enabled,
    // then the cloud panel should not be visible.
    if (enabled &&
        (stage === BulkPinStage.GETTING_FREE_SPACE ||
            stage === BulkPinStage.LISTING_FILES ||
            stage === BulkPinStage.SYNCING)) {
        return true;
    }
    // For the PAUSED... states the preference should still be enabled, however,
    // for the latter the preference will have been disabled.
    if ((stage === BulkPinStage.PAUSED_OFFLINE && enabled) ||
        (stage === BulkPinStage.PAUSED_BATTERY_SAVER && enabled) ||
        stage === BulkPinStage.NOT_ENOUGH_SPACE) {
        return true;
    }
    return false;
}
export function debug(...vars) {
    // eslint-disable-next-line no-console
    console.debug(...vars);
}
