// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { isDirectoryEntry, isNativeEntry, isSameEntry, unwrapEntry } from '../../common/js/entry_utils.js';
import { getType } from '../../common/js/file_type.js';
import { strf } from '../../common/js/translations.js';
import {} from '../elements/files_metadata_box.js';
import {} from './metadata/metadata_item.js';
import { PathComponent } from './path_component.js';
function isTrashEntry(entry) {
    return 'restoreEntry' in entry;
}
/**
 * Controller of metadata box.
 * This should be initialized with |init| method.
 */
export class MetadataBoxController {
    constructor(metadataModel_, quickViewModel_, fileMetadataFormatter_, volumeManager_) {
        this.metadataModel_ = metadataModel_;
        this.quickViewModel_ = quickViewModel_;
        this.fileMetadataFormatter_ = fileMetadataFormatter_;
        this.volumeManager_ = volumeManager_;
        this.metadataBox_ = null;
        this.quickView_ = null;
        this.isDirectorySizeLoading_ = false;
        this.onDirectorySizeLoaded_ = null;
    }
    /**
     * Initialize the controller with quick view which will be lazily loaded.
     */
    init(quickView) {
        this.quickView_ = quickView;
        this.fileMetadataFormatter_.addEventListener('date-time-format-changed', this.updateView_.bind(this));
        this.quickView_.addEventListener('metadata-box-active-changed', this.updateView_.bind(this));
        this.quickViewModel_.addEventListener('selected-entry-changed', this.updateView_.bind(this));
        this.metadataBox_ = this.quickView_.getFilesMetadataBox();
        this.metadataBox_.clear(false);
        this.metadataBox_.setAttribute('files-ng', '');
    }
    /**
     * Update the view of metadata box.
     */
    updateView_(event) {
        if (!this.quickView_?.metadataBoxActive) {
            return;
        }
        const entry = this.quickViewModel_.getSelectedEntry();
        const sameEntry = isSameEntry(entry, this.previousEntry_);
        this.previousEntry_ = entry;
        if (!entry) {
            this.metadataBox?.clear(false);
            return;
        }
        if (event.type === 'date-time-format-changed') {
            // Update the displayed entry modificationTime format only, and return.
            this.metadataModel_.get([entry], ['modificationTime'])
                .then(this.updateModificationTime_.bind(this, entry));
            return;
        }
        // Do not clear isSizeLoading and size fields when the entry is not changed.
        this.metadataBox.clear(sameEntry);
        const metadata = [
            ...GENERAL_METADATA_NAMES,
            'alternateUrl',
            'externalFileUrl',
            'hosted',
        ];
        this.metadataModel_.get([entry], metadata)
            .then(this.onGeneralMetadataLoaded_.bind(this, entry, sameEntry));
    }
    /**
     * Accessor to get a guaranteed `FilesMetadataBox`.
     */
    get metadataBox() {
        return this.metadataBox_;
    }
    /**
     * Updates the metadata box with general and file-specific metadata.
     *
     * @param isSameEntry if the entry is not changed from the last time.
     */
    onGeneralMetadataLoaded_(entry, isSameEntry, items) {
        const type = getType(entry).type;
        const item = items[0];
        if (isDirectoryEntry(entry)) {
            this.setDirectorySize_(entry, isSameEntry);
        }
        else if (item?.size) {
            this.metadataBox.size =
                this.fileMetadataFormatter_.formatSize(item.size, item.hosted, true);
            this.metadataBox.metadataRendered('size');
        }
        if (isTrashEntry(entry)) {
            this.metadataBox.originalLocation =
                this.getFileLocationLabel_(entry.restoreEntry);
            this.metadataBox.metadataRendered('originalLocation');
        }
        this.updateModificationTime_(entry, items);
        if (!entry.isDirectory) {
            // Extra metadata types for local video media.
            let media = [];
            let sniffMimeType = 'mediaMimeType';
            if (item?.externalFileUrl || item?.alternateUrl) {
                sniffMimeType = 'contentMimeType';
            }
            else if (type === 'video') {
                media = EXTRA_METADATA_NAMES;
            }
            this.metadataModel_.get([entry], [sniffMimeType, ...media])
                .then(items => {
                let mimeType = items[0] &&
                    items[0][sniffMimeType] ||
                    '';
                const newType = getType(entry, mimeType);
                if (newType.encrypted) {
                    mimeType =
                        strf('METADATA_BOX_ENCRYPTED', newType.originalMimeType);
                }
                this.metadataBox.mediaMimeType = mimeType;
                this.metadataBox.metadataRendered('mime');
                this.metadataBox.fileLocation = this.getFileLocationLabel_(entry);
                this.metadataBox.metadataRendered('location');
            });
        }
        if (['image', 'video', 'audio'].includes(type)) {
            if (item?.externalFileUrl || item?.alternateUrl) {
                this.metadataModel_.get([entry], ['imageHeight', 'imageWidth'])
                    .then(items => {
                    this.metadataBox.imageWidth = items[0]?.imageWidth || 0;
                    this.metadataBox.imageHeight = items[0]?.imageHeight || 0;
                    this.metadataBox.setFileTypeInfo(type);
                    this.metadataBox.metadataRendered('meta');
                });
            }
            else {
                const data = EXTRA_METADATA_NAMES;
                this.metadataModel_.get([entry], data).then(items => {
                    const item = items[0];
                    this.metadataBox.setProperties({
                        ifd: item.ifd || null,
                        imageHeight: item.imageHeight || 0,
                        imageWidth: item.imageWidth || 0,
                        mediaAlbum: item.mediaAlbum || '',
                        mediaArtist: item.mediaArtist || '',
                        mediaDuration: item.mediaDuration || 0,
                        mediaGenre: item.mediaGenre || '',
                        mediaTitle: item.mediaTitle || '',
                        mediaTrack: item.mediaTrack || '',
                        mediaYearRecorded: item.mediaYearRecorded || '',
                    });
                    this.metadataBox.setFileTypeInfo(type);
                    this.metadataBox.metadataRendered('meta');
                });
            }
        }
        else if (type === 'raw') {
            this.metadataModel_.get([entry], ['ifd']).then(items => {
                const raw = items[0]?.ifd ? items[0].ifd : null;
                this.metadataBox.ifd = raw ? { raw } : undefined;
                this.metadataBox.imageWidth = raw?.width || 0;
                this.metadataBox.imageHeight = raw?.height || 0;
                this.metadataBox.setFileTypeInfo('image');
                this.metadataBox.metadataRendered('meta');
            });
        }
    }
    /**
     * Updates the metadata box modificationTime.
     */
    updateModificationTime_(_, items) {
        const item = items[0];
        this.metadataBox.modificationTime =
            this.fileMetadataFormatter_.formatModDate(item?.modificationTime);
    }
    /**
     * Set a current directory's size in metadata box.
     *
     * A loading animation is shown while fetching the directory size. However, it
     * won't show if there is no size value. Use a dummy value ' ' in that case.
     *
     * To avoid flooding the OS system with chrome.getDirectorySize requests, if a
     * previous request is active, store the new request and return. Only the most
     * recent new request is stored. When the active request returns, it calls the
     * stored request instead of updating the size field.
     *
     * `isSameEntry` is True if the entry is not changed from the last time. False
     * enables the loading animation.
     */
    setDirectorySize_(entry, sameEntry) {
        if (!isDirectoryEntry(entry)) {
            return;
        }
        const directoryEntry = unwrapEntry(entry);
        if (!isNativeEntry(directoryEntry)) {
            const typeName = ('typeName' in directoryEntry) ?
                directoryEntry.typeName :
                'no typeName';
            console.warn('Supplied directory is not a native type:', typeName);
            return;
        }
        if (this.metadataBox.size === '') {
            this.metadataBox.size = ' '; // Provide a dummy size value.
        }
        if (this.isDirectorySizeLoading_) {
            if (!sameEntry) {
                this.metadataBox.isSizeLoading = true;
            }
            // Store the new setDirectorySize_ request and return.
            this.onDirectorySizeLoaded_ = lastEntry => {
                this.setDirectorySize_(entry, isSameEntry(entry, lastEntry));
            };
            return;
        }
        this.metadataBox.isSizeLoading = !sameEntry;
        this.isDirectorySizeLoading_ = true;
        chrome.fileManagerPrivate.getDirectorySize(directoryEntry, (size) => {
            this.isDirectorySizeLoading_ = false;
            if (this.onDirectorySizeLoaded_) {
                setTimeout(this.onDirectorySizeLoaded_.bind(null, entry));
                this.onDirectorySizeLoaded_ = null;
                return;
            }
            if (this.quickViewModel_.getSelectedEntry() !== entry) {
                return;
            }
            if (chrome.runtime.lastError) {
                console.warn(chrome.runtime.lastError);
                size = undefined;
            }
            this.metadataBox.size =
                this.fileMetadataFormatter_.formatSize(size, true, true);
            this.metadataBox.isSizeLoading = false;
            this.metadataBox.metadataRendered('size');
        });
    }
    /**
     * Returns a label to display the file's location.
     */
    getFileLocationLabel_(entry) {
        const components = PathComponent.computeComponentsFromEntry(entry, this.volumeManager_);
        return components.map(c => c.name).join('/');
    }
}
export const GENERAL_METADATA_NAMES = [
    'size',
    'modificationTime',
];
export const EXTRA_METADATA_NAMES = [
    'ifd',
    'imageHeight',
    'imageWidth',
    'mediaAlbum',
    'mediaArtist',
    'mediaDuration',
    'mediaGenre',
    'mediaTitle',
    'mediaTrack',
    'mediaYearRecorded',
];
