// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview This is a table column representation
 */
import { dispatchPropertyChange } from 'chrome://resources/ash/common/cr_deprecated.js';
import { NativeEventTarget as EventTarget } from 'chrome://resources/ash/common/event_target.js';
import { jsSetter } from '../../../../common/js/cr_ui.js';
/**
 * A table column that wraps column ids and settings.
 */
export class TableColumn extends EventTarget {
    constructor(id_, name_, width_, endAlign_) {
        super();
        this.id_ = id_;
        this.name_ = name_;
        this.width_ = width_;
        this.endAlign_ = endAlign_;
        this.visible_ = true;
        this.defaultOrder_ = 'asc';
    }
    /**
     * Clones column.
     * @return Clone of the given column.
     */
    clone() {
        const tableColumn = new TableColumn(this.id_, this.name_, this.width_, this.endAlign_);
        tableColumn.renderFunction = this.renderFunction_;
        tableColumn.headerRenderFunction = this.headerRenderFunction_;
        tableColumn.defaultOrder = this.defaultOrder_;
        tableColumn.visible_ = this.visible_;
        return tableColumn;
    }
    /**
     * Renders table cell. This is the default render function.
     * @param dataItem The data item to be rendered.
     * @param columnId The column id.
     * @param table The table.
     * @return Rendered element.
     */
    renderFunction_(dataItem, columnId, table) {
        const div = table.ownerDocument.createElement('div');
        div.textContent = dataItem[columnId];
        div.hidden = !this.visible;
        return div;
    }
    /**
     * Renders table header. This is the default render function.
     * @param table The table.
     * @return Rendered text node.
     */
    headerRenderFunction_(table) {
        return table.ownerDocument.createTextNode(this.name);
    }
    /**
     * The width of the column.  Hidden columns have zero width.
     */
    get width() {
        return this.visible_ ? this.width_ : 0;
    }
    set width(value) {
        const oldValue = this.width;
        if (value !== oldValue) {
            this.width_ = value;
            dispatchPropertyChange(this, 'width', value, oldValue);
        }
    }
    /**
     * The width of the column, disregarding visibility.  For hidden columns,
     * this would be the width of the column if it were to be made visible.
     */
    get absoluteWidth() {
        return this.width_;
    }
    get id() {
        return this.id_;
    }
    set id(value) {
        jsSetter(this, 'id', value);
    }
    get name() {
        return this.name_;
    }
    set name(value) {
        jsSetter(this, 'name', value);
    }
    get visible() {
        return this.visible_;
    }
    set visible(value) {
        jsSetter(this, 'visible', value);
    }
    get endAlign() {
        return !!this.endAlign_;
    }
    set endAlign(value) {
        jsSetter(this, 'endAlign', value);
    }
    get renderFunction() {
        return this.renderFunction_;
    }
    set renderFunction(value) {
        jsSetter(this, 'renderFunction', value);
    }
    get headerRenderFunction() {
        return this.headerRenderFunction_;
    }
    set headerRenderFunction(value) {
        jsSetter(this, 'headerRenderFunction', value);
    }
    get defaultOrder() {
        return this.defaultOrder_;
    }
    set defaultOrder(value) {
        jsSetter(this, 'defaultOrder', value);
    }
}
