/**
 * @license
 * Copyright 2024 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import '../card/card';
import { css, html, LitElement } from 'lit';
import { AccordionItem } from './accordion_item';
/** A chromeOS compliant accordion. */
export class Accordion extends LitElement {
    /** @nocollapse */
    static { this.styles = css `
      cros-card {
        --cros-card-padding: 0;
      }

      ::slotted(cros-accordion-item:last-of-type) {
        --cros-accordion-item-separator-display: none;
      }
  `; }
    /** @nocollapse */
    static { this.properties = {
        autoCollapse: { type: Boolean, attribute: 'auto-collapse' },
        quick: { type: Boolean },
        variant: { type: String },
        cardStyle: { type: String, attribute: 'card-style' },
    }; }
    constructor() {
        super();
        this.autoCollapse = false;
        this.quick = false;
        this.variant = 'default';
        this.cardStyle = 'outline';
        this.addEventListener(AccordionItem.events.ACCORDION_ITEM_EXPANDED, (e) => this.onAccordionItemExpanded(e));
    }
    updated(changedProperties) {
        if (changedProperties.has('quick')) {
            // Set the `quick` property on all child `<cros-accordion-item>`s.
            for (const child of this.children) {
                if (child instanceof AccordionItem) {
                    child.quick = this.quick;
                }
            }
        }
        if (changedProperties.has('variant')) {
            // Set the `variant` property on all child `<cros-accordion-item>`s.
            for (const child of this.children) {
                if (child instanceof AccordionItem) {
                    child.variant = this.variant;
                }
            }
        }
    }
    render() {
        return html `
      <cros-card cardStyle=${this.cardStyle} part="card">
        <slot></slot>
      </cros-card>
    `;
    }
    onAccordionItemExpanded(e) {
        if (this.autoCollapse) {
            this.closeOtherAccordionItems(e.detail.accordionItem);
        }
    }
    closeOtherAccordionItems(accordionItem) {
        for (const child of this.children) {
            if (child instanceof AccordionItem && child !== accordionItem) {
                child.expanded = false;
            }
        }
    }
}
customElements.define('cros-accordion', Accordion);
