/**
 * @license
 * Copyright 2024 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { css } from 'lit';
import { MenuItem } from '../menu/menu_item';
/** A chromeOS compliant option to be used in cros-icon-dropdown. */
export class IconDropdownOption extends MenuItem {
    /** @nocollapse */
    static { this.shadowRootOptions = {
        ...MenuItem.shadowRootOptions,
    }; }
    /** @nocollapse */
    static { this.styles = [
        MenuItem.styles, css `
      md-menu-item {
        --md-menu-item-label-text-font: var(--cros-icon-dropdown-option-text-font, var(--cros-button-2-font-family));
      }
    `
    ]; }
    /** @nocollapse */
    static { this.properties = {
        ...MenuItem.properties,
        value: { type: String, reflect: true },
    }; }
    /** @nocollapse */
    static { this.events = {
        ...MenuItem.events,
        OPTION_TRIGGERED: 'cros-icon-dropdown-option-triggered',
    }; }
    get value() {
        if (this.internalValue === '') {
            return this.headline;
        }
        return this.internalValue;
    }
    set value(value) {
        this.internalValue = value;
    }
    // When extending a lit element, it seems that getters and setters get
    // clobbered. To avoid this we specifically reimplement needed getters/setters
    // below to ensure they function correctly on cros-icon-dropdown-option. These
    // should be identical to the functions they override in cros-menu-item.
    get selected() {
        return this.renderRoot?.querySelector('md-menu-item')?.selected ?? false;
    }
    set selected(selected) {
        const item = this.renderRoot?.querySelector('md-menu-item');
        if (!item) {
            this.missedPropertySets.selected = selected;
        }
        else {
            item.selected = selected;
        }
    }
    get switchSelected() {
        const crosSwitch = this.renderRoot?.querySelector('cros-switch');
        if (crosSwitch) {
            return crosSwitch.selected;
        }
        return this.missedPropertySets.switchSelected ?? false;
    }
    set switchSelected(value) {
        const crosSwitch = this.renderRoot?.querySelector('cros-switch');
        if (!crosSwitch) {
            this.missedPropertySets.switchSelected = value;
        }
        else {
            crosSwitch.selected = value;
        }
    }
    get typeaheadText() {
        return this.renderRoot?.querySelector('md-menu-item')?.typeaheadText ?? '';
    }
    set typeaheadText(text) {
        const item = this.renderRoot?.querySelector('md-menu-item');
        if (!item) {
            this.missedPropertySets.typeaheadText = text;
        }
        else {
            item.typeaheadText = text;
        }
    }
    // SelectOption implementation:
    get displayText() {
        return this.headline;
    }
    constructor() {
        super();
        this.onClickHandler = () => {
            this.onItemTriggered();
        };
        this.onKeyDownHandler = (e) => {
            if (e.key === 'Enter' || e.key === ' ') {
                this.onItemTriggered();
            }
        };
        // Default type to 'option' to match cros-icon-dropdown default menutype of
        // 'listbox'.
        this.type = 'option';
        this.internalValue = '';
        this.keepOpen = false;
    }
    connectedCallback() {
        super.connectedCallback();
        this.addEventListener('click', this.onClickHandler);
        this.addEventListener('keydown', this.onKeyDownHandler);
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.removeEventListener('click', this.onClickHandler);
        this.removeEventListener('keydown', this.onKeyDownHandler);
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('checked')) {
            const ariaSelected = this.checked ? 'true' : 'false';
            this.renderRoot?.querySelector('md-menu-item')
                ?.setAttribute('aria-selected', ariaSelected);
        }
    }
    // Notifies the parent icon-dropdown that this option was triggered.
    onItemTriggered() {
        // If the itemEnd is "switch" then do not notify the parent icon-dropdown
        // that this option was triggered.
        if (this.itemEnd === 'switch') {
            return;
        }
        this.checked = true;
        this.dispatchEvent(new CustomEvent(IconDropdownOption.events.OPTION_TRIGGERED, {
            bubbles: true,
            composed: true,
            detail: { menuItem: this },
        }));
    }
}
customElements.define('cros-icon-dropdown-option', IconDropdownOption);
