/**
 * @license
 * Copyright 2023 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import '@material/web/menu/menu.js';
import '@material/web/focus/md-focus-ring.js';
import { Corner, FocusState } from '@material/web/menu/menu.js';
import { css, html, LitElement } from 'lit';
/**
 * A chromeOS menu component.
 */
export class Menu extends LitElement {
    /** @nocollapse */
    static { this.styles = css `
    :host {
      display: flex;
      /* Do not affect parent width */
      width: 0px;
      --cros-menu-width: 96px;
    }
    md-menu {
      --md-menu-container-color: var(--cros-sys-base_elevated);
      --md-menu-item-container-color: var(--cros-sys-base_elevated);
      --md-menu-item-label-text-font: var(--cros-button-2-font-family);
      --md-menu-item-label-text-size: var(--cros-button-2-font-size);
      --md-menu-item-label-text-line-height: var(--cros-button-2-line-height);
      --md-menu-item-label-text-weight: var(--cros-button-2-font-weight);
      --md-menu-item-one-line-container-height: 36px;
      --md-menu-item-top-space: 0px;
      --md-menu-item-bottom-space: 0px;
      --md-menu-container-shape: 8px;
      --md-menu-container-surface-tint-layer-color: white;
      min-width: max(var(--cros-menu-width), 96px);
      /*
       * By default menu is display:contents. To make it focusable, it must be
       * something visible.
       */
      display: flex;
    }
    md-menu:not([open]) {
      /*
       * Makes menu non-tab-focusable when closed. We can do this as long as
       * there  is no animation set on menu as this might interfere with the
       * animation.
       *
       * Additionally we don't set tabindex on the menu when open as submenus
       * programmatically call .focus() when this.open still might not reflect
       * the actual state of the menu being open.
       */
      display: none;
    }
    md-focus-ring {
      --md-focus-ring-color: var(--cros-sys-focus_ring);
      --md-focus-ring-width: 2px;
      --md-focus-ring-active-width: 2px;
      --md-focus-ring-shape: 8px;
    }
    md-menu::part(elevation) {
      --md-menu-container-elevation: 0;
      box-shadow: 0px 12px 12px 0px rgba(var(--cros-sys-shadow-rgb), 0.2);
    }
    md-menu:focus-visible {
      outline: none;
    }
  `; }
    /** @nocollapse */
    static { this.properties = {
        anchor: { type: String },
        anchorCorner: { type: String, attribute: 'anchor-corner' },
        hasOverflow: { type: Boolean, attribute: 'has-overflow' },
        open: { type: Boolean },
        quick: { type: Boolean },
        menuCorner: { type: String, attribute: 'menu-corner' },
        defaultFocus: { type: String, attribute: 'default-focus' },
        skipRestoreFocus: { type: Boolean, attribute: 'skip-restore-focus' },
        stayOpenOnOutsideClick: { type: Boolean, attribute: 'stay-open-on-outside-click' },
        isSubmenu: { type: Boolean, attribute: 'is-submenu' },
        usePopover: { type: Boolean, attribute: 'use-popover' },
        type: { type: String }
    }; }
    /** @nocollapse */
    static { this.shadowRootOptions = {
        ...LitElement.shadowRootOptions,
        delegatesFocus: true
    }; }
    async focusFirstItem() {
        this.mdMenu.defaultFocus = FocusState.FIRST_ITEM;
        await this.mdMenu.updateComplete;
        this.show();
    }
    async focusLastItem() {
        this.mdMenu.defaultFocus = FocusState.LAST_ITEM;
        await this.mdMenu.updateComplete;
        this.show();
    }
    get mdMenu() {
        return this.renderRoot.querySelector('md-menu');
    }
    /** @export */
    get anchorElement() {
        if (this.anchor) {
            return this.getRootNode()
                .querySelector(`#${this.anchor}`);
        }
        return this.currentAnchorElement;
    }
    /** @export */
    set anchorElement(element) {
        this.currentAnchorElement = element;
        this.requestUpdate('anchorElement');
    }
    constructor() {
        super();
        this.anchorKeydownListener = async (e) => {
            // We need to let this propagate and see if another listener has handled
            // this event. e.g. in the case of a submenu anchoring to a sub-menu-item
            // we do not want to open this menu if the user is just pressing down to
            // navigate past that menu item.
            await new Promise(resolve => {
                setTimeout(resolve);
            });
            // This was handled by another listener e.g. md-menu
            if (e.defaultPrevented)
                return;
            if (e.key === 'ArrowDown') {
                await this.focusFirstItem();
            }
            else if (e.key === 'ArrowUp') {
                await this.focusLastItem();
            }
        };
        /** @private */
        this.currentAnchorElement = null;
        this.anchor = '';
        this.anchorElement = null;
        this.anchorCorner = Corner.END_START;
        this.hasOverflow = false;
        this.open = false;
        this.quick = true;
        this.menuCorner = Corner.START_START;
        // TODO: b/300001060 - confirm whether should change to FIRST_ITEM
        // NOTE this comment also applies to the tabindex=0 in the template
        this.defaultFocus = FocusState.LIST_ROOT;
        this.skipRestoreFocus = false;
        this.isSubmenu = false;
        this.usePopover = true;
        this.type = 'menu';
    }
    firstUpdated() {
        // Per spec, the submenu needs to be offset to account for the 8px padding
        // at the top of the menu. This way the first element of the submenu is
        // inline with the parent (anchor) sub-menu-item as per spec.
        if (this.slot === 'submenu') {
            this.mdMenu.yOffset = -8;
        }
    }
    // We add the listener & aria-expanded here instead of
    // connectedCallback/firstUpdated in case the anchor is added after initial
    // menu render.
    updated(changedProperties) {
        super.updated(changedProperties);
        const anchorElement = this.anchorElement;
        if ((changedProperties.has('anchor') ||
            changedProperties.has('anchorElement')) &&
            anchorElement) {
            anchorElement.addEventListener('keydown', this.anchorKeydownListener);
            anchorElement.setAttribute('aria-expanded', 'false');
            anchorElement.setAttribute('aria-haspopup', 'menu');
        }
    }
    disconnectedCallback() {
        const anchorElement = this.anchorElement;
        if (anchorElement) {
            anchorElement.removeEventListener('keydown', this.anchorKeydownListener);
        }
        super.disconnectedCallback();
    }
    /**
     * The menu items associated with this menu. The items must be `MenuItem`s and
     * have both the `md-menu-item` and `md-list-item` attributes.
     */
    get items() {
        // This accessor is required for keyboard navigation
        if (!this.mdMenu) {
            return [];
        }
        return this.mdMenu.items;
    }
    render() {
        return html `
      <md-menu
          id="menu"
          tabindex="0"
          role=${this.type}
          .anchorElement=${this.anchorElement}
          .open=${this.open}
          .hasOverflow=${this.hasOverflow}
          .anchorCorner=${this.anchorCorner}
          .menuCorner=${this.menuCorner}
          .quick=${this.quick}
          .defaultFocus=${this.defaultFocus}
          .skipRestoreFocus=${this.skipRestoreFocus}
          .isSubmenu=${this.isSubmenu}
          .positioning=${this.usePopover ? 'popover' : 'absolute'}
          @closed=${this.onMdMenuClosed}
          @opened=${this.show}
          @closing=${this.onClosing}
          @opening=${this.onOpening}
        >
        <slot></slot>
        <md-focus-ring for="menu"></md-focus-ring>
      </md-menu>
    `;
    }
    /**
     * Opens the menu.
     * @export
     */
    show() {
        this.open = true;
        // these non-bubbling events need to be re-dispatched for keyboard
        // navigation to work on submenus.
        this.dispatchEvent(new Event('opened'));
        this.anchorElement?.setAttribute('aria-expanded', 'true');
    }
    /**
     * Closes the menu.
     * @export
     */
    close() {
        this.performClose();
    }
    performClose() {
        this.open = false;
        // Set default focus back correctly (in case keyboard arrow nav has
        // changed it temporarily). We avoid doing this in focusFirstItem() to
        // avoid race conditions with menu open.
        this.renderRoot.querySelector('md-menu').defaultFocus = this.defaultFocus;
        this.anchorElement?.setAttribute('aria-expanded', 'false');
    }
    /**
     * Handles the md-menu closed event. Dispatch the 'closed' event here to
     * ensure that only a single 'closed' event is dispatched regardless of
     * whether the menu is closed via UI interaction or via the `close()` method.
     */
    onMdMenuClosed() {
        this.dispatchEvent(new Event('closed'));
        this.performClose();
    }
    onClosing() {
        this.dispatchEvent(new Event('closing'));
    }
    onOpening() {
        this.dispatchEvent(new Event('opening'));
    }
    // This is just in case `delegatesFocus` doesn't do what is intended (which is
    // unfortunately often across browser updates).
    focus() {
        this.renderRoot.querySelector('md-menu')?.focus();
    }
    getBoundingClientRect() {
        if (!this.mdMenu) {
            return super.getBoundingClientRect();
        }
        return this.mdMenu.getBoundingClientRect();
    }
}
customElements.define('cros-menu', Menu);
