// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { EventGenerator } from '../../event_generator.js';
import { Macro, MacroError } from './macro.js';
import { MacroName } from './macro_names.js';
var SyntheticMouseEventButton = chrome.accessibilityPrivate.SyntheticMouseEventButton;
/**
 * Class that implements a macro to send a single synthetic mouse click.
 */
export class MouseClickMacro extends Macro {
    leftClick_;
    location_;
    /**
     * Pass the location in density-independent pixels. Defaults to left click.
     */
    constructor(location, leftClick = true) {
        super(leftClick ? MacroName.MOUSE_CLICK_LEFT : MacroName.MOUSE_CLICK_RIGHT);
        this.leftClick_ = leftClick;
        this.location_ = location;
    }
    /** Invalid context if location isn't set. */
    checkContext() {
        if (!this.location_) {
            return this.createFailureCheckContextResult_(MacroError.BAD_CONTEXT);
        }
        return this.createSuccessCheckContextResult_();
    }
    updateLocation(location) {
        this.location_ = location;
    }
    run() {
        if (!this.location_) {
            return this.createRunMacroResult_(/*isSuccess=*/ false);
        }
        const mouseButton = this.leftClick_ ? SyntheticMouseEventButton.LEFT :
            SyntheticMouseEventButton.RIGHT;
        EventGenerator.sendMouseClick(this.location_.x, this.location_.y, {
            mouseButton,
        });
        return this.createRunMacroResult_(/*isSuccess=*/ true);
    }
}
/** Class that implements a macro to send a double left click. */
export class MouseClickLeftDoubleMacro extends Macro {
    location_;
    constructor(location) {
        super(MacroName.MOUSE_CLICK_LEFT_DOUBLE);
        this.location_ = location;
    }
    checkContext() {
        if (!this.location_) {
            return this.createFailureCheckContextResult_(MacroError.BAD_CONTEXT);
        }
        return this.createSuccessCheckContextResult_();
    }
    run() {
        if (!this.location_) {
            return this.createRunMacroResult_(/*isSuccess=*/ false);
        }
        EventGenerator.sendMouseClick(this.location_.x, this.location_.y, {
            clickArgs: { isDoubleClick: true },
        });
        return this.createRunMacroResult_(/*isSuccess=*/ true);
    }
}
/** Class that implements a macro to send a triple left click. */
export class MouseClickLeftTripleMacro extends Macro {
    location_;
    constructor(location) {
        super(MacroName.MOUSE_CLICK_LEFT_TRIPLE);
        this.location_ = location;
    }
    checkContext() {
        if (!this.location_) {
            return this.createFailureCheckContextResult_(MacroError.BAD_CONTEXT);
        }
        return this.createSuccessCheckContextResult_();
    }
    run() {
        if (!this.location_) {
            return this.createRunMacroResult_(/*isSuccess=*/ false);
        }
        EventGenerator.sendMouseClick(this.location_.x, this.location_.y, {
            clickArgs: { isTripleClick: true },
        });
        return this.createRunMacroResult_(/*isSuccess=*/ true);
    }
}
