"use strict";
export var ResizerType = /* @__PURE__ */ ((ResizerType2) => {
  ResizerType2["WIDTH"] = "width";
  ResizerType2["HEIGHT"] = "height";
  ResizerType2["BIDIRECTION"] = "bidirection";
  return ResizerType2;
})(ResizerType || {});
const cursorByResizerType = /* @__PURE__ */ new Map([
  ["width" /* WIDTH */, "ew-resize"],
  ["height" /* HEIGHT */, "ns-resize"],
  ["bidirection" /* BIDIRECTION */, "nwse-resize"]
]);
export class DragResizeHandler {
  document;
  delegate;
  originX;
  originY;
  boundMousemove;
  boundMousedown;
  constructor(document, delegate) {
    this.document = document;
    this.delegate = delegate;
    this.boundMousemove = this.onMousemove.bind(this);
    this.boundMousedown = this.onMousedown.bind(this);
  }
  install() {
    this.document.body.addEventListener("mousemove", this.boundMousemove);
    this.document.body.addEventListener("mousedown", this.boundMousedown);
  }
  uninstall() {
    this.document.body.removeEventListener("mousemove", this.boundMousemove);
    this.document.body.removeEventListener("mousedown", this.boundMousedown);
  }
  /**
   * Updates the cursor style of the mouse is hovered over a resizeable area.
   */
  onMousemove(event) {
    const match = this.delegate.getDraggable(event.clientX, event.clientY);
    if (!match) {
      this.document.body.style.cursor = "default";
      return;
    }
    this.document.body.style.cursor = cursorByResizerType.get(match.type) || "default";
  }
  /**
   * Starts dragging
   */
  onMousedown(event) {
    const match = this.delegate.getDraggable(event.clientX, event.clientY);
    if (!match) {
      return;
    }
    const boundOnDrag = this.onDrag.bind(this, match);
    event.stopPropagation();
    event.preventDefault();
    if (match.initialWidth !== void 0 && (match.type === "width" /* WIDTH */ || match.type === "bidirection" /* BIDIRECTION */)) {
      this.originX = {
        coord: Math.round(event.clientX),
        value: match.initialWidth
      };
    }
    if (match.initialHeight !== void 0 && (match.type === "height" /* HEIGHT */ || match.type === "bidirection" /* BIDIRECTION */)) {
      this.originY = {
        coord: Math.round(event.clientY),
        value: match.initialHeight
      };
    }
    this.document.body.removeEventListener("mousemove", this.boundMousemove);
    this.document.body.style.cursor = cursorByResizerType.get(match.type) || "default";
    const endDrag = (event2) => {
      event2.stopPropagation();
      event2.preventDefault();
      this.originX = void 0;
      this.originY = void 0;
      this.document.body.style.cursor = "default";
      this.document.body.removeEventListener("mousemove", boundOnDrag);
      this.document.body.addEventListener("mousemove", this.boundMousemove);
    };
    this.document.body.addEventListener("mouseup", endDrag, { once: true });
    window.addEventListener("mouseout", endDrag, { once: true });
    this.document.body.addEventListener("mousemove", boundOnDrag);
  }
  /**
   * Computes the new value while the cursor is being dragged and calls InspectorOverlayHost with the new value.
   */
  onDrag(match, e) {
    if (!this.originX && !this.originY) {
      return;
    }
    let width;
    let height;
    if (this.originX) {
      const delta = this.originX.coord - e.clientX;
      width = Math.round(this.originX.value - delta);
    }
    if (this.originY) {
      const delta = this.originY.coord - e.clientY;
      height = Math.round(this.originY.value - delta);
    }
    match.update({ width, height });
  }
}
//# sourceMappingURL=drag_resize_handler.js.map
