import 'chrome://downloads/strings.m.js';
import { css, CrLitElement, html, nothing, render } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
export { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { mojo } from 'chrome://resources/mojo/mojo/public/js/bindings.js';
import { addWebUiListener, removeWebUiListener } from 'chrome://resources/js/cr.js';

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * The class name to set on the document element.
 */
const CLASS_NAME = 'focus-outline-visible';
const docsToManager = new Map();
/**
 * This class sets a CSS class name on the HTML element of |doc| when the user
 * presses a key. It removes the class name when the user clicks anywhere.
 *
 * This allows you to write CSS like this:
 *
 * html.focus-outline-visible my-element:focus {
 *   outline: 5px auto -webkit-focus-ring-color;
 * }
 *
 * And the outline will only be shown if the user uses the keyboard to get to
 * it.
 *
 */
class FocusOutlineManager {
    // Whether focus change is triggered by a keyboard event.
    focusByKeyboard_ = true;
    classList_;
    /**
     * @param doc The document to attach the focus outline manager to.
     */
    constructor(doc) {
        this.classList_ = doc.documentElement.classList;
        doc.addEventListener('keydown', (e) => this.onEvent_(true, e), true);
        doc.addEventListener('mousedown', (e) => this.onEvent_(false, e), true);
        this.updateVisibility();
    }
    onEvent_(focusByKeyboard, e) {
        if (this.focusByKeyboard_ === focusByKeyboard) {
            return;
        }
        if (e instanceof KeyboardEvent && e.repeat) {
            // A repeated keydown should not trigger the focus state. For example,
            // there is a repeated ALT keydown if ALT+CLICK is used to open the
            // context menu and ALT is not released.
            return;
        }
        this.focusByKeyboard_ = focusByKeyboard;
        this.updateVisibility();
    }
    updateVisibility() {
        this.visible = this.focusByKeyboard_;
    }
    /**
     * Whether the focus outline should be visible.
     */
    set visible(visible) {
        this.classList_.toggle(CLASS_NAME, visible);
    }
    get visible() {
        return this.classList_.contains(CLASS_NAME);
    }
    /**
     * Gets a per document singleton focus outline manager.
     * @param doc The document to get the |FocusOutlineManager| for.
     * @return The per document singleton focus outline manager.
     */
    static forDocument(doc) {
        let manager = docsToManager.get(doc);
        if (!manager) {
            manager = new FocusOutlineManager(doc);
            docsToManager.set(doc, manager);
        }
        return manager;
    }
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Verify |value| is truthy.
 * @param value A value to check for truthiness. Note that this
 *     may be used to test whether |value| is defined or not, and we don't want
 *     to force a cast to boolean.
 */
function assert(value, message) {
    if (value) {
        return;
    }
    throw new Error('Assertion failed' + (message ? `: ${message}` : ''));
}
function assertInstanceof(value, type, message) {
    if (value instanceof type) {
        return;
    }
    throw new Error(`Value ${value} is not of type ${type.name || typeof type}`);
}
/**
 * Call this from places in the code that should never be reached.
 *
 * For example, handling all the values of enum with a switch() like this:
 *
 *   function getValueFromEnum(enum) {
 *     switch (enum) {
 *       case ENUM_FIRST_OF_TWO:
 *         return first
 *       case ENUM_LAST_OF_TWO:
 *         return last;
 *     }
 *     assertNotReached();
 *   }
 *
 * This code should only be hit in the case of serious programmer error or
 * unexpected input.
 */
function assertNotReached(message = 'Unreachable code hit') {
    assert(false, message);
}

// Copyright 2011 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview EventTracker is a simple class that manages the addition and
 * removal of DOM event listeners. In particular, it keeps track of all
 * listeners that have been added and makes it easy to remove some or all of
 * them without requiring all the information again. This is particularly handy
 * when the listener is a generated function such as a lambda or the result of
 * calling Function.bind.
 */
class EventTracker {
    listeners_ = [];
    /**
     * Add an event listener - replacement for EventTarget.addEventListener.
     * @param target The DOM target to add a listener to.
     * @param eventType The type of event to subscribe to.
     * @param listener The listener to add.
     * @param capture Whether to invoke during the capture phase. Defaults to
     *     false.
     */
    add(target, eventType, listener, capture = false) {
        const h = {
            target: target,
            eventType: eventType,
            listener: listener,
            capture: capture,
        };
        this.listeners_.push(h);
        target.addEventListener(eventType, listener, capture);
    }
    /**
     * Remove any specified event listeners added with this EventTracker.
     * @param target The DOM target to remove a listener from.
     * @param eventType The type of event to remove.
     */
    remove(target, eventType) {
        this.listeners_ = this.listeners_.filter(listener => {
            if (listener.target === target &&
                (!eventType || (listener.eventType === eventType))) {
                EventTracker.removeEventListener(listener);
                return false;
            }
            return true;
        });
    }
    /** Remove all event listeners added with this EventTracker. */
    removeAll() {
        this.listeners_.forEach(listener => EventTracker.removeEventListener(listener));
        this.listeners_ = [];
    }
    /**
     * Remove a single event listener given it's tracking entry. It's up to the
     * caller to ensure the entry is removed from listeners_.
     * @param entry The entry describing the listener to
     * remove.
     */
    static removeEventListener(entry) {
        entry.target.removeEventListener(entry.eventType, entry.listener, entry.capture);
    }
}

let instance$s = null;
function getCss$p() {
    return instance$s || (instance$s = [...[], css `:host{bottom:0;display:block;left:0;overflow:hidden;pointer-events:none;position:absolute;right:0;top:0;transform:translate3d(0,0,0)}.ripple{background-color:currentcolor;left:0;opacity:var(--paper-ripple-opacity,0.25);pointer-events:none;position:absolute;will-change:height,transform,width}.ripple,:host(.circle){border-radius:50%}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const MAX_RADIUS_PX = 300;
const MIN_DURATION_MS = 800;
/** @return The distance between (x1, y1) and (x2, y2). */
function distance(x1, y1, x2, y2) {
    const xDelta = x1 - x2;
    const yDelta = y1 - y2;
    return Math.sqrt(xDelta * xDelta + yDelta * yDelta);
}
class CrRippleElement extends CrLitElement {
    static get is() {
        return 'cr-ripple';
    }
    static get styles() {
        return getCss$p();
    }
    static get properties() {
        return {
            holdDown: { type: Boolean },
            recenters: { type: Boolean },
            noink: { type: Boolean },
        };
    }
    #holdDown_accessor_storage = false;
    get holdDown() { return this.#holdDown_accessor_storage; }
    set holdDown(value) { this.#holdDown_accessor_storage = value; }
    #recenters_accessor_storage = false;
    get recenters() { return this.#recenters_accessor_storage; }
    set recenters(value) { this.#recenters_accessor_storage = value; }
    #noink_accessor_storage = false;
    get noink() { return this.#noink_accessor_storage; }
    set noink(value) { this.#noink_accessor_storage = value; }
    ripples_ = [];
    eventTracker_ = new EventTracker();
    connectedCallback() {
        super.connectedCallback();
        assert(this.parentNode);
        const keyEventTarget = this.parentNode.nodeType === Node.DOCUMENT_FRAGMENT_NODE ?
            this.parentNode.host :
            this.parentElement;
        this.eventTracker_.add(keyEventTarget, 'pointerdown', (e) => this.uiDownAction(e));
        this.eventTracker_.add(keyEventTarget, 'pointerup', () => this.uiUpAction());
        // 'pointerup' does not fire if the pointer is moved outside the bounds of
        // `keyEventTarget` before releasing, so also listen for `pointerout`.
        this.eventTracker_.add(keyEventTarget, 'pointerout', () => this.uiUpAction());
        this.eventTracker_.add(keyEventTarget, 'keydown', (e) => {
            if (e.defaultPrevented) {
                return;
            }
            if (e.key === 'Enter') {
                this.onEnterKeydown_();
                return;
            }
            if (e.key === ' ') {
                this.onSpaceKeydown_();
            }
        });
        this.eventTracker_.add(keyEventTarget, 'keyup', (e) => {
            if (e.defaultPrevented) {
                return;
            }
            if (e.key === ' ') {
                this.onSpaceKeyup_();
            }
        });
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.eventTracker_.removeAll();
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('holdDown')) {
            this.holdDownChanged_(this.holdDown, changedProperties.get('holdDown'));
        }
    }
    uiDownAction(e) {
        if (e !== undefined && e.button !== 0) {
            // Ignore secondary mouse button clicks.
            return;
        }
        if (!this.noink) {
            this.downAction_(e);
        }
    }
    downAction_(e) {
        if (this.ripples_.length && this.holdDown) {
            return;
        }
        this.showRipple_(e);
    }
    clear() {
        this.hideRipple_();
        this.holdDown = false;
    }
    showAndHoldDown() {
        this.ripples_.forEach(ripple => {
            ripple.remove();
        });
        this.ripples_ = [];
        this.holdDown = true;
    }
    showRipple_(e) {
        const rect = this.getBoundingClientRect();
        const roundedCenterX = function () {
            return Math.round(rect.width / 2);
        };
        const roundedCenterY = function () {
            return Math.round(rect.height / 2);
        };
        let x = 0;
        let y = 0;
        const centered = !e;
        if (centered) {
            x = roundedCenterX();
            y = roundedCenterY();
        }
        else {
            x = Math.round(e.clientX - rect.left);
            y = Math.round(e.clientY - rect.top);
        }
        const corners = [
            { x: 0, y: 0 },
            { x: rect.width, y: 0 },
            { x: 0, y: rect.height },
            { x: rect.width, y: rect.height },
        ];
        const cornerDistances = corners.map(function (corner) {
            return Math.round(distance(x, y, corner.x, corner.y));
        });
        const radius = Math.min(MAX_RADIUS_PX, Math.max.apply(Math, cornerDistances));
        const startTranslate = `${x - radius}px, ${y - radius}px`;
        let endTranslate = startTranslate;
        if (this.recenters && !centered) {
            endTranslate =
                `${roundedCenterX() - radius}px, ${roundedCenterY() - radius}px`;
        }
        const ripple = document.createElement('div');
        ripple.classList.add('ripple');
        ripple.style.height = ripple.style.width = (2 * radius) + 'px';
        this.ripples_.push(ripple);
        this.shadowRoot.appendChild(ripple);
        ripple.animate({
            transform: [
                `translate(${startTranslate}) scale(0)`,
                `translate(${endTranslate}) scale(1)`,
            ],
        }, {
            duration: Math.max(MIN_DURATION_MS, Math.log(radius) * radius) || 0,
            easing: 'cubic-bezier(.2, .9, .1, .9)',
            fill: 'forwards',
        });
    }
    uiUpAction() {
        if (!this.noink) {
            this.upAction_();
        }
    }
    upAction_() {
        if (!this.holdDown) {
            this.hideRipple_();
        }
    }
    hideRipple_() {
        if (this.ripples_.length === 0) {
            return;
        }
        this.ripples_.forEach(function (ripple) {
            const opacity = ripple.computedStyleMap().get('opacity');
            if (opacity === null) {
                ripple.remove();
                return;
            }
            const animation = ripple.animate({
                opacity: [opacity.value, 0],
            }, {
                duration: 150,
                fill: 'forwards',
            });
            animation.finished.then(() => {
                ripple.remove();
            });
        });
        this.ripples_ = [];
    }
    onEnterKeydown_() {
        this.uiDownAction();
        window.setTimeout(() => {
            this.uiUpAction();
        }, 1);
    }
    onSpaceKeydown_() {
        this.uiDownAction();
    }
    onSpaceKeyup_() {
        this.uiUpAction();
    }
    holdDownChanged_(newHoldDown, oldHoldDown) {
        if (oldHoldDown === undefined) {
            return;
        }
        if (newHoldDown) {
            this.downAction_();
        }
        else {
            this.upAction_();
        }
    }
}
customElements.define(CrRippleElement.is, CrRippleElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const CrRippleMixin = (superClass) => {
    class CrRippleMixin extends superClass {
        static get properties() {
            return {
                /**
                 * If true, the element will not produce a ripple effect when
                 * interacted with via the pointer.
                 */
                noink: { type: Boolean },
            };
        }
        #noink_accessor_storage = false;
        get noink() { return this.#noink_accessor_storage; }
        set noink(value) { this.#noink_accessor_storage = value; }
        rippleContainer = null;
        ripple_ = null;
        updated(changedProperties) {
            super.updated(changedProperties);
            if (changedProperties.has('noink') && this.hasRipple()) {
                assert(this.ripple_);
                this.ripple_.noink = this.noink;
            }
        }
        ensureRippleOnPointerdown() {
            // 'capture: true' is necessary so that the cr-ripple is created early
            // enough so that it also receives the 'pointerdown' event. Otherwise
            // the ripple is created, but not shown on the 1st click.
            this.addEventListener('pointerdown', () => this.ensureRipple(), { capture: true });
        }
        /**
         * Ensures this element contains a ripple effect. For startup efficiency
         * the ripple effect is dynamically added on demand when needed.
         */
        ensureRipple() {
            if (this.hasRipple()) {
                return;
            }
            this.ripple_ = this.createRipple();
            this.ripple_.noink = this.noink;
            const rippleContainer = this.rippleContainer || this.shadowRoot;
            assert(rippleContainer);
            rippleContainer.appendChild(this.ripple_);
        }
        /**
         * Returns the `<cr-ripple>` element used by this element to create
         * ripple effects. The element's ripple is created on demand, when
         * necessary, and calling this method will force the
         * ripple to be created.
         */
        getRipple() {
            this.ensureRipple();
            assert(this.ripple_);
            return this.ripple_;
        }
        /**
         * Returns true if this element currently contains a ripple effect.
         */
        hasRipple() {
            return Boolean(this.ripple_);
        }
        /**
         * Create the element's ripple effect via creating a `<cr-ripple
         * id="ink">` instance. Override this method to customize the ripple
         * element.
         */
        createRipple() {
            const ripple = document.createElement('cr-ripple');
            ripple.id = 'ink';
            return ripple;
        }
    }
    return CrRippleMixin;
};

let instance$r = null;
function getCss$o() {
    return instance$r || (instance$r = [...[], css `[hidden],:host([hidden]){display:none !important}`]);
}

const sheet = new CSSStyleSheet();
sheet.replaceSync(`html{--google-blue-50-rgb:232,240,254;--google-blue-50:rgb(var(--google-blue-50-rgb));--google-blue-100-rgb:210,227,252;--google-blue-100:rgb(var(--google-blue-100-rgb));--google-blue-200-rgb:174,203,250;--google-blue-200:rgb(var(--google-blue-200-rgb));--google-blue-300-rgb:138,180,248;--google-blue-300:rgb(var(--google-blue-300-rgb));--google-blue-400-rgb:102,157,246;--google-blue-400:rgb(var(--google-blue-400-rgb));--google-blue-500-rgb:66,133,244;--google-blue-500:rgb(var(--google-blue-500-rgb));--google-blue-600-rgb:26,115,232;--google-blue-600:rgb(var(--google-blue-600-rgb));--google-blue-700-rgb:25,103,210;--google-blue-700:rgb(var(--google-blue-700-rgb));--google-blue-800-rgb:24,90,188;--google-blue-800:rgb(var(--google-blue-800-rgb));--google-blue-900-rgb:23,78,166;--google-blue-900:rgb(var(--google-blue-900-rgb));--google-green-50-rgb:230,244,234;--google-green-50:rgb(var(--google-green-50-rgb));--google-green-200-rgb:168,218,181;--google-green-200:rgb(var(--google-green-200-rgb));--google-green-300-rgb:129,201,149;--google-green-300:rgb(var(--google-green-300-rgb));--google-green-400-rgb:91,185,116;--google-green-400:rgb(var(--google-green-400-rgb));--google-green-500-rgb:52,168,83;--google-green-500:rgb(var(--google-green-500-rgb));--google-green-600-rgb:30,142,62;--google-green-600:rgb(var(--google-green-600-rgb));--google-green-700-rgb:24,128,56;--google-green-700:rgb(var(--google-green-700-rgb));--google-green-800-rgb:19,115,51;--google-green-800:rgb(var(--google-green-800-rgb));--google-green-900-rgb:13,101,45;--google-green-900:rgb(var(--google-green-900-rgb));--google-grey-50-rgb:248,249,250;--google-grey-50:rgb(var(--google-grey-50-rgb));--google-grey-100-rgb:241,243,244;--google-grey-100:rgb(var(--google-grey-100-rgb));--google-grey-200-rgb:232,234,237;--google-grey-200:rgb(var(--google-grey-200-rgb));--google-grey-300-rgb:218,220,224;--google-grey-300:rgb(var(--google-grey-300-rgb));--google-grey-400-rgb:189,193,198;--google-grey-400:rgb(var(--google-grey-400-rgb));--google-grey-500-rgb:154,160,166;--google-grey-500:rgb(var(--google-grey-500-rgb));--google-grey-600-rgb:128,134,139;--google-grey-600:rgb(var(--google-grey-600-rgb));--google-grey-700-rgb:95,99,104;--google-grey-700:rgb(var(--google-grey-700-rgb));--google-grey-800-rgb:60,64,67;--google-grey-800:rgb(var(--google-grey-800-rgb));--google-grey-900-rgb:32,33,36;--google-grey-900:rgb(var(--google-grey-900-rgb));--google-grey-900-white-4-percent:#292a2d;--google-purple-200-rgb:215,174,251;--google-purple-200:rgb(var(--google-purple-200-rgb));--google-purple-900-rgb:104,29,168;--google-purple-900:rgb(var(--google-purple-900-rgb));--google-red-100-rgb:244,199,195;--google-red-100:rgb(var(--google-red-100-rgb));--google-red-300-rgb:242,139,130;--google-red-300:rgb(var(--google-red-300-rgb));--google-red-500-rgb:234,67,53;--google-red-500:rgb(var(--google-red-500-rgb));--google-red-600-rgb:217,48,37;--google-red-600:rgb(var(--google-red-600-rgb));--google-red-700-rgb:197,57,41;--google-red-700:rgb(var(--google-red-700-rgb));--google-yellow-50-rgb:254,247,224;--google-yellow-50:rgb(var(--google-yellow-50-rgb));--google-yellow-100-rgb:254,239,195;--google-yellow-100:rgb(var(--google-yellow-100-rgb));--google-yellow-200-rgb:253,226,147;--google-yellow-200:rgb(var(--google-yellow-200-rgb));--google-yellow-300-rgb:253,214,51;--google-yellow-300:rgb(var(--google-yellow-300-rgb));--google-yellow-400-rgb:252,201,52;--google-yellow-400:rgb(var(--google-yellow-400-rgb));--google-yellow-500-rgb:251,188,4;--google-yellow-500:rgb(var(--google-yellow-500-rgb));--google-yellow-700-rgb:240,147,0;--google-yellow-700:rgb(var(--google-yellow-700-rgb));--cr-card-background-color:white;--cr-shadow-key-color_:color-mix(in srgb,var(--cr-shadow-color) 30%,transparent);--cr-shadow-ambient-color_:color-mix(in srgb,var(--cr-shadow-color) 15%,transparent);--cr-elevation-1:var(--cr-shadow-key-color_) 0 1px 2px 0,var(--cr-shadow-ambient-color_) 0 1px 3px 1px;--cr-elevation-2:var(--cr-shadow-key-color_) 0 1px 2px 0,var(--cr-shadow-ambient-color_) 0 2px 6px 2px;--cr-elevation-3:var(--cr-shadow-key-color_) 0 1px 3px 0,var(--cr-shadow-ambient-color_) 0 4px 8px 3px;--cr-elevation-4:var(--cr-shadow-key-color_) 0 2px 3px 0,var(--cr-shadow-ambient-color_) 0 6px 10px 4px;--cr-elevation-5:var(--cr-shadow-key-color_) 0 4px 4px 0,var(--cr-shadow-ambient-color_) 0 8px 12px 6px;--cr-card-shadow:var(--cr-elevation-2);--cr-focused-item-color:var(--google-grey-300);--cr-form-field-label-color:var(--google-grey-700);--cr-hairline-rgb:0,0,0;--cr-iph-anchor-highlight-color:rgba(var(--google-blue-600-rgb),0.1);--cr-menu-background-color:white;--cr-menu-background-focus-color:var(--google-grey-400);--cr-menu-shadow:var(--cr-elevation-2);--cr-separator-color:rgba(0,0,0,.06);--cr-title-text-color:rgb(90,90,90);--cr-scrollable-border-color:var(--google-grey-300)}@media (prefers-color-scheme:dark){html{--cr-card-background-color:var(--google-grey-900-white-4-percent);--cr-focused-item-color:var(--google-grey-800);--cr-form-field-label-color:var(--dark-secondary-color);--cr-hairline-rgb:255,255,255;--cr-iph-anchor-highlight-color:rgba(var(--google-grey-100-rgb),0.1);--cr-menu-background-color:var(--google-grey-900);--cr-menu-background-focus-color:var(--google-grey-700);--cr-menu-background-sheen:rgba(255,255,255,.06);--cr-menu-shadow:rgba(0,0,0,.3) 0 1px 2px 0,rgba(0,0,0,.15) 0 3px 6px 2px;--cr-separator-color:rgba(255,255,255,.1);--cr-title-text-color:var(--cr-primary-text-color);--cr-scrollable-border-color:var(--google-grey-700)}}@media (forced-colors:active){html{--cr-focus-outline-hcm:2px solid transparent;--cr-border-hcm:2px solid transparent}}html{--cr-button-edge-spacing:12px;--cr-controlled-by-spacing:24px;--cr-default-input-max-width:264px;--cr-icon-ripple-size:36px;--cr-icon-ripple-padding:8px;--cr-icon-size:20px;--cr-icon-button-margin-start:16px;--cr-icon-ripple-margin:calc(var(--cr-icon-ripple-padding) * -1);--cr-section-min-height:48px;--cr-section-two-line-min-height:64px;--cr-section-padding:20px;--cr-section-vertical-padding:12px;--cr-section-indent-width:40px;--cr-section-indent-padding:calc(var(--cr-section-padding) + var(--cr-section-indent-width));--cr-section-vertical-margin:21px;--cr-centered-card-max-width:680px;--cr-centered-card-width-percentage:0.96;--cr-hairline:1px solid rgba(var(--cr-hairline-rgb),.14);--cr-separator-height:1px;--cr-separator-line:var(--cr-separator-height) solid var(--cr-separator-color);--cr-toolbar-overlay-animation-duration:150ms;--cr-toolbar-height:56px;--cr-container-shadow-height:6px;--cr-container-shadow-margin:calc(-1 * var(--cr-container-shadow-height));--cr-container-shadow-max-opacity:1;--cr-card-border-radius:8px;--cr-disabled-opacity:.38;--cr-form-field-bottom-spacing:16px;--cr-form-field-label-font-size:.625rem;--cr-form-field-label-height:1em;--cr-form-field-label-line-height:1}html{--cr-fallback-color-outline:rgb(116,119,117);--cr-fallback-color-primary:rgb(11,87,208);--cr-fallback-color-on-primary:rgb(255,255,255);--cr-fallback-color-primary-container:rgb(211,227,253);--cr-fallback-color-on-primary-container:rgb(4,30,73);--cr-fallback-color-secondary-container:rgb(194,231,255);--cr-fallback-color-on-secondary-container:rgb(0,29,53);--cr-fallback-color-neutral-container:rgb(242,242,242);--cr-fallback-color-neutral-outline:rgb(199,199,199);--cr-fallback-color-surface:rgb(255,255,255);--cr-fallback-color-surface1:rgb(248,250,253);--cr-fallback-color-surface2:rgb(243,246,252);--cr-fallback-color-surface3:rgb(239,243,250);--cr-fallback-color-on-surface-rgb:31,31,31;--cr-fallback-color-on-surface:rgb(var(--cr-fallback-color-on-surface-rgb));--cr-fallback-color-surface-variant:rgb(225,227,225);--cr-fallback-color-on-surface-variant:rgb(68,71,70);--cr-fallback-color-on-surface-subtle:rgb(71,71,71);--cr-fallback-color-inverse-primary:rgb(168,199,250);--cr-fallback-color-inverse-surface:rgb(48,48,48);--cr-fallback-color-inverse-on-surface:rgb(242,242,242);--cr-fallback-color-tonal-container:rgb(211,227,253);--cr-fallback-color-on-tonal-container:rgb(4,30,73);--cr-fallback-color-tonal-outline:rgb(168,199,250);--cr-fallback-color-error:rgb(179,38,30);--cr-fallback-color-divider:rgb(211,227,253);--cr-fallback-color-state-hover-on-prominent_:rgba(253,252,251,.1);--cr-fallback-color-state-on-subtle-rgb_:31,31,31;--cr-fallback-color-state-hover-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.06);--cr-fallback-color-state-ripple-neutral-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.08);--cr-fallback-color-state-ripple-primary-rgb_:124,172,248;--cr-fallback-color-state-ripple-primary_:rgba(var(--cr-fallback-color-state-ripple-primary-rgb_),0.32);--cr-fallback-color-base-container:rgb(236,239,247);--cr-fallback-color-disabled-background:rgba(var(--cr-fallback-color-on-surface-rgb),.12);--cr-fallback-color-disabled-foreground:rgba(var(--cr-fallback-color-on-surface-rgb),var(--cr-disabled-opacity));--cr-hover-background-color:var(--color-sys-state-hover,rgba(var(--cr-fallback-color-on-surface-rgb),.08));--cr-hover-on-prominent-background-color:var(--color-sys-state-hover-on-prominent,var(--cr-fallback-color-state-hover-on-prominent_));--cr-hover-on-subtle-background-color:var(--color-sys-state-hover-on-subtle,var(--cr-fallback-color-state-hover-on-subtle_));--cr-active-background-color:var(--color-sys-state-pressed,rgba(var(--cr-fallback-color-on-surface-rgb),.12));--cr-active-on-primary-background-color:var(--color-sys-state-ripple-primary,var(--cr-fallback-color-state-ripple-primary_));--cr-active-neutral-on-subtle-background-color:var(--color-sys-state-ripple-neutral-on-subtle,var(--cr-fallback-color-state-ripple-neutral-on-subtle_));--cr-focus-outline-color:var(--color-sys-state-focus-ring,var(--cr-fallback-color-primary));--cr-focus-outline-inverse-color:var(--color-sys-state-focus-ring-inverse,var(--cr-fallback-color-inverse-primary));--cr-primary-text-color:var(--color-primary-foreground,var(--cr-fallback-color-on-surface));--cr-secondary-text-color:var(--color-secondary-foreground,var(--cr-fallback-color-on-surface-variant));--cr-link-color:var(--color-link-foreground-default,var(--cr-fallback-color-primary));--cr-button-height:36px;--cr-shadow-color:var(--color-sys-shadow,rgb(0,0,0));--cr-checked-color:var(--color-checkbox-foreground-checked,var(--cr-fallback-color-primary))}@media (prefers-color-scheme:dark){html{--cr-fallback-color-outline:rgb(142,145,143);--cr-fallback-color-primary:rgb(168,199,250);--cr-fallback-color-on-primary:rgb(6,46,111);--cr-fallback-color-primary-container:rgb(8,66,160);--cr-fallback-color-on-primary-container:rgb(211,227,253);--cr-fallback-color-secondary-container:rgb(0,74,119);--cr-fallback-color-on-secondary-container:rgb(194,231,255);--cr-fallback-color-neutral-container:rgb(40,40,40);--cr-fallback-color-neutral-outline:rgb(117,117,117);--cr-fallback-color-surface:rgb(31,31,31);--cr-fallback-color-surface1:rgb(39,40,42);--cr-fallback-color-surface2:rgb(45,47,49);--cr-fallback-color-surface3:rgb(51,52,56);--cr-fallback-color-on-surface-rgb:227,227,227;--cr-fallback-color-surface-variant:rgb(68,71,70);--cr-fallback-color-on-surface-variant:rgb(196,199,197);--cr-fallback-color-on-surface-subtle:rgb(199,199,199);--cr-fallback-color-inverse-primary:rgb(11,87,208);--cr-fallback-color-inverse-surface:rgb(227,227,227);--cr-fallback-color-inverse-on-surface:rgb(31,31,31);--cr-fallback-color-tonal-container:rgb(0,74,119);--cr-fallback-color-on-tonal-container:rgb(194,231,255);--cr-fallback-color-tonal-outline:rgb(4,125,183);--cr-fallback-color-error:rgb(242,184,181);--cr-fallback-color-divider:rgb(94,94,94);--cr-fallback-color-state-hover-on-prominent_:rgba(31,31,31,.06);--cr-fallback-color-state-on-subtle-rgb_:253,252,251;--cr-fallback-color-state-hover-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.10);--cr-fallback-color-state-ripple-neutral-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.16);--cr-fallback-color-state-ripple-primary-rgb_:76,141,246;--cr-fallback-color-base-container:rgba(40,40,40,1)}}@media (forced-colors:active){html{--cr-fallback-color-disabled-background:Canvas;--cr-fallback-color-disabled-foreground:GrayText}}`);
document.adoptedStyleSheets = [...document.adoptedStyleSheets, sheet];

let instance$q = null;
function getCss$n() {
    return instance$q || (instance$q = [...[getCss$o()], css `:host{--cr-button-background-color:transparent;--cr-button-border-color:var(--color-button-border,var(--cr-fallback-color-tonal-outline));--cr-button-text-color:var(--color-button-foreground,var(--cr-fallback-color-primary));--cr-button-ripple-opacity:1;--cr-button-ripple-color:var(--cr-active-background-color);--cr-button-disabled-background-color:transparent;--cr-button-disabled-border-color:var(--color-button-border-disabled,var(--cr-fallback-color-disabled-background));--cr-button-disabled-text-color:var(--color-button-foreground-disabled,var(--cr-fallback-color-disabled-foreground))}:host(.action-button){--cr-button-background-color:var(--color-button-background-prominent,var(--cr-fallback-color-primary));--cr-button-text-color:var(--color-button-foreground-prominent,var(--cr-fallback-color-on-primary));--cr-button-ripple-color:var(--cr-active-on-primary-background-color);--cr-button-border:none;--cr-button-disabled-background-color:var(--color-button-background-prominent-disabled,var(--cr-fallback-color-disabled-background));--cr-button-disabled-text-color:var(--color-button-foreground-disabled,var(--cr-fallback-color-disabled-foreground));--cr-button-disabled-border:none}:host(.tonal-button),:host(.floating-button){--cr-button-background-color:var(--color-button-background-tonal,var(--cr-fallback-color-secondary-container));--cr-button-text-color:var(--color-button-foreground-tonal,var(--cr-fallback-color-on-tonal-container));--cr-button-border:none;--cr-button-disabled-background-color:var(--color-button-background-tonal-disabled,var(--cr-fallback-color-disabled-background));--cr-button-disabled-text-color:var(--color-button-foreground-disabled,var(--cr-fallback-color-disabled-foreground));--cr-button-disabled-border:none}:host{flex-shrink:0;display:inline-flex;align-items:center;justify-content:center;box-sizing:border-box;min-width:5.14em;height:var(--cr-button-height);padding:8px 16px;outline-width:0;overflow:hidden;position:relative;cursor:pointer;user-select:none;-webkit-tap-highlight-color:transparent;border:var(--cr-button-border,1px solid var(--cr-button-border-color));border-radius:100px;background:var(--cr-button-background-color);color:var(--cr-button-text-color);font-weight:500;line-height:20px;isolation:isolate}@media (forced-colors:active){:host{forced-color-adjust:none}}:host(.floating-button){border-radius:8px;height:40px;transition:box-shadow 80ms linear}:host(.floating-button:hover){box-shadow:var(--cr-elevation-3)}:host([has-prefix-icon_]),:host([has-suffix-icon_]){--iron-icon-height:20px;--iron-icon-width:20px;--icon-block-padding-large:16px;--icon-block-padding-small:12px;gap:8px;padding-block-end:8px;padding-block-start:8px}:host([has-prefix-icon_]){padding-inline-end:var(--icon-block-padding-large);padding-inline-start:var(--icon-block-padding-small)}:host([has-suffix-icon_]){padding-inline-end:var(--icon-block-padding-small);padding-inline-start:var(--icon-block-padding-large)}:host-context(.focus-outline-visible):host(:focus){box-shadow:none;outline:2px solid var(--cr-focus-outline-color);outline-offset:2px}#background{border-radius:inherit;inset:0;pointer-events:none;position:absolute}#content{display:inline}#hoverBackground{content:'';display:none;inset:0;pointer-events:none;position:absolute;z-index:1}:host(:hover) #hoverBackground{background:var(--cr-hover-background-color);display:block}:host(.action-button:hover) #hoverBackground{background:var(--cr-hover-on-prominent-background-color)}:host([disabled]){background:var(--cr-button-disabled-background-color);border:var(--cr-button-disabled-border,1px solid var(--cr-button-disabled-border-color));color:var(--cr-button-disabled-text-color);cursor:auto;pointer-events:none}:host(.cancel-button){margin-inline-end:8px}:host(.action-button),:host(.cancel-button){line-height:154%}#ink{color:var(--cr-button-ripple-color);--paper-ripple-opacity:var(--cr-button-ripple-opacity)}#background{z-index:0}#hoverBackground,cr-ripple{z-index:1}#content,::slotted(*){z-index:2}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$e() {
    return html `
<div id="background"></div>
<slot id="prefixIcon" name="prefix-icon"
    @slotchange="${this.onPrefixIconSlotChanged_}">
</slot>
<span id="content"><slot></slot></span>
<slot id="suffixIcon" name="suffix-icon"
    @slotchange="${this.onSuffixIconSlotChanged_}">
</slot>
<div id="hoverBackground" part="hoverBackground"></div>`;
}

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-button' is a button which displays slotted elements. It can
 * be interacted with like a normal button using click as well as space and
 * enter to effectively click the button and fire a 'click' event. It can also
 * style an icon inside of the button with the [has-icon] attribute.
 */
const CrButtonElementBase = CrRippleMixin(CrLitElement);
class CrButtonElement extends CrButtonElementBase {
    static get is() {
        return 'cr-button';
    }
    static get styles() {
        return getCss$n();
    }
    render() {
        return getHtml$e.bind(this)();
    }
    static get properties() {
        return {
            disabled: {
                type: Boolean,
                reflect: true,
            },
            hasPrefixIcon_: {
                type: Boolean,
                reflect: true,
            },
            hasSuffixIcon_: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #hasPrefixIcon__accessor_storage = false;
    get hasPrefixIcon_() { return this.#hasPrefixIcon__accessor_storage; }
    set hasPrefixIcon_(value) { this.#hasPrefixIcon__accessor_storage = value; }
    #hasSuffixIcon__accessor_storage = false;
    get hasSuffixIcon_() { return this.#hasSuffixIcon__accessor_storage; }
    set hasSuffixIcon_(value) { this.#hasSuffixIcon__accessor_storage = value; }
    /**
     * It is possible to activate a tab when the space key is pressed down. When
     * this element has focus, the keyup event for the space key should not
     * perform a 'click'. |spaceKeyDown_| tracks when a space pressed and
     * handled by this element. Space keyup will only result in a 'click' when
     * |spaceKeyDown_| is true. |spaceKeyDown_| is set to false when element
     * loses focus.
     */
    spaceKeyDown_ = false;
    timeoutIds_ = new Set();
    constructor() {
        super();
        this.addEventListener('blur', this.onBlur_.bind(this));
        // Must be added in constructor so that stopImmediatePropagation() works as
        // expected.
        this.addEventListener('click', this.onClick_.bind(this));
        this.addEventListener('keydown', this.onKeyDown_.bind(this));
        this.addEventListener('keyup', this.onKeyUp_.bind(this));
        this.ensureRippleOnPointerdown();
    }
    firstUpdated() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'button');
        }
        if (!this.hasAttribute('tabindex')) {
            this.setAttribute('tabindex', '0');
        }
        FocusOutlineManager.forDocument(document);
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('disabled')) {
            this.setAttribute('aria-disabled', this.disabled ? 'true' : 'false');
            this.disabledChanged_(this.disabled, changedProperties.get('disabled'));
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.timeoutIds_.forEach(clearTimeout);
        this.timeoutIds_.clear();
    }
    setTimeout_(fn, delay) {
        if (!this.isConnected) {
            return;
        }
        const id = setTimeout(() => {
            this.timeoutIds_.delete(id);
            fn();
        }, delay);
        this.timeoutIds_.add(id);
    }
    disabledChanged_(newValue, oldValue) {
        if (!newValue && oldValue === undefined) {
            return;
        }
        if (this.disabled) {
            this.blur();
        }
        this.setAttribute('tabindex', String(this.disabled ? -1 : 0));
    }
    onBlur_() {
        this.spaceKeyDown_ = false;
        // If a keyup event is never fired (e.g. after keydown the focus is moved to
        // another element), we need to clear the ripple here. 100ms delay was
        // chosen manually as a good time period for the ripple to be visible.
        this.setTimeout_(() => this.getRipple().uiUpAction(), 100);
    }
    onClick_(e) {
        if (this.disabled) {
            e.stopImmediatePropagation();
        }
    }
    onPrefixIconSlotChanged_() {
        this.hasPrefixIcon_ = this.$.prefixIcon.assignedElements().length > 0;
    }
    onSuffixIconSlotChanged_() {
        this.hasSuffixIcon_ = this.$.suffixIcon.assignedElements().length > 0;
    }
    onKeyDown_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (e.repeat) {
            return;
        }
        this.getRipple().uiDownAction();
        if (e.key === 'Enter') {
            this.click();
            // Delay was chosen manually as a good time period for the ripple to be
            // visible.
            this.setTimeout_(() => this.getRipple().uiUpAction(), 100);
        }
        else if (e.key === ' ') {
            this.spaceKeyDown_ = true;
        }
    }
    onKeyUp_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (this.spaceKeyDown_ && e.key === ' ') {
            this.spaceKeyDown_ = false;
            this.click();
            this.getRipple().uiUpAction();
        }
    }
}
customElements.define(CrButtonElement.is, CrButtonElement);

let instance$p = null;
function getCss$m() {
    return instance$p || (instance$p = [...[getCss$o()], css `:host{align-items:center;display:inline-flex;justify-content:center;position:relative;vertical-align:middle;fill:var(--iron-icon-fill-color,currentcolor);stroke:var(--iron-icon-stroke-color,none);width:var(--iron-icon-width,24px);height:var(--iron-icon-height,24px)}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
let iconsetMap = null;
class IconsetMap extends EventTarget {
    iconsets_ = new Map();
    static getInstance() {
        return iconsetMap || (iconsetMap = new IconsetMap());
    }
    static resetInstanceForTesting(instance) {
        iconsetMap = instance;
    }
    get(id) {
        return this.iconsets_.get(id) || null;
    }
    set(id, iconset) {
        assert(!this.iconsets_.has(id), `Tried to add a second iconset with id '${id}'`);
        this.iconsets_.set(id, iconset);
        this.dispatchEvent(new CustomEvent('cr-iconset-added', { detail: id }));
    }
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class CrIconElement extends CrLitElement {
    static get is() {
        return 'cr-icon';
    }
    static get styles() {
        return getCss$m();
    }
    static get properties() {
        return {
            /**
             * The name of the icon to use. The name should be of the form:
             * `iconset_name:icon_name`.
             */
            icon: { type: String },
        };
    }
    #icon_accessor_storage = '';
    get icon() { return this.#icon_accessor_storage; }
    set icon(value) { this.#icon_accessor_storage = value; }
    iconsetName_ = '';
    iconName_ = '';
    iconset_ = null;
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('icon')) {
            const [iconsetName, iconName] = this.icon.split(':');
            this.iconName_ = iconName || '';
            this.iconsetName_ = iconsetName || '';
            this.updateIcon_();
        }
    }
    updateIcon_() {
        if (this.iconName_ === '' && this.iconset_) {
            this.iconset_.removeIcon(this);
        }
        else if (this.iconsetName_) {
            const iconsetMap = IconsetMap.getInstance();
            this.iconset_ = iconsetMap.get(this.iconsetName_);
            assert(this.iconset_, `Could not find iconset for: '${this.iconsetName_}:${this.iconName_}'`);
            this.iconset_.applyIcon(this, this.iconName_);
        }
    }
}
customElements.define(CrIconElement.is, CrIconElement);

let instance$o = null;
function getCss$l() {
    return instance$o || (instance$o = [...[], css `:host{--cr-icon-button-fill-color:currentColor;--cr-icon-button-icon-start-offset:0;--cr-icon-button-icon-size:20px;--cr-icon-button-size:32px;--cr-icon-button-height:var(--cr-icon-button-size);--cr-icon-button-transition:150ms ease-in-out;--cr-icon-button-width:var(--cr-icon-button-size);-webkit-tap-highlight-color:transparent;border-radius:50%;color:var(--cr-icon-button-stroke-color,var(--cr-icon-button-fill-color));cursor:pointer;display:inline-flex;flex-shrink:0;height:var(--cr-icon-button-height);margin-inline-end:var(--cr-icon-button-margin-end,var(--cr-icon-ripple-margin));margin-inline-start:var(--cr-icon-button-margin-start);outline:none;overflow:hidden;position:relative;user-select:none;vertical-align:middle;width:var(--cr-icon-button-width)}:host(:hover){background-color:var(--cr-icon-button-hover-background-color,var(--cr-hover-background-color))}:host(:focus-visible:focus){box-shadow:inset 0 0 0 2px var(--cr-icon-button-focus-outline-color,var(--cr-focus-outline-color))}@media (forced-colors:active){:host(:focus-visible:focus){outline:var(--cr-focus-outline-hcm)}}#ink{--paper-ripple-opacity:1;color:var(--cr-icon-button-active-background-color,var(--cr-active-background-color))}:host([disabled]){cursor:initial;opacity:var(--cr-disabled-opacity);pointer-events:none}:host(.no-overlap){--cr-icon-button-margin-end:0;--cr-icon-button-margin-start:0}:host-context([dir=rtl]):host(:not([suppress-rtl-flip]):not([multiple-icons_])){transform:scaleX(-1)}:host-context([dir=rtl]):host(:not([suppress-rtl-flip])[multiple-icons_]) cr-icon{transform:scaleX(-1)}:host(:not([iron-icon])) #maskedImage{-webkit-mask-image:var(--cr-icon-image);-webkit-mask-position:center;-webkit-mask-repeat:no-repeat;-webkit-mask-size:var(--cr-icon-button-icon-size);-webkit-transform:var(--cr-icon-image-transform,none);background-color:var(--cr-icon-button-fill-color);height:100%;transition:background-color var(--cr-icon-button-transition);width:100%}@media (forced-colors:active){:host(:not([iron-icon])) #maskedImage{background-color:ButtonText}}#icon{align-items:center;border-radius:4px;display:flex;height:100%;justify-content:center;padding-inline-start:var(--cr-icon-button-icon-start-offset);position:relative;width:100%}cr-icon{--iron-icon-fill-color:var(--cr-icon-button-fill-color);--iron-icon-stroke-color:var(--cr-icon-button-stroke-color,none);--iron-icon-height:var(--cr-icon-button-icon-size);--iron-icon-width:var(--cr-icon-button-icon-size);transition:fill var(--cr-icon-button-transition),stroke var(--cr-icon-button-transition)}@media (prefers-color-scheme:dark){:host{--cr-icon-button-fill-color:var(--google-grey-500)}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$d() {
    return html `
<div id="icon">
  <div id="maskedImage"></div>
</div>`;
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-icon-button' is a button which displays an icon with a
 * ripple. It can be interacted with like a normal button using click as well as
 * space and enter to effectively click the button and fire a 'click' event.
 *
 * There are two sources to icons:
 * Option 1: CSS classes defined in cr_icons.css.
 * Option 2: SVG icons defined in a cr-iconset or iron-iconset-svg,
 *     with the name passed to cr-icon-button via the |ironIcon| property.
 *
 * Example of using CSS classes:
 * In the .html.ts template file (if using a .html template file instead, the
 * import should be in the corresponding .ts file):
 * import 'chrome://resources/cr_elements/cr_icons.css.js';
 *
 * export function getHtml() {
 *   return html`
 *     <cr-icon-button class="icon-class-name"></cr-icon-button>`;
 * }
 *
 * When an icon is specified using a class, the expectation is the
 * class will set an image to the --cr-icon-image variable.
 *
 * Example of using a cr-iconset to supply an icon via the iron-icon parameter:
 * In the .html.ts template file (if using a .html template file instead, the
 * import should be in the corresponding .ts file):
 * import 'chrome://resources/cr_elements/icons.html.js';
 *
 * export function getHtml() {
 *   return html`
 *     <cr-icon-button iron-icon="cr:icon-key"></cr-icon-button>`;
 * }
 *
 * The color of the icon can be overridden using CSS variables. When using
 * the ironIcon property to populate cr-icon-button's internal <cr-icon>, the
 * following CSS variables for fill and stroke can be overridden for cr-icon:
 * --iron-icon-button-fill-color
 * --iron-icon-button-stroke-color
 *
 * When not using the ironIcon property, cr-icon-button will not create a
 * <cr-icon>, so the cr-icon related CSS variables above are ignored.
 *
 * When using the ironIcon property, more than one icon can be specified by
 * setting the |ironIcon| property to a comma-delimited list of keys.
 */
const CrIconbuttonElementBase = CrRippleMixin(CrLitElement);
class CrIconButtonElement extends CrIconbuttonElementBase {
    static get is() {
        return 'cr-icon-button';
    }
    static get styles() {
        return getCss$l();
    }
    render() {
        return getHtml$d.bind(this)();
    }
    static get properties() {
        return {
            disabled: {
                type: Boolean,
                reflect: true,
            },
            ironIcon: {
                type: String,
                reflect: true,
            },
            suppressRtlFlip: {
                type: Boolean,
                value: false,
                reflect: true,
            },
            multipleIcons_: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #ironIcon_accessor_storage;
    get ironIcon() { return this.#ironIcon_accessor_storage; }
    set ironIcon(value) { this.#ironIcon_accessor_storage = value; }
    #multipleIcons__accessor_storage = false;
    get multipleIcons_() { return this.#multipleIcons__accessor_storage; }
    set multipleIcons_(value) { this.#multipleIcons__accessor_storage = value; }
    /**
     * It is possible to activate a tab when the space key is pressed down. When
     * this element has focus, the keyup event for the space key should not
     * perform a 'click'. |spaceKeyDown_| tracks when a space pressed and
     * handled by this element. Space keyup will only result in a 'click' when
     * |spaceKeyDown_| is true. |spaceKeyDown_| is set to false when element
     * loses focus.
     */
    spaceKeyDown_ = false;
    constructor() {
        super();
        this.addEventListener('blur', this.onBlur_.bind(this));
        this.addEventListener('click', this.onClick_.bind(this));
        this.addEventListener('keydown', this.onKeyDown_.bind(this));
        this.addEventListener('keyup', this.onKeyUp_.bind(this));
        this.ensureRippleOnPointerdown();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('ironIcon')) {
            const icons = (this.ironIcon || '').split(',');
            this.multipleIcons_ = icons.length > 1;
        }
    }
    firstUpdated() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'button');
        }
        if (!this.hasAttribute('tabindex')) {
            this.setAttribute('tabindex', '0');
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('disabled')) {
            this.setAttribute('aria-disabled', this.disabled ? 'true' : 'false');
            this.disabledChanged_(this.disabled, changedProperties.get('disabled'));
        }
        if (changedProperties.has('ironIcon')) {
            this.onIronIconChanged_();
        }
    }
    disabledChanged_(newValue, oldValue) {
        if (!newValue && oldValue === undefined) {
            return;
        }
        if (this.disabled) {
            this.blur();
        }
        this.setAttribute('tabindex', String(this.disabled ? -1 : 0));
    }
    onBlur_() {
        this.spaceKeyDown_ = false;
    }
    onClick_(e) {
        if (this.disabled) {
            e.stopImmediatePropagation();
        }
    }
    onIronIconChanged_() {
        this.shadowRoot.querySelectorAll('cr-icon').forEach(el => el.remove());
        if (!this.ironIcon) {
            return;
        }
        const icons = (this.ironIcon || '').split(',');
        icons.forEach(async (icon) => {
            const crIcon = document.createElement('cr-icon');
            crIcon.icon = icon;
            this.$.icon.appendChild(crIcon);
            await crIcon.updateComplete;
            crIcon.shadowRoot.querySelectorAll('svg, img')
                .forEach(child => child.setAttribute('role', 'none'));
        });
    }
    onKeyDown_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (e.repeat) {
            return;
        }
        if (e.key === 'Enter') {
            this.click();
        }
        else if (e.key === ' ') {
            this.spaceKeyDown_ = true;
        }
    }
    onKeyUp_(e) {
        if (e.key === ' ' || e.key === 'Enter') {
            e.preventDefault();
            e.stopPropagation();
        }
        if (this.spaceKeyDown_ && e.key === ' ') {
            this.spaceKeyDown_ = false;
            this.click();
        }
    }
}
customElements.define(CrIconButtonElement.is, CrIconButtonElement);

let instance$n = null;
function getCss$k() {
    return instance$n || (instance$n = [...[], css `.icon-arrow-back{--cr-icon-image:url(//resources/images/icon_arrow_back.svg)}.icon-arrow-dropdown{--cr-icon-image:url(//resources/images/icon_arrow_dropdown.svg)}.icon-arrow-drop-down-cr23{--cr-icon-image:url(//resources/images/icon_arrow_drop_down_cr23.svg)}.icon-arrow-drop-up-cr23{--cr-icon-image:url(//resources/images/icon_arrow_drop_up_cr23.svg)}.icon-arrow-upward{--cr-icon-image:url(//resources/images/icon_arrow_upward.svg)}.icon-cancel{--cr-icon-image:url(//resources/images/icon_cancel.svg)}.icon-clear{--cr-icon-image:url(//resources/images/icon_clear.svg)}.icon-copy-content{--cr-icon-image:url(//resources/images/icon_copy_content.svg)}.icon-delete-gray{--cr-icon-image:url(//resources/images/icon_delete_gray.svg)}.icon-edit{--cr-icon-image:url(//resources/images/icon_edit.svg)}.icon-file{--cr-icon-image:url(//resources/images/icon_filetype_generic.svg)}.icon-folder-open{--cr-icon-image:url(//resources/images/icon_folder_open.svg)}.icon-picture-delete{--cr-icon-image:url(//resources/images/icon_picture_delete.svg)}.icon-expand-less{--cr-icon-image:url(//resources/images/icon_expand_less.svg)}.icon-expand-more{--cr-icon-image:url(//resources/images/icon_expand_more.svg)}.icon-external{--cr-icon-image:url(//resources/images/open_in_new.svg)}.icon-more-vert{--cr-icon-image:url(//resources/images/icon_more_vert.svg)}.icon-refresh{--cr-icon-image:url(//resources/images/icon_refresh.svg)}.icon-search{--cr-icon-image:url(//resources/images/icon_search.svg)}.icon-settings{--cr-icon-image:url(//resources/images/icon_settings.svg)}.icon-visibility{--cr-icon-image:url(//resources/images/icon_visibility.svg)}.icon-visibility-off{--cr-icon-image:url(//resources/images/icon_visibility_off.svg)}.subpage-arrow{--cr-icon-image:url(//resources/images/arrow_right.svg)}.cr-icon{-webkit-mask-image:var(--cr-icon-image);-webkit-mask-position:center;-webkit-mask-repeat:no-repeat;-webkit-mask-size:var(--cr-icon-size);background-color:var(--cr-icon-color,var(--google-grey-700));flex-shrink:0;height:var(--cr-icon-ripple-size);margin-inline-end:var(--cr-icon-ripple-margin);margin-inline-start:var(--cr-icon-button-margin-start);user-select:none;width:var(--cr-icon-ripple-size)}:host-context([dir=rtl]) .cr-icon{transform:scaleX(-1)}.cr-icon.no-overlap{margin-inline-end:0;margin-inline-start:0}@media (prefers-color-scheme:dark){.cr-icon{background-color:var(--cr-icon-color,var(--google-grey-500))}}`]);
}

let instance$m = null;
function getCss$j() {
    return instance$m || (instance$m = [...[], css `.cr-scrollable{anchor-name:--cr-scrollable;anchor-scope:--cr-scrollable;container-type:scroll-state;overflow:auto}.cr-scrollable-top,.cr-scrollable-top-shadow,.cr-scrollable-bottom{display:none;position:fixed;position-anchor:--cr-scrollable;left:anchor(left);width:anchor-size(width);pointer-events:none;&:where(.force-on){display:block}}.cr-scrollable-top{top:anchor(top);border-top:1px solid var(--cr-scrollable-border-color);@container scroll-state(scrollable:top){display:block}}.cr-scrollable-bottom{bottom:anchor(bottom);border-bottom:1px solid var(--cr-scrollable-border-color);@container scroll-state(scrollable:bottom){display:block}}.cr-scrollable-top-shadow{box-shadow:inset 0 5px 6px -3px rgba(0,0,0,.4);display:block;height:8px;opacity:0;top:anchor(top);transition:opacity 500ms;z-index:1;&:where(.force-on){opacity:1}@container scroll-state(scrollable:top){opacity:1}}`]);
}

let instance$l = null;
function getCss$i() {
    return instance$l || (instance$l = [...[getCss$o(), getCss$k(), getCss$j()], css `dialog{background-color:var(--cr-dialog-background-color,white);border:0;border-radius:var(--cr-dialog-border-radius,8px);bottom:50%;box-shadow:0 0 16px rgba(0,0,0,0.12),0 16px 16px rgba(0,0,0,0.24);color:inherit;line-height:20px;max-height:initial;max-width:initial;overflow-y:hidden;padding:0;position:absolute;top:50%;width:var(--cr-dialog-width,512px)}@media (prefers-color-scheme:dark){dialog{background-color:var(--cr-dialog-background-color,var(--google-grey-900));background-image:linear-gradient(rgba(255,255,255,.04),rgba(255,255,255,.04))}}@media (forced-colors:active){dialog{border:var(--cr-border-hcm)}}dialog[open] #content-wrapper{display:flex;flex-direction:column;max-height:100vh;overflow:auto}.top-container,:host ::slotted([slot=button-container]),:host ::slotted([slot=footer]){flex-shrink:0}dialog::backdrop{background-color:rgba(0,0,0,0.6);bottom:0;left:0;position:fixed;right:0;top:0}:host ::slotted([slot=body]){color:var(--cr-secondary-text-color);padding:0 var(--cr-dialog-body-padding-horizontal,20px)}:host ::slotted([slot=title]){color:var(--cr-primary-text-color);flex:1;font-family:var(--cr-dialog-font-family,inherit);font-size:var(--cr-dialog-title-font-size,calc(15 / 13 * 100%));line-height:1;padding-bottom:var(--cr-dialog-title-slot-padding-bottom,16px);padding-inline-end:var(--cr-dialog-title-slot-padding-end,20px);padding-inline-start:var(--cr-dialog-title-slot-padding-start,20px);padding-top:var(--cr-dialog-title-slot-padding-top,20px)}:host ::slotted([slot=button-container]){display:flex;justify-content:flex-end;padding-bottom:var(--cr-dialog-button-container-padding-bottom,16px);padding-inline-end:var(--cr-dialog-button-container-padding-horizontal,16px);padding-inline-start:var(--cr-dialog-button-container-padding-horizontal,16px);padding-top:var(--cr-dialog-button-container-padding-top,16px)}:host ::slotted([slot=footer]){border-bottom-left-radius:inherit;border-bottom-right-radius:inherit;border-top:1px solid #dbdbdb;margin:0;padding:16px 20px}:host([hide-backdrop]) dialog::backdrop{opacity:0}@media (prefers-color-scheme:dark){:host ::slotted([slot=footer]){border-top-color:var(--cr-separator-color)}}.body-container{box-sizing:border-box;display:flex;flex-direction:column;min-height:1.375rem;overflow:auto}.top-container{align-items:flex-start;display:flex;min-height:var(--cr-dialog-top-container-min-height,31px)}.title-container{display:flex;flex:1;font-size:inherit;font-weight:inherit;margin:0;outline:none}#close{align-self:flex-start;margin-inline-end:4px;margin-top:4px}@container style(--cr-dialog-body-border-top){.cr-scrollable-top{display:block;border-top:var(--cr-dialog-body-border-top)}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$c() {
    // clang-format off
    return html `
<dialog id="dialog" @close="${this.onNativeDialogClose_}"
    @cancel="${this.onNativeDialogCancel_}" part="dialog"
    aria-labelledby="title"
    aria-description="${this.ariaDescriptionText || nothing}">
<!-- This wrapper is necessary, such that the "pulse" animation is not
    erroneously played when the user clicks on the outer-most scrollbar. -->
  <div id="content-wrapper" part="wrapper">
    <div class="top-container">
      <h2 id="title" class="title-container" tabindex="-1">
        <slot name="title"></slot>
      </h2>
      ${this.showCloseButton ? html `
        <cr-icon-button id="close" class="icon-clear"
            aria-label="${this.closeText || nothing}"
            title="${this.closeText || nothing}"
            @click="${this.cancel}" @keypress="${this.onCloseKeypress_}">
        </cr-icon-button>
       ` : ''}
    </div>
    <slot name="header"></slot>
    <div class="body-container cr-scrollable" id="container"
        part="body-container">
      <div class="cr-scrollable-top"></div>
      <slot name="body"></slot>
      <div class="cr-scrollable-bottom"></div>
    </div>
    <slot name="button-container"></slot>
    <slot name="footer"></slot>
  </div>
</dialog>`;
    // clang-format on
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-dialog' is a component for showing a modal dialog. If the
 * dialog is closed via close(), a 'close' event is fired. If the dialog is
 * canceled via cancel(), a 'cancel' event is fired followed by a 'close' event.
 *
 * Additionally clients can get a reference to the internal native <dialog> via
 * calling getNative() and inspecting the |returnValue| property inside
 * the 'close' event listener to determine whether it was canceled or just
 * closed, where a truthy value means success, and a falsy value means it was
 * canceled.
 *
 * Note that <cr-dialog> wrapper itself always has 0x0 dimensions, and
 * specifying width/height on <cr-dialog> directly will have no effect on the
 * internal native <dialog>. Instead use cr-dialog::part(dialog) to specify
 * width/height (as well as other available mixins to style other parts of the
 * dialog contents).
 */
class CrDialogElement extends CrLitElement {
    static get is() {
        return 'cr-dialog';
    }
    static get styles() {
        return getCss$i();
    }
    render() {
        return getHtml$c.bind(this)();
    }
    static get properties() {
        return {
            open: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Alt-text for the dialog close button.
             */
            closeText: { type: String },
            /**
             * True if the dialog should remain open on 'popstate' events. This is
             * used for navigable dialogs that have their separate navigation handling
             * code.
             */
            ignorePopstate: { type: Boolean },
            /**
             * True if the dialog should ignore 'Enter' keypresses.
             */
            ignoreEnterKey: { type: Boolean },
            /**
             * True if the dialog should consume 'keydown' events. If ignoreEnterKey
             * is true, 'Enter' key won't be consumed.
             */
            consumeKeydownEvent: { type: Boolean },
            /**
             * True if the dialog should not be able to be cancelled, which will
             * prevent 'Escape' key presses from closing the dialog.
             */
            noCancel: { type: Boolean },
            // True if dialog should show the 'X' close button.
            showCloseButton: { type: Boolean },
            showOnAttach: { type: Boolean },
            /**
             * Text for the aria description.
             */
            ariaDescriptionText: { type: String },
        };
    }
    #closeText_accessor_storage;
    get closeText() { return this.#closeText_accessor_storage; }
    set closeText(value) { this.#closeText_accessor_storage = value; }
    #consumeKeydownEvent_accessor_storage = false;
    get consumeKeydownEvent() { return this.#consumeKeydownEvent_accessor_storage; }
    set consumeKeydownEvent(value) { this.#consumeKeydownEvent_accessor_storage = value; }
    #ignoreEnterKey_accessor_storage = false;
    get ignoreEnterKey() { return this.#ignoreEnterKey_accessor_storage; }
    set ignoreEnterKey(value) { this.#ignoreEnterKey_accessor_storage = value; }
    #ignorePopstate_accessor_storage = false;
    get ignorePopstate() { return this.#ignorePopstate_accessor_storage; }
    set ignorePopstate(value) { this.#ignorePopstate_accessor_storage = value; }
    #noCancel_accessor_storage = false;
    get noCancel() { return this.#noCancel_accessor_storage; }
    set noCancel(value) { this.#noCancel_accessor_storage = value; }
    #open_accessor_storage = false;
    get open() { return this.#open_accessor_storage; }
    set open(value) { this.#open_accessor_storage = value; }
    #showCloseButton_accessor_storage = false;
    get showCloseButton() { return this.#showCloseButton_accessor_storage; }
    set showCloseButton(value) { this.#showCloseButton_accessor_storage = value; }
    #showOnAttach_accessor_storage = false;
    get showOnAttach() { return this.#showOnAttach_accessor_storage; }
    set showOnAttach(value) { this.#showOnAttach_accessor_storage = value; }
    #ariaDescriptionText_accessor_storage;
    get ariaDescriptionText() { return this.#ariaDescriptionText_accessor_storage; }
    set ariaDescriptionText(value) { this.#ariaDescriptionText_accessor_storage = value; }
    mutationObserver_ = null;
    boundKeydown_ = null;
    firstUpdated() {
        // If the active history entry changes (i.e. user clicks back button),
        // all open dialogs should be cancelled.
        window.addEventListener('popstate', () => {
            if (!this.ignorePopstate && this.$.dialog.open) {
                this.cancel();
            }
        });
        if (!this.ignoreEnterKey) {
            this.addEventListener('keypress', this.onKeypress_.bind(this));
        }
        this.addEventListener('pointerdown', e => this.onPointerdown_(e));
    }
    connectedCallback() {
        super.connectedCallback();
        const mutationObserverCallback = () => {
            if (this.$.dialog.open) {
                this.addKeydownListener_();
            }
            else {
                this.removeKeydownListener_();
            }
        };
        this.mutationObserver_ = new MutationObserver(mutationObserverCallback);
        this.mutationObserver_.observe(this.$.dialog, {
            attributes: true,
            attributeFilter: ['open'],
        });
        // In some cases dialog already has the 'open' attribute by this point.
        mutationObserverCallback();
        if (this.showOnAttach) {
            this.showModal();
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.removeKeydownListener_();
        if (this.mutationObserver_) {
            this.mutationObserver_.disconnect();
            this.mutationObserver_ = null;
        }
    }
    addKeydownListener_() {
        if (!this.consumeKeydownEvent) {
            return;
        }
        this.boundKeydown_ = this.boundKeydown_ || this.onKeydown_.bind(this);
        this.addEventListener('keydown', this.boundKeydown_);
        // Sometimes <body> is key event's target and in that case the event
        // will bypass cr-dialog. We should consume those events too in order to
        // behave modally. This prevents accidentally triggering keyboard commands.
        document.body.addEventListener('keydown', this.boundKeydown_);
    }
    removeKeydownListener_() {
        if (!this.boundKeydown_) {
            return;
        }
        this.removeEventListener('keydown', this.boundKeydown_);
        document.body.removeEventListener('keydown', this.boundKeydown_);
        this.boundKeydown_ = null;
    }
    async showModal() {
        if (this.showOnAttach) {
            const element = this.querySelector('[autofocus]');
            if (element && element instanceof CrLitElement && !element.shadowRoot) {
                // Force initial render, so that any inner elements with [autofocus] are
                // picked up by the browser.
                element.ensureInitialRender();
            }
        }
        this.$.dialog.showModal();
        assert(this.$.dialog.open);
        this.open = true;
        await this.updateComplete;
        this.fire('cr-dialog-open');
    }
    cancel() {
        this.fire('cancel');
        this.$.dialog.close();
        assert(!this.$.dialog.open);
        this.open = false;
    }
    close() {
        this.$.dialog.close('success');
        assert(!this.$.dialog.open);
        this.open = false;
    }
    /**
     * Set the title of the dialog for a11y reader.
     * @param title Title of the dialog.
     */
    setTitleAriaLabel(title) {
        this.$.dialog.removeAttribute('aria-labelledby');
        this.$.dialog.setAttribute('aria-label', title);
    }
    onCloseKeypress_(e) {
        // Because the dialog may have a default Enter key handler, prevent
        // keypress events from bubbling up from this element.
        e.stopPropagation();
    }
    onNativeDialogClose_(e) {
        // Ignore any 'close' events not fired directly by the <dialog> element.
        if (e.target !== this.getNative()) {
            return;
        }
        // Catch and re-fire the 'close' event such that it bubbles across Shadow
        // DOM v1.
        this.fire('close');
    }
    async onNativeDialogCancel_(e) {
        // Ignore any 'cancel' events not fired directly by the <dialog> element.
        if (e.target !== this.getNative()) {
            return;
        }
        if (this.noCancel) {
            e.preventDefault();
            return;
        }
        // When the dialog is dismissed using the 'Esc' key, need to manually update
        // the |open| property (since close() is not called).
        this.open = false;
        await this.updateComplete;
        // Catch and re-fire the native 'cancel' event such that it bubbles across
        // Shadow DOM v1.
        this.fire('cancel');
    }
    /**
     * Expose the inner native <dialog> for some rare cases where it needs to be
     * directly accessed (for example to programmatically setheight/width, which
     * would not work on the wrapper).
     */
    getNative() {
        return this.$.dialog;
    }
    onKeypress_(e) {
        if (e.key !== 'Enter') {
            return;
        }
        // Accept Enter keys from either the dialog itself, or a child cr-input,
        // considering that the event may have been retargeted, for example if the
        // cr-input is nested inside another element. Also exclude inputs of type
        // 'search', since hitting 'Enter' on a search field most likely intends to
        // trigger searching.
        const accept = e.target === this ||
            e.composedPath().some(el => el.tagName === 'CR-INPUT' &&
                el.type !== 'search');
        if (!accept) {
            return;
        }
        const actionButton = this.querySelector('.action-button:not([disabled]):not([hidden])');
        if (actionButton) {
            actionButton.click();
            e.preventDefault();
        }
    }
    onKeydown_(e) {
        assert(this.consumeKeydownEvent);
        if (!this.getNative().open) {
            return;
        }
        if (this.ignoreEnterKey && e.key === 'Enter') {
            return;
        }
        // Stop propagation to behave modally.
        e.stopPropagation();
    }
    onPointerdown_(e) {
        // Only show pulse animation if user left-clicked outside of the dialog
        // contents.
        if (e.button !== 0 ||
            e.composedPath()[0].tagName !== 'DIALOG') {
            return;
        }
        this.$.dialog.animate([
            { transform: 'scale(1)', offset: 0 },
            { transform: 'scale(1.02)', offset: 0.4 },
            { transform: 'scale(1.02)', offset: 0.6 },
            { transform: 'scale(1)', offset: 1 },
        ], {
            duration: 180,
            easing: 'ease-in-out',
            iterations: 1,
        });
        // Prevent any text from being selected within the dialog when clicking in
        // the backdrop area.
        e.preventDefault();
    }
    focus() {
        const titleContainer = this.shadowRoot.querySelector('.title-container');
        assert(titleContainer);
        titleContainer.focus();
    }
}
customElements.define(CrDialogElement.is, CrDialogElement);

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Base class for Web Components that don't use Polymer.
 * See the following file for usage:
 * chrome/test/data/webui/js/custom_element_test.js
 */
function emptyHTML() {
    return window.trustedTypes ? window.trustedTypes.emptyHTML : '';
}
class CustomElement extends HTMLElement {
    static get template() {
        return emptyHTML();
    }
    constructor() {
        super();
        this.attachShadow({ mode: 'open' });
        const template = document.createElement('template');
        template.innerHTML =
            this.constructor.template || emptyHTML();
        this.shadowRoot.appendChild(template.content.cloneNode(true));
    }
    $(query) {
        return this.shadowRoot.querySelector(query);
    }
    $all(query) {
        return this.shadowRoot.querySelectorAll(query);
    }
    getRequiredElement(query) {
        const el = this.shadowRoot.querySelector(query);
        assert(el);
        assert(el instanceof HTMLElement);
        return el;
    }
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @return Whether the passed tagged template literal is a valid array.
 */
function isValidArray(arr) {
    if (arr instanceof Array && Object.isFrozen(arr)) {
        return true;
    }
    return false;
}
/**
 * Checks if the passed tagged template literal only contains static string.
 * And return the string in the literal if so.
 * Throws an Error if the passed argument is not supported literals.
 */
function getStaticString(literal) {
    const isStaticString = isValidArray(literal) && !!literal.raw &&
        isValidArray(literal.raw) && literal.length === literal.raw.length &&
        literal.length === 1;
    assert(isStaticString, 'static_types.js only allows static strings');
    return literal.join('');
}
function createTypes(_ignore, literal) {
    return getStaticString(literal);
}
/**
 * Rules used to enforce static literal checks.
 */
const rules = {
    createHTML: createTypes,
    createScript: createTypes,
    createScriptURL: createTypes,
};
/**
 * This policy returns Trusted Types if the passed literal is static.
 */
let staticPolicy;
if (window.trustedTypes) {
    staticPolicy = window.trustedTypes.createPolicy('static-types', rules);
}
else {
    staticPolicy = rules;
}
/**
 * Returns TrustedHTML if the passed literal is static.
 */
function getTrustedHTML(literal) {
    return staticPolicy.createHTML('', literal);
}

function getTemplate() {
    return getTrustedHTML `<!--_html_template_start_--><style>:host{clip:rect(0 0 0 0);height:1px;overflow:hidden;position:fixed;width:1px}</style>

<div id="messages" role="alert" aria-live="polite" aria-relevant="additions">
</div>
<!--_html_template_end_-->`;
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * 150ms seems to be around the minimum time required for screen readers to
 * read out consecutively queued messages.
 */
const TIMEOUT_MS = 150;
/**
 * A map of an HTML element to its corresponding CrA11yAnnouncerElement. There
 * may be multiple CrA11yAnnouncerElements on a page, especially for cases in
 * which the DocumentElement's CrA11yAnnouncerElement becomes hidden or
 * deactivated (eg. when a modal dialog causes the CrA11yAnnouncerElement to
 * become inaccessible).
 */
const instances = new Map();
function getInstance(container = document.body) {
    if (instances.has(container)) {
        return instances.get(container);
    }
    assert(container.isConnected);
    const instance = new CrA11yAnnouncerElement();
    container.appendChild(instance);
    instances.set(container, instance);
    return instance;
}
class CrA11yAnnouncerElement extends CustomElement {
    static get is() {
        return 'cr-a11y-announcer';
    }
    static get template() {
        return getTemplate();
    }
    currentTimeout_ = null;
    messages_ = [];
    disconnectedCallback() {
        if (this.currentTimeout_ !== null) {
            clearTimeout(this.currentTimeout_);
            this.currentTimeout_ = null;
        }
        for (const [parent, instance] of instances) {
            if (instance === this) {
                instances.delete(parent);
                break;
            }
        }
    }
    announce(message, timeout = TIMEOUT_MS) {
        if (this.currentTimeout_ !== null) {
            clearTimeout(this.currentTimeout_);
            this.currentTimeout_ = null;
        }
        this.messages_.push(message);
        this.currentTimeout_ = setTimeout(() => {
            const messagesDiv = this.shadowRoot.querySelector('#messages');
            messagesDiv.innerHTML = window.trustedTypes.emptyHTML;
            // 
            // VoiceOver on Mac does not seem to consistently read out the contents of
            // a static alert element. Toggling the role of alert seems to force VO
            // to consistently read out the messages.
            messagesDiv.removeAttribute('role');
            messagesDiv.setAttribute('role', 'alert');
            // 
            for (const message of this.messages_) {
                const div = document.createElement('div');
                div.textContent = message;
                messagesDiv.appendChild(div);
            }
            // Dispatch a custom event to allow consumers to know when certain alerts
            // have been sent to the screen reader.
            this.dispatchEvent(new CustomEvent('cr-a11y-announcer-messages-sent', { bubbles: true, detail: { messages: this.messages_.slice() } }));
            this.messages_.length = 0;
            this.currentTimeout_ = null;
        }, timeout);
    }
}
customElements.define(CrA11yAnnouncerElement.is, CrA11yAnnouncerElement);

let instance$k = null;
function getCss$h() {
    return instance$k || (instance$k = [...[], css `#body{display:flex}.tonal-button{margin-inline-end:8px}#icon-wrapper,cr-icon{height:var(--cr-icon-size);width:var(--cr-icon-size);color:var(--google-red-700)}@media (prefers-color-scheme:dark){#icon-wrapper,cr-icon{color:var(--google-red-300)}}#icon-wrapper{margin-inline-end:8px}#body-text{color:var(--cr-primary-text-color)}#file-name{font-weight:bold}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$b() {
    return html `<!--_html_template_start_-->
<cr-dialog show-on-attach id="dialog">
  <div slot="title">$i18n{warningBypassDialogTitle}</div>
  <div slot="body" id="body">
    <div id="icon-wrapper" role="img"
        aria-label="$i18n{accessibleLabelDangerous}">
      <cr-icon icon="downloads:dangerous"></cr-icon>
    </div>
    <div id="body-text">
      <div id="file-name">${this.fileName}</div>
      <div id="danger-description">$i18n{warningBypassPromptDescription}</div>
      <div id="learn-more-link">
        <!-- noopener cuts off the script connection between chrome://downloads,
          which has sensitive capabilities, and the newly opened web renderer,
          which may be more readily compromised. -->
        <a href="$i18n{blockedLearnMoreUrl}" target="_blank" rel="noopener">
          $i18n{warningBypassPromptLearnMoreLink}
        </a>
      </div>
    </div>
  </div>
  <div slot="button-container">
    <cr-button class="tonal-button" @click="${this.onDownloadDangerousClick_}"
        id="download-dangerous-button">
      $i18n{controlKeepDangerous}
    </cr-button>
    <!-- The cancel button is the primary action because we don't want the user
      to download the dangerous file. -->
    <cr-button class="action-button" @click="${this.onCancelClick_}"
        id="cancel-button">
      $i18n{warningBypassDialogCancel}
    </cr-button>
  </div>
</cr-dialog>
<!--_html_template_end_-->`;
}

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'downloads-bypass-warning-confirmation-dialog' is the dialog
 * that allows bypassing a download warning (keeping a file flagged as
 * dangerous). A 'success' indicates the warning bypass was confirmed and the
 * dangerous file was downloaded.
 */
class DownloadsBypassWarningConfirmationDialogElement extends CrLitElement {
    static get is() {
        return 'downloads-bypass-warning-confirmation-dialog';
    }
    static get styles() {
        return getCss$h();
    }
    render() {
        return getHtml$b.bind(this)();
    }
    static get properties() {
        return {
            fileName: { type: String },
        };
    }
    #fileName_accessor_storage = '';
    get fileName() { return this.#fileName_accessor_storage; }
    set fileName(value) { this.#fileName_accessor_storage = value; }
    wasConfirmed() {
        return this.$.dialog.getNative().returnValue === 'success';
    }
    onDownloadDangerousClick_() {
        getInstance().announce(loadTimeData.getString('screenreaderSavedDangerous'));
        this.$.dialog.close();
    }
    onCancelClick_() {
        this.$.dialog.cancel();
    }
}
customElements.define(DownloadsBypassWarningConfirmationDialogElement.is, DownloadsBypassWarningConfirmationDialogElement);

let instance$j = null;
function getCss$g() {
    return instance$j || (instance$j = [...[], css `:host{display:none}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$a() {
    return html `
<svg id="baseSvg" xmlns="http://www.w3.org/2000/svg"
     viewBox="0 0 ${this.size} ${this.size}"
     preserveAspectRatio="xMidYMid meet" focusable="false"
     style="pointer-events: none; display: block; width: 100%; height: 100%;">
 </svg>
<slot></slot>
`;
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const APPLIED_ICON_CLASS = 'cr-iconset-svg-icon_';
class CrIconsetElement extends CrLitElement {
    static get is() {
        return 'cr-iconset';
    }
    static get styles() {
        return getCss$g();
    }
    render() {
        return getHtml$a.bind(this)();
    }
    static get properties() {
        return {
            /**
             * The name of the iconset.
             */
            name: { type: String },
            /**
             * The size of an individual icon. Note that icons must be square.
             */
            size: { type: Number },
        };
    }
    #name_accessor_storage = '';
    get name() { return this.#name_accessor_storage; }
    set name(value) { this.#name_accessor_storage = value; }
    #size_accessor_storage = 24;
    get size() { return this.#size_accessor_storage; }
    set size(value) { this.#size_accessor_storage = value; }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('name')) {
            assert(changedProperties.get('name') === undefined);
            IconsetMap.getInstance().set(this.name, this);
        }
    }
    /**
     * Applies an icon to the given element.
     *
     * An svg icon is prepended to the element's shadowRoot, which should always
     * exist.
     * @param element Element to which the icon is applied.
     * @param iconName Name of the icon to apply.
     * @return The svg element which renders the icon.
     */
    applyIcon(element, iconName) {
        // Remove old svg element
        this.removeIcon(element);
        // install new svg element
        const svg = this.cloneIcon_(iconName);
        if (svg) {
            // Add special class so we can identify it in remove.
            svg.classList.add(APPLIED_ICON_CLASS);
            // insert svg element into shadow root
            element.shadowRoot.insertBefore(svg, element.shadowRoot.childNodes[0]);
            return svg;
        }
        return null;
    }
    /**
     * Produce installable clone of the SVG element matching `id` in this
     * iconset, or null if there is no matching element.
     * @param iconName Name of the icon to apply.
     */
    createIcon(iconName) {
        return this.cloneIcon_(iconName);
    }
    /**
     * Remove an icon from the given element by undoing the changes effected
     * by `applyIcon`.
     */
    removeIcon(element) {
        // Remove old svg element
        const oldSvg = element.shadowRoot.querySelector(`.${APPLIED_ICON_CLASS}`);
        if (oldSvg) {
            oldSvg.remove();
        }
    }
    /**
     * Produce installable clone of the SVG element matching `id` in this
     * iconset, or `undefined` if there is no matching element.
     *
     * Returns an installable clone of the SVG element matching `id` or null if
     * no such element exists.
     */
    cloneIcon_(id) {
        const sourceSvg = this.querySelector(`g[id="${id}"]`);
        if (!sourceSvg) {
            return null;
        }
        const svgClone = this.$.baseSvg.cloneNode(true);
        const content = sourceSvg.cloneNode(true);
        content.removeAttribute('id');
        const contentViewBox = content.getAttribute('viewBox');
        if (contentViewBox) {
            svgClone.setAttribute('viewBox', contentViewBox);
        }
        svgClone.appendChild(content);
        return svgClone;
    }
}
customElements.define(CrIconsetElement.is, CrIconsetElement);

const div$1 = document.createElement('div');
div$1.innerHTML = getTrustedHTML `<cr-iconset name="downloads" size="24">
  <svg>
    <defs>
      <!--
      These icons are copied from Polymer's iron-icons and kept in sorted order.
      See http://goo.gl/Y1OdAq for instructions on adding additional icons.
      -->
      <g id="dangerous"><!-- Source: https://fonts.google.com/icons -->
        <path
          d="M 8.25 21 L 3 15.75 L 3 8.25 L 8.25 3 L 15.75 3 L 21 8.25 L 21 15.75 L 15.75 21 Z M 9.148438 16.25 L 12 13.398438 L 14.851562 16.25 L 16.25 14.851562 L 13.398438 12 L 16.25 9.148438 L 14.851562 7.75 L 12 10.601562 L 9.148438 7.75 L 7.75 9.148438 L 10.601562 12 L 7.75 14.851562 Z M 9.148438 16.25">
        </path>
      </g>
      <g id="shield">
        <path fill="#0b57d0" fill-opacity="1" fill-rule="nonzero" d="M12 22.210938c-2.375-.597657-4.335938-1.949219-5.882812-4.066407-1.546876-2.113281-2.320313-4.460937-2.320313-7.042969V4.859375L12 1.789062l8.203125 3.070313v6.242187c0 2.582032-.773437 4.929688-2.320313 7.042969C16.335938 20.261719 14.375 21.613281 12 22.210938Zm0-2.375c1.589844-.527344 2.917969-1.511719 3.992188-2.964844C17.066406 15.421875 17.695312 13.796875 17.878906 12H12V4.207031L6.070312 6.433594v5.117187c0 .125.019532.273438.050782.449219H12Zm0 0">
        </path>
      </g>
      <g id="link">
        <path d="M3.9 12c0-1.71 1.39-3.1 3.1-3.1h4V7H7c-2.76 0-5 2.24-5 5s2.24 5 5 5h4v-1.9H7c-1.71 0-3.1-1.39-3.1-3.1zM8 13h8v-2H8v2zm9-6h-4v1.9h4c1.71 0 3.1 1.39 3.1 3.1s-1.39 3.1-3.1 3.1h-4V17h4c2.76 0 5-2.24 5-5s-2.24-5-5-5z">
        </path>
      </g>
    </defs>
  </svg>
</cr-iconset>
`;
const iconsets$1 = div$1.querySelectorAll('cr-iconset');
for (const iconset of iconsets$1) {
    document.head.appendChild(iconset);
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @return The currently focused element (including elements that are
 *     behind a shadow root), or null if nothing is focused.
 */
function getDeepActiveElement() {
    let a = document.activeElement;
    while (a && a.shadowRoot && a.shadowRoot.activeElement) {
        a = a.shadowRoot.activeElement;
    }
    return a;
}
/**
 * Check the directionality of the page.
 * @return True if Chrome is running an RTL UI.
 */
function isRTL() {
    return document.documentElement.dir === 'rtl';
}
/**
 * Replaces '&', '<', '>', '"', and ''' characters with their HTML encoding.
 * @param original The original string.
 * @return The string with all the characters mentioned above replaced.
 */
function htmlEscape(original) {
    return original.replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;')
        .replace(/"/g, '&quot;')
        .replace(/'/g, '&#39;');
}
/**
 * @return Whether a modifier key was down when processing |e|.
 */
function hasKeyModifiers(e) {
    return !!(e.altKey || e.ctrlKey || e.metaKey || e.shiftKey);
}

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// clang-format off
// clang-format on
const ACTIVE_CLASS = 'focus-row-active';
/**
 * A class to manage focus between given horizontally arranged elements.
 *
 * Pressing left cycles backward and pressing right cycles forward in item
 * order. Pressing Home goes to the beginning of the list and End goes to the
 * end of the list.
 *
 * If an item in this row is focused, it'll stay active (accessible via tab).
 * If no items in this row are focused, the row can stay active until focus
 * changes to a node inside |this.boundary_|. If |boundary| isn't specified,
 * any focus change deactivates the row.
 */
class FocusRow {
    root;
    delegate;
    eventTracker = new EventTracker();
    boundary_;
    /**
     * @param root The root of this focus row. Focus classes are
     *     applied to |root| and all added elements must live within |root|.
     * @param boundary Focus events are ignored outside of this element.
     * @param delegate An optional event delegate.
     */
    constructor(root, boundary, delegate) {
        this.root = root;
        this.boundary_ = boundary || document.documentElement;
        this.delegate = delegate;
    }
    /**
     * Whether it's possible that |element| can be focused.
     */
    static isFocusable(element) {
        if (!element || element.disabled) {
            return false;
        }
        // We don't check that element.tabIndex >= 0 here because inactive rows
        // set a tabIndex of -1.
        let current = element;
        while (true) {
            assertInstanceof(current, Element);
            const style = window.getComputedStyle(current);
            if (style.visibility === 'hidden' || style.display === 'none') {
                return false;
            }
            const parent = current.parentNode;
            if (!parent) {
                return false;
            }
            if (parent === current.ownerDocument ||
                parent instanceof DocumentFragment) {
                return true;
            }
            current = parent;
        }
    }
    /**
     * A focus override is a function that returns an element that should gain
     * focus. The element may not be directly selectable for example the element
     * that can gain focus is in a shadow DOM. Allowing an override via a
     * function leaves the details of how the element is retrieved to the
     * component.
     */
    static getFocusableElement(element) {
        const withFocusable = element;
        if (withFocusable.getFocusableElement) {
            return withFocusable.getFocusableElement();
        }
        return element;
    }
    /**
     * Register a new type of focusable element (or add to an existing one).
     *
     * Example: an (X) button might be 'delete' or 'close'.
     *
     * When FocusRow is used within a FocusGrid, these types are used to
     * determine equivalent controls when Up/Down are pressed to change rows.
     *
     * Another example: mutually exclusive controls that hide each other on
     * activation (i.e. Play/Pause) could use the same type (i.e. 'play-pause')
     * to indicate they're equivalent.
     *
     * @param type The type of element to track focus of.
     * @param selectorOrElement The selector of the element
     *    from this row's root, or the element itself.
     * @return Whether a new item was added.
     */
    addItem(type, selectorOrElement) {
        assert(type);
        let element;
        if (typeof selectorOrElement === 'string') {
            element = this.root.querySelector(selectorOrElement);
        }
        else {
            element = selectorOrElement;
        }
        if (!element) {
            return false;
        }
        element.setAttribute('focus-type', type);
        element.tabIndex = this.isActive() ? 0 : -1;
        this.eventTracker.add(element, 'blur', this.onBlur_.bind(this));
        this.eventTracker.add(element, 'focus', this.onFocus_.bind(this));
        this.eventTracker.add(element, 'keydown', this.onKeydown_.bind(this));
        this.eventTracker.add(element, 'mousedown', this.onMousedown_.bind(this));
        return true;
    }
    /** Dereferences nodes and removes event handlers. */
    destroy() {
        this.eventTracker.removeAll();
    }
    /**
     * @param sampleElement An element for to find an equivalent
     *     for.
     * @return An equivalent element to focus for
     *     |sampleElement|.
     */
    getCustomEquivalent(_sampleElement) {
        const focusable = this.getFirstFocusable();
        assert(focusable);
        return focusable;
    }
    /**
     * @return All registered elements (regardless of focusability).
     */
    getElements() {
        return Array.from(this.root.querySelectorAll('[focus-type]'))
            .map(FocusRow.getFocusableElement);
    }
    /**
     * Find the element that best matches |sampleElement|.
     * @param sampleElement An element from a row of the same
     *     type which previously held focus.
     * @return The element that best matches sampleElement.
     */
    getEquivalentElement(sampleElement) {
        if (this.getFocusableElements().indexOf(sampleElement) >= 0) {
            return sampleElement;
        }
        const sampleFocusType = this.getTypeForElement(sampleElement);
        if (sampleFocusType) {
            const sameType = this.getFirstFocusable(sampleFocusType);
            if (sameType) {
                return sameType;
            }
        }
        return this.getCustomEquivalent(sampleElement);
    }
    /**
     * @param type An optional type to search for.
     * @return The first focusable element with |type|.
     */
    getFirstFocusable(type) {
        const element = this.getFocusableElements().find(el => !type || el.getAttribute('focus-type') === type);
        return element || null;
    }
    /** @return Registered, focusable elements. */
    getFocusableElements() {
        return this.getElements().filter(FocusRow.isFocusable);
    }
    /**
     * @param element An element to determine a focus type for.
     * @return The focus type for |element| or '' if none.
     */
    getTypeForElement(element) {
        return element.getAttribute('focus-type') || '';
    }
    /** @return Whether this row is currently active. */
    isActive() {
        return this.root.classList.contains(ACTIVE_CLASS);
    }
    /**
     * Enables/disables the tabIndex of the focusable elements in the FocusRow.
     * tabIndex can be set properly.
     * @param active True if tab is allowed for this row.
     */
    makeActive(active) {
        if (active === this.isActive()) {
            return;
        }
        this.getElements().forEach(function (element) {
            element.tabIndex = active ? 0 : -1;
        });
        this.root.classList.toggle(ACTIVE_CLASS, active);
    }
    onBlur_(e) {
        if (!this.boundary_.contains(e.relatedTarget)) {
            return;
        }
        const currentTarget = e.currentTarget;
        if (this.getFocusableElements().indexOf(currentTarget) >= 0) {
            this.makeActive(false);
        }
    }
    onFocus_(e) {
        if (this.delegate) {
            this.delegate.onFocus(this, e);
        }
    }
    onMousedown_(e) {
        // Only accept left mouse clicks.
        if (e.button) {
            return;
        }
        // Allow the element under the mouse cursor to be focusable.
        const target = e.currentTarget;
        if (!target.disabled) {
            target.tabIndex = 0;
        }
    }
    onKeydown_(e) {
        const elements = this.getFocusableElements();
        const currentElement = FocusRow.getFocusableElement(e.currentTarget);
        const elementIndex = elements.indexOf(currentElement);
        assert(elementIndex >= 0);
        if (this.delegate && this.delegate.onKeydown(this, e)) {
            return;
        }
        const isShiftTab = !e.altKey && !e.ctrlKey && !e.metaKey && e.shiftKey &&
            e.key === 'Tab';
        if (hasKeyModifiers(e) && !isShiftTab) {
            return;
        }
        let index = -1;
        let shouldStopPropagation = true;
        if (isShiftTab) {
            // This always moves back one element, even in RTL.
            index = elementIndex - 1;
            if (index < 0) {
                // Bubble up to focus on the previous element outside the row.
                return;
            }
        }
        else if (e.key === 'ArrowLeft') {
            index = elementIndex + (isRTL() ? 1 : -1);
        }
        else if (e.key === 'ArrowRight') {
            index = elementIndex + (isRTL() ? -1 : 1);
        }
        else if (e.key === 'Home') {
            index = 0;
        }
        else if (e.key === 'End') {
            index = elements.length - 1;
        }
        else {
            shouldStopPropagation = false;
        }
        const elementToFocus = elements[index];
        if (elementToFocus) {
            this.getEquivalentElement(elementToFocus).focus();
            e.preventDefault();
        }
        if (shouldStopPropagation) {
            e.stopPropagation();
        }
    }
}
class VirtualFocusRow extends FocusRow {
    constructor(root, delegate) {
        super(root, /* boundary */ null, delegate);
    }
    getCustomEquivalent(sampleElement) {
        const equivalent = this.delegate ? this.delegate.getCustomEquivalent(sampleElement) : null;
        return equivalent || super.getCustomEquivalent(sampleElement);
    }
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
let hideInk = false;
document.addEventListener('pointerdown', function () {
    hideInk = true;
}, true);
document.addEventListener('keydown', function () {
    hideInk = false;
}, true);
/**
 * Attempts to track whether focus outlines should be shown, and if they
 * shouldn't, removes the "ink" (ripple) from a control while focusing it.
 * This is helpful when a user is clicking/touching, because it's not super
 * helpful to show focus ripples in that case. This is Polymer-specific.
 */
function focusWithoutInk(toFocus) {
    // |toFocus| does not have a 'noink' property, so it's unclear whether the
    // element has "ink" and/or whether it can be suppressed. Just focus().
    if (!('noink' in toFocus) || !hideInk) {
        toFocus.focus();
        return;
    }
    const toFocusWithNoInk = toFocus;
    // Make sure the element is in the document we're listening to events on.
    assert(document === toFocusWithNoInk.ownerDocument);
    const { noink } = toFocusWithNoInk;
    toFocusWithNoInk.noink = true;
    toFocusWithNoInk.focus();
    toFocusWithNoInk.noink = noink;
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/* @fileoverview Utilities for determining the current platform. */
/** Whether we are using a Mac or not. */
const isMac = /Mac/.test(navigator.platform);
/** Whether this is on the Windows platform or not. */
const isWindows = /Win/.test(navigator.platform);

let instance$i = null;
function getCss$f() {
    return instance$i || (instance$i = [...[], css `:host{--cr-hairline:1px solid var(--color-menu-separator,var(--cr-fallback-color-divider));--cr-action-menu-disabled-item-color:var(--color-menu-item-foreground-disabled,var(--cr-fallback-color-disabled-foreground));--cr-action-menu-disabled-item-opacity:1;--cr-menu-background-color:var(--color-menu-background,var(--cr-fallback-color-surface));--cr-menu-background-focus-color:var(--cr-hover-background-color);--cr-menu-shadow:var(--cr-elevation-2);--cr-primary-text-color:var(--color-menu-item-foreground,var(--cr-fallback-color-on-surface))}:host dialog{background-color:var(--cr-menu-background-color);border:none;border-radius:var(--cr-menu-border-radius,4px);box-shadow:var(--cr-menu-shadow);margin:0;min-width:128px;outline:none;overflow:var(--cr-action-menu-overflow,auto);padding:0;position:absolute}@media (forced-colors:active){:host dialog{border:var(--cr-border-hcm)}}:host dialog::backdrop{background-color:transparent}:host ::slotted(.dropdown-item){-webkit-tap-highlight-color:transparent;background:none;border:none;border-radius:0;box-sizing:border-box;color:var(--cr-primary-text-color);font:inherit;min-height:32px;padding:8px 24px;text-align:start;user-select:none;width:100%}:host ::slotted(.dropdown-item:not([hidden])){align-items:center;display:flex}:host ::slotted(.dropdown-item[disabled]){color:var(--cr-action-menu-disabled-item-color,var(--cr-primary-text-color));opacity:var(--cr-action-menu-disabled-item-opacity,0.65)}:host ::slotted(.dropdown-item:not([disabled])){cursor:pointer}:host ::slotted(.dropdown-item:focus){background-color:var(--cr-menu-background-focus-color);outline:none}:host ::slotted(.dropdown-item:focus-visible){outline:solid 2px var(--cr-focus-outline-color);outline-offset:-2px}@media (forced-colors:active){:host ::slotted(.dropdown-item:focus){outline:var(--cr-focus-outline-hcm)}}.item-wrapper{outline:none;padding:var(--cr-action-menu-padding,8px 0)}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$9() {
    return html `
<dialog id="dialog" part="dialog" @close="${this.onNativeDialogClose_}"
    role="application"
    aria-roledescription="${this.roleDescription || nothing}">
  <div id="wrapper" class="item-wrapper" role="menu" tabindex="-1"
      aria-label="${this.accessibilityLabel || nothing}">
    <slot id="contentNode" @slotchange="${this.onSlotchange_}"></slot>
  </div>
</dialog>`;
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var AnchorAlignment;
(function (AnchorAlignment) {
    AnchorAlignment[AnchorAlignment["BEFORE_START"] = -2] = "BEFORE_START";
    AnchorAlignment[AnchorAlignment["AFTER_START"] = -1] = "AFTER_START";
    AnchorAlignment[AnchorAlignment["CENTER"] = 0] = "CENTER";
    AnchorAlignment[AnchorAlignment["BEFORE_END"] = 1] = "BEFORE_END";
    AnchorAlignment[AnchorAlignment["AFTER_END"] = 2] = "AFTER_END";
})(AnchorAlignment || (AnchorAlignment = {}));
const DROPDOWN_ITEM_CLASS = 'dropdown-item';
const SELECTABLE_DROPDOWN_ITEM_QUERY = `.${DROPDOWN_ITEM_CLASS}:not([hidden]):not([disabled])`;
const AFTER_END_OFFSET = 10;
/**
 * Returns the point to start along the X or Y axis given a start and end
 * point to anchor to, the length of the target and the direction to anchor
 * in. If honoring the anchor would force the menu outside of min/max, this
 * will ignore the anchor position and try to keep the menu within min/max.
 */
function getStartPointWithAnchor(start, end, menuLength, anchorAlignment, min, max) {
    let startPoint = 0;
    switch (anchorAlignment) {
        case AnchorAlignment.BEFORE_START:
            startPoint = start - menuLength;
            break;
        case AnchorAlignment.AFTER_START:
            startPoint = start;
            break;
        case AnchorAlignment.CENTER:
            startPoint = (start + end - menuLength) / 2;
            break;
        case AnchorAlignment.BEFORE_END:
            startPoint = end - menuLength;
            break;
        case AnchorAlignment.AFTER_END:
            startPoint = end;
            break;
    }
    if (startPoint + menuLength > max) {
        startPoint = end - menuLength;
    }
    if (startPoint < min) {
        startPoint = start;
    }
    startPoint = Math.max(min, Math.min(startPoint, max - menuLength));
    return startPoint;
}
function getDefaultShowConfig() {
    return {
        top: 0,
        left: 0,
        height: 0,
        width: 0,
        anchorAlignmentX: AnchorAlignment.AFTER_START,
        anchorAlignmentY: AnchorAlignment.AFTER_START,
        minX: 0,
        minY: 0,
        maxX: 0,
        maxY: 0,
    };
}
class CrActionMenuElement extends CrLitElement {
    static get is() {
        return 'cr-action-menu';
    }
    static get styles() {
        return getCss$f();
    }
    render() {
        return getHtml$9.bind(this)();
    }
    static get properties() {
        return {
            // Accessibility text of the menu. Should be something along the lines of
            // "actions", or "more actions".
            accessibilityLabel: { type: String },
            // Setting this flag will make the menu listen for content size changes
            // and reposition to its anchor accordingly.
            autoReposition: { type: Boolean },
            open: {
                type: Boolean,
                notify: true,
            },
            // Descriptor of the menu. Should be something along the lines of "menu"
            roleDescription: { type: String },
        };
    }
    #accessibilityLabel_accessor_storage;
    get accessibilityLabel() { return this.#accessibilityLabel_accessor_storage; }
    set accessibilityLabel(value) { this.#accessibilityLabel_accessor_storage = value; }
    #autoReposition_accessor_storage = false;
    get autoReposition() { return this.#autoReposition_accessor_storage; }
    set autoReposition(value) { this.#autoReposition_accessor_storage = value; }
    #open_accessor_storage = false;
    get open() { return this.#open_accessor_storage; }
    set open(value) { this.#open_accessor_storage = value; }
    #roleDescription_accessor_storage;
    get roleDescription() { return this.#roleDescription_accessor_storage; }
    set roleDescription(value) { this.#roleDescription_accessor_storage = value; }
    boundClose_ = null;
    resizeObserver_ = null;
    hasMousemoveListener_ = false;
    anchorElement_ = null;
    lastConfig_ = null;
    firstUpdated() {
        this.addEventListener('keydown', this.onKeyDown_.bind(this));
        this.addEventListener('mouseover', this.onMouseover_);
        this.addEventListener('click', this.onClick_);
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.removeListeners_();
    }
    /**
     * Exposing internal <dialog> elements for tests.
     */
    getDialog() {
        return this.$.dialog;
    }
    removeListeners_() {
        window.removeEventListener('resize', this.boundClose_);
        window.removeEventListener('popstate', this.boundClose_);
        if (this.resizeObserver_) {
            this.resizeObserver_.disconnect();
            this.resizeObserver_ = null;
        }
    }
    onNativeDialogClose_(e) {
        // Ignore any 'close' events not fired directly by the <dialog> element.
        if (e.target !== this.$.dialog) {
            return;
        }
        // Catch and re-fire the 'close' event such that it bubbles across Shadow
        // DOM v1.
        this.fire('close');
    }
    onClick_(e) {
        if (e.target === this) {
            this.close();
            e.stopPropagation();
        }
    }
    onKeyDown_(e) {
        e.stopPropagation();
        if (e.key === 'Tab' || e.key === 'Escape') {
            this.close();
            if (e.key === 'Tab') {
                this.fire('tabkeyclose', { shiftKey: e.shiftKey });
            }
            e.preventDefault();
            return;
        }
        if (e.key !== 'Enter' && e.key !== 'ArrowUp' && e.key !== 'ArrowDown') {
            return;
        }
        const options = Array.from(this.querySelectorAll(SELECTABLE_DROPDOWN_ITEM_QUERY));
        if (options.length === 0) {
            return;
        }
        const focused = getDeepActiveElement();
        const index = options.findIndex(option => FocusRow.getFocusableElement(option) === focused);
        if (e.key === 'Enter') {
            // If a menu item has focus, don't change focus or close menu on 'Enter'.
            if (index !== -1) {
                return;
            }
            if (isWindows || isMac) {
                this.close();
                e.preventDefault();
                return;
            }
        }
        e.preventDefault();
        this.updateFocus_(options, index, e.key !== 'ArrowUp');
        if (!this.hasMousemoveListener_) {
            this.hasMousemoveListener_ = true;
            this.addEventListener('mousemove', e => {
                this.onMouseover_(e);
                this.hasMousemoveListener_ = false;
            }, { once: true });
        }
    }
    onMouseover_(e) {
        const item = e.composedPath()
            .find(el => el.matches && el.matches(SELECTABLE_DROPDOWN_ITEM_QUERY));
        (item || this.$.wrapper).focus();
    }
    updateFocus_(options, focusedIndex, next) {
        const numOptions = options.length;
        assert(numOptions > 0);
        let index;
        if (focusedIndex === -1) {
            index = next ? 0 : numOptions - 1;
        }
        else {
            const delta = next ? 1 : -1;
            index = (numOptions + focusedIndex + delta) % numOptions;
        }
        options[index].focus();
    }
    close() {
        if (!this.open) {
            return;
        }
        // Removing 'resize' and 'popstate' listeners when dialog is closed.
        this.removeListeners_();
        this.$.dialog.close();
        this.open = false;
        if (this.anchorElement_) {
            assert(this.anchorElement_);
            focusWithoutInk(this.anchorElement_);
            this.anchorElement_ = null;
        }
        if (this.lastConfig_) {
            this.lastConfig_ = null;
        }
    }
    /**
     * Shows the menu anchored to the given element.
     */
    showAt(anchorElement, config) {
        this.anchorElement_ = anchorElement;
        // Scroll the anchor element into view so that the bounding rect will be
        // accurate for where the menu should be shown.
        this.anchorElement_.scrollIntoViewIfNeeded();
        const rect = this.anchorElement_.getBoundingClientRect();
        let height = rect.height;
        if (config && !config.noOffset &&
            config.anchorAlignmentY === AnchorAlignment.AFTER_END) {
            // When an action menu is positioned after the end of an element, the
            // action menu can appear too far away from the anchor element, typically
            // because anchors tend to have padding. So we offset the height a bit
            // so the menu shows up slightly closer to the content of anchor.
            height -= AFTER_END_OFFSET;
        }
        this.showAtPosition(Object.assign({
            top: rect.top,
            left: rect.left,
            height: height,
            width: rect.width,
            // Default to anchoring towards the left.
            anchorAlignmentX: AnchorAlignment.BEFORE_END,
        }, config));
        this.$.wrapper.focus();
    }
    /**
     * Shows the menu anchored to the given box. The anchor alignment is
     * specified as an X and Y alignment which represents a point in the anchor
     * where the menu will align to, which can have the menu either before or
     * after the given point in each axis. Center alignment places the center of
     * the menu in line with the center of the anchor. Coordinates are relative to
     * the top-left of the viewport.
     *
     *            y-start
     *         _____________
     *         |           |
     *         |           |
     *         |   CENTER  |
     * x-start |     x     | x-end
     *         |           |
     *         |anchor box |
     *         |___________|
     *
     *             y-end
     *
     * For example, aligning the menu to the inside of the top-right edge of
     * the anchor, extending towards the bottom-left would use a alignment of
     * (BEFORE_END, AFTER_START), whereas centering the menu below the bottom
     * edge of the anchor would use (CENTER, AFTER_END).
     */
    showAtPosition(config) {
        // Save the scroll position of the viewport.
        const doc = document.scrollingElement;
        const scrollLeft = doc.scrollLeft;
        const scrollTop = doc.scrollTop;
        // Reset position so that layout isn't affected by the previous position,
        // and so that the dialog is positioned at the top-start corner of the
        // document.
        this.resetStyle_();
        this.$.dialog.showModal();
        this.open = true;
        config.top += scrollTop;
        config.left += scrollLeft;
        this.positionDialog_(Object.assign({
            minX: scrollLeft,
            minY: scrollTop,
            maxX: scrollLeft + doc.clientWidth,
            maxY: scrollTop + doc.clientHeight,
        }, config));
        // Restore the scroll position.
        doc.scrollTop = scrollTop;
        doc.scrollLeft = scrollLeft;
        this.addListeners_();
        // Focus the first selectable item.
        const openedByKey = FocusOutlineManager.forDocument(document).visible;
        if (openedByKey) {
            const firstSelectableItem = this.querySelector(SELECTABLE_DROPDOWN_ITEM_QUERY);
            if (firstSelectableItem) {
                requestAnimationFrame(() => {
                    // Wait for the next animation frame for the dialog to become visible.
                    firstSelectableItem.focus();
                });
            }
        }
    }
    resetStyle_() {
        this.$.dialog.style.left = '';
        this.$.dialog.style.right = '';
        this.$.dialog.style.top = '0';
    }
    /**
     * Position the dialog using the coordinates in config. Coordinates are
     * relative to the top-left of the viewport when scrolled to (0, 0).
     */
    positionDialog_(config) {
        this.lastConfig_ = config;
        const c = Object.assign(getDefaultShowConfig(), config);
        const top = c.top;
        const left = c.left;
        const bottom = top + c.height;
        const right = left + c.width;
        // Flip the X anchor in RTL.
        const rtl = getComputedStyle(this).direction === 'rtl';
        if (rtl) {
            c.anchorAlignmentX *= -1;
        }
        const offsetWidth = this.$.dialog.offsetWidth;
        const menuLeft = getStartPointWithAnchor(left, right, offsetWidth, c.anchorAlignmentX, c.minX, c.maxX);
        if (rtl) {
            const menuRight = document.scrollingElement.clientWidth - menuLeft - offsetWidth;
            this.$.dialog.style.right = menuRight + 'px';
        }
        else {
            this.$.dialog.style.left = menuLeft + 'px';
        }
        const menuTop = getStartPointWithAnchor(top, bottom, this.$.dialog.offsetHeight, c.anchorAlignmentY, c.minY, c.maxY);
        this.$.dialog.style.top = menuTop + 'px';
    }
    onSlotchange_() {
        for (const node of this.$.contentNode.assignedElements({ flatten: true })) {
            if (node.classList.contains(DROPDOWN_ITEM_CLASS) &&
                !node.getAttribute('role')) {
                node.setAttribute('role', 'menuitem');
            }
        }
    }
    addListeners_() {
        this.boundClose_ = this.boundClose_ || (() => {
            if (this.$.dialog.open) {
                this.close();
            }
        });
        window.addEventListener('resize', this.boundClose_);
        window.addEventListener('popstate', this.boundClose_);
        if (this.autoReposition) {
            this.resizeObserver_ = new ResizeObserver(() => {
                if (this.lastConfig_) {
                    this.positionDialog_(this.lastConfig_);
                    this.fire('cr-action-menu-repositioned'); // For easier testing.
                }
            });
            this.resizeObserver_.observe(this.$.dialog);
        }
    }
}
customElements.define(CrActionMenuElement.is, CrActionMenuElement);

let instance$h = null;
function getCss$e() {
    return instance$h || (instance$h = [...[getCss$o()], css `:host{display:block;width:200px;position:relative;overflow:hidden}#progressContainer{position:relative}#progressContainer,:host([indeterminate]) #primaryProgress::after{height:var(--cr-progress-height,4px)}#primaryProgress,:host([indeterminate]) #primaryProgress::after{position:absolute;top:0;right:0;bottom:0;left:0}#progressContainer,:host([indeterminate]) #primaryProgress::after{background:var(--cr-progress-container-color,var(--google-grey-300))}#primaryProgress{transform-origin:left center;transform:scaleX(0);will-change:transform}#primaryProgress{background:var(--cr-progress-active-color,var(--google-green-500))}:host([disabled]) #primaryProgress{background:var(--cr-progress-disabled-active-color,var(--google-grey-500))}:host([indeterminate]:not([disabled])) #primaryProgress{transform-origin:right center;animation:indeterminate-bar var(--cr-progress-indeterminate-cycle-duration,2s) linear infinite}:host([indeterminate]:not([disabled])) #primaryProgress::after{content:"";transform-origin:center center;animation:indeterminate-splitter var(--cr-progress-indeterminate-cycle-duration,2s) linear infinite}@keyframes indeterminate-bar{0%{transform:scaleX(1) translateX(-100%)}50%{transform:scaleX(1) translateX(0%)}75%{transform:scaleX(1) translateX(0%);animation-timing-function:cubic-bezier(.28,.62,.37,.91)}100%{transform:scaleX(0) translateX(0%)}}@keyframes indeterminate-splitter{0%{transform:scaleX(.75) translateX(-125%)}30%{transform:scaleX(.75) translateX(-125%);animation-timing-function:cubic-bezier(.42,0,.6,.8)}90%{transform:scaleX(.75) translateX(125%)}100%{transform:scaleX(.75) translateX(125%)}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$8() {
    return html `
    <div id="progressContainer">
      <div id="primaryProgress"></div>
    </div>`;
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class CrProgressElement extends CrLitElement {
    static get is() {
        return 'cr-progress';
    }
    static get styles() {
        return getCss$e();
    }
    render() {
        return getHtml$8.bind(this)();
    }
    static get properties() {
        return {
            /**
             * The number that represents the current value.
             */
            value: { type: Number },
            /**
             * The number that indicates the minimum value of the range.
             */
            min: { type: Number },
            /**
             * The number that indicates the maximum value of the range.
             */
            max: { type: Number },
            /**
             * Specifies the value granularity of the range's value.
             */
            step: { type: Number },
            /**
             * Use an indeterminate progress indicator.
             */
            indeterminate: {
                type: Boolean,
                reflect: true,
            },
            /**
             * True if the progress is disabled.
             */
            disabled: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #value_accessor_storage = 0;
    get value() { return this.#value_accessor_storage; }
    set value(value) { this.#value_accessor_storage = value; }
    #min_accessor_storage = 0;
    get min() { return this.#min_accessor_storage; }
    set min(value) { this.#min_accessor_storage = value; }
    #max_accessor_storage = 100;
    get max() { return this.#max_accessor_storage; }
    set max(value) { this.#max_accessor_storage = value; }
    #step_accessor_storage = 1;
    get step() { return this.#step_accessor_storage; }
    set step(value) { this.#step_accessor_storage = value; }
    #indeterminate_accessor_storage = false;
    get indeterminate() { return this.#indeterminate_accessor_storage; }
    set indeterminate(value) { this.#indeterminate_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    firstUpdated(changedProperties) {
        super.firstUpdated(changedProperties);
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'progressbar');
        }
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        // Clamp the value to the range.
        if (changedProperties.has('min') || changedProperties.has('max') ||
            changedProperties.has('value') || changedProperties.has('step')) {
            const previous = changedProperties.get('value') || 0;
            const clampedValue = this.clampValue_(this.value);
            this.value = Number.isNaN(clampedValue) ? previous : clampedValue;
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('min') || changedProperties.has('max') ||
            changedProperties.has('value') || changedProperties.has('step')) {
            const ratio = (this.value - this.min) / (this.max - this.min);
            this.$.primaryProgress.style.transform = `scaleX(${ratio})`;
            this.setAttribute('aria-valuemin', this.min.toString());
            this.setAttribute('aria-valuemax', this.max.toString());
        }
        if (changedProperties.has('indeterminate') ||
            changedProperties.has('value')) {
            if (this.indeterminate) {
                this.removeAttribute('aria-valuenow');
            }
            else {
                this.setAttribute('aria-valuenow', this.value.toString());
            }
        }
        if (changedProperties.has('disabled')) {
            this.setAttribute('aria-disabled', this.disabled ? 'true' : 'false');
        }
    }
    clampValue_(value) {
        return Math.min(this.max, Math.max(this.min, this.calcStep_(value)));
    }
    calcStep_(value) {
        value = Number.parseFloat(value.toString());
        if (!this.step) {
            return value;
        }
        const numSteps = Math.round((value - this.min) / this.step);
        if (this.step < 1) {
            /**
             * For small values of this.step, if we calculate the step using
             * `Math.round(value / step) * step` we may hit a precision point issue
             * eg. 0.1 * 0.2 =  0.020000000000000004
             * http://docs.oracle.com/cd/E19957-01/806-3568/ncg_goldberg.html
             *
             * as a work around we can divide by the reciprocal of `step`
             */
            return numSteps / (1 / this.step) + this.min;
        }
        else {
            return numSteps * this.step + this.min;
        }
    }
}
customElements.define(CrProgressElement.is, CrProgressElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/* List commonly used icons here to prevent duplication.
 * Do not add rarely used icons here; place those in your application.
 * Note that 20px and 24px icons are specified separately (size="", below).
 *
 * Icons are rendered at 20x20 px, but we don't have 20 px SVGs for everything.
 * The 24 px icons are used where 20 px icons are unavailable (which may appear
 * blurry at 20 px). Please use 20 px icons when available.
 */
const div = document.createElement('div');
div.innerHTML = getTrustedHTML `
<cr-iconset name="cr20" size="20">
  <svg>
    <defs>
      <!--
      Keep these in sorted order by id="".
      -->
      <g id="block">
        <path fill-rule="evenodd" clip-rule="evenodd"
          d="M10 0C4.48 0 0 4.48 0 10C0 15.52 4.48 20 10 20C15.52 20 20 15.52 20 10C20 4.48 15.52 0 10 0ZM2 10C2 5.58 5.58 2 10 2C11.85 2 13.55 2.63 14.9 3.69L3.69 14.9C2.63 13.55 2 11.85 2 10ZM5.1 16.31C6.45 17.37 8.15 18 10 18C14.42 18 18 14.42 18 10C18 8.15 17.37 6.45 16.31 5.1L5.1 16.31Z">
        </path>
      </g>
      <g id="cloud-off">
        <path
          d="M16 18.125L13.875 16H5C3.88889 16 2.94444 15.6111 2.16667 14.8333C1.38889 14.0556 1 13.1111 1 12C1 10.9444 1.36111 10.0347 2.08333 9.27083C2.80556 8.50694 3.6875 8.09028 4.72917 8.02083C4.77083 7.86805 4.8125 7.72222 4.85417 7.58333C4.90972 7.44444 4.97222 7.30555 5.04167 7.16667L1.875 4L2.9375 2.9375L17.0625 17.0625L16 18.125ZM5 14.5H12.375L6.20833 8.33333C6.15278 8.51389 6.09722 8.70139 6.04167 8.89583C6 9.07639 5.95139 9.25694 5.89583 9.4375L4.83333 9.52083C4.16667 9.57639 3.61111 9.84028 3.16667 10.3125C2.72222 10.7708 2.5 11.3333 2.5 12C2.5 12.6944 2.74306 13.2847 3.22917 13.7708C3.71528 14.2569 4.30556 14.5 5 14.5ZM17.5 15.375L16.3958 14.2917C16.7153 14.125 16.9792 13.8819 17.1875 13.5625C17.3958 13.2431 17.5 12.8889 17.5 12.5C17.5 11.9444 17.3056 11.4722 16.9167 11.0833C16.5278 10.6944 16.0556 10.5 15.5 10.5H14.125L14 9.14583C13.9028 8.11806 13.4722 7.25694 12.7083 6.5625C11.9444 5.85417 11.0417 5.5 10 5.5C9.65278 5.5 9.31944 5.54167 9 5.625C8.69444 5.70833 8.39583 5.82639 8.10417 5.97917L7.02083 4.89583C7.46528 4.61806 7.93056 4.40278 8.41667 4.25C8.91667 4.08333 9.44444 4 10 4C11.4306 4 12.6736 4.48611 13.7292 5.45833C14.7847 6.41667 15.375 7.59722 15.5 9C16.4722 9 17.2986 9.34028 17.9792 10.0208C18.6597 10.7014 19 11.5278 19 12.5C19 13.0972 18.8611 13.6458 18.5833 14.1458C18.3194 14.6458 17.9583 15.0556 17.5 15.375Z">
        </path>
      </g>
      <g id="delete">
        <path
          d="M 5.832031 17.5 C 5.375 17.5 4.984375 17.335938 4.65625 17.011719 C 4.328125 16.683594 4.167969 16.292969 4.167969 15.832031 L 4.167969 5 L 3.332031 5 L 3.332031 3.332031 L 7.5 3.332031 L 7.5 2.5 L 12.5 2.5 L 12.5 3.332031 L 16.667969 3.332031 L 16.667969 5 L 15.832031 5 L 15.832031 15.832031 C 15.832031 16.292969 15.671875 16.683594 15.34375 17.011719 C 15.015625 17.335938 14.625 17.5 14.167969 17.5 Z M 14.167969 5 L 5.832031 5 L 5.832031 15.832031 L 14.167969 15.832031 Z M 7.5 14.167969 L 9.167969 14.167969 L 9.167969 6.667969 L 7.5 6.667969 Z M 10.832031 14.167969 L 12.5 14.167969 L 12.5 6.667969 L 10.832031 6.667969 Z M 5.832031 5 L 5.832031 15.832031 Z M 5.832031 5 ">
        </path>
      </g>
      <g id="domain" viewBox="0 -960 960 960">
        <path d="M96-144v-672h384v144h384v528H96Zm72-72h72v-72h-72v72Zm0-152h72v-72h-72v72Zm0-152h72v-72h-72v72Zm0-152h72v-72h-72v72Zm168 456h72v-72h-72v72Zm0-152h72v-72h-72v72Zm0-152h72v-72h-72v72Zm0-152h72v-72h-72v72Zm144 456h312v-384H480v80h72v72h-72v80h72v72h-72v80Zm168-232v-72h72v72h-72Zm0 152v-72h72v72h-72Z"></path>
      </g>
      <g id="kite">
        <path fill-rule="evenodd" clip-rule="evenodd"
          d="M4.6327 8.00094L10.3199 2L16 8.00094L10.1848 16.8673C10.0995 16.9873 10.0071 17.1074 9.90047 17.2199C9.42417 17.7225 8.79147 18 8.11611 18C7.44076 18 6.80806 17.7225 6.33175 17.2199C5.85545 16.7173 5.59242 16.0497 5.59242 15.3371C5.59242 14.977 5.46445 14.647 5.22275 14.3919C4.98104 14.1369 4.66825 14.0019 4.32701 14.0019H4V12.6667H4.32701C5.00237 12.6667 5.63507 12.9442 6.11137 13.4468C6.58768 13.9494 6.85071 14.617 6.85071 15.3296C6.85071 15.6896 6.97867 16.0197 7.22038 16.2747C7.46209 16.5298 7.77488 16.6648 8.11611 16.6648C8.45735 16.6648 8.77014 16.5223 9.01185 16.2747C9.02396 16.2601 9.03607 16.246 9.04808 16.2319C9.08541 16.1883 9.12176 16.1458 9.15403 16.0947L9.55213 15.4946L4.6327 8.00094ZM10.3199 13.9371L6.53802 8.17116L10.3199 4.1814L14.0963 8.17103L10.3199 13.9371Z">
        </path>
      </g>
      <g id="menu">
        <path d="M2 4h16v2H2zM2 9h16v2H2zM2 14h16v2H2z"></path>
      </g>
      <g id="password">
        <path d="M5.833 11.667c.458 0 .847-.16 1.167-.479.333-.333.5-.729.5-1.188s-.167-.847-.5-1.167a1.555 1.555 0 0 0-1.167-.5c-.458 0-.854.167-1.188.5A1.588 1.588 0 0 0 4.166 10c0 .458.16.854.479 1.188.333.319.729.479 1.188.479Zm0 3.333c-1.389 0-2.569-.486-3.542-1.458C1.319 12.569.833 11.389.833 10c0-1.389.486-2.569 1.458-3.542C3.264 5.486 4.444 5 5.833 5c.944 0 1.813.243 2.604.729a4.752 4.752 0 0 1 1.833 1.979h7.23c.458 0 .847.167 1.167.5.333.319.5.708.5 1.167v3.958c0 .458-.167.854-.5 1.188A1.588 1.588 0 0 1 17.5 15h-3.75a1.658 1.658 0 0 1-1.188-.479 1.658 1.658 0 0 1-.479-1.188v-1.042H10.27a4.59 4.59 0 0 1-1.813 2A5.1 5.1 0 0 1 5.833 15Zm3.292-4.375h4.625v2.708H15v-1.042a.592.592 0 0 1 .167-.438.623.623 0 0 1 .458-.188c.181 0 .327.063.438.188a.558.558 0 0 1 .188.438v1.042H17.5V9.375H9.125a3.312 3.312 0 0 0-1.167-1.938 3.203 3.203 0 0 0-2.125-.77 3.21 3.21 0 0 0-2.354.979C2.827 8.298 2.5 9.083 2.5 10s.327 1.702.979 2.354a3.21 3.21 0 0 0 2.354.979c.806 0 1.514-.25 2.125-.75.611-.514 1-1.167 1.167-1.958Z"></path>
      </g>
      
  </svg>
</cr-iconset>

<!-- NOTE: In the common case that the final icon will be 20x20, export the SVG
     at 20px and place it in the section above. -->
<cr-iconset name="cr" size="24">
  <svg>
    <defs>
      <!--
      These icons are copied from Polymer's iron-icons and kept in sorted order.
      -->
      <g id="add">
        <path d="M19 13h-6v6h-2v-6H5v-2h6V5h2v6h6v2z" />
      </g>
      <g id="arrow-back">
        <path
          d="m7.824 13 5.602 5.602L12 20l-8-8 8-8 1.426 1.398L7.824 11H20v2Zm0 0">
        </path>
      </g>
      <g id="arrow-drop-up">
        <path d="M7 14l5-5 5 5z"></path>
      </g>
      <g id="arrow-drop-down">
        <path d="M7 10l5 5 5-5z"></path>
      </g>
      <g id="arrow-forward">
        <path
          d="M12 4l-1.41 1.41L16.17 11H4v2h12.17l-5.58 5.59L12 20l8-8z">
        </path>
      </g>
      <g id="arrow-right">
        <path d="M10 7l5 5-5 5z"></path>
      </g>
      <g id="cancel">
        <path
          d="M12 2C6.47 2 2 6.47 2 12s4.47 10 10 10 10-4.47 10-10S17.53 2 12 2zm5 13.59L15.59 17 12 13.41 8.41 17 7 15.59 10.59 12 7 8.41 8.41 7 12 10.59 15.59 7 17 8.41 13.41 12 17 15.59z">
        </path>
      </g>
      <g id="check">
        <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z"></path>
      </g>
      <g id="check-circle" viewBox="0 -960 960 960">
        <path d="m424-296 282-282-56-56-226 226-114-114-56 56 170 170Zm56 216q-83 0-156-31.5T197-197q-54-54-85.5-127T80-480q0-83 31.5-156T197-763q54-54 127-85.5T480-880q83 0 156 31.5T763-763q54 54 85.5 127T880-480q0 83-31.5 156T763-197q-54 54-127 85.5T480-80Zm0-80q134 0 227-93t93-227q0-134-93-227t-227-93q-134 0-227 93t-93 227q0 134 93 227t227 93Zm0-320Z"></path>
      </g>
      <g id="chevron-left">
        <path d="M15.41 7.41L14 6l-6 6 6 6 1.41-1.41L10.83 12z"></path>
      </g>
      <g id="chevron-right">
        <path d="M10 6L8.59 7.41 13.17 12l-4.58 4.59L10 18l6-6z"></path>
      </g>
      <g id="clear">
        <path
          d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z">
        </path>
      </g>
      <g id="chrome-product" viewBox="0 -960 960 960">
        <path d="M336-479q0 60 42 102t102 42q60 0 102-42t42-102q0-60-42-102t-102-42q-60 0-102 42t-42 102Zm144 216q11 0 22.5-.5T525-267L427-99q-144-16-237.5-125T96-479q0-43 9.5-84.5T134-645l160 274q28 51 78 79.5T480-263Zm0-432q-71 0-126.5 42T276-545l-98-170q53-71 132.5-109.5T480-863q95 0 179 45t138 123H480Zm356 72q15 35 21.5 71t6.5 73q0 155-100 260.5T509-96l157-275q14-25 22-52t8-56q0-40-15-77t-41-67h196Z">
        </path>
      </g>
      <g id="close">
        <path
          d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z">
        </path>
      </g>
      <g id="computer">
        <path
          d="M20 18c1.1 0 1.99-.9 1.99-2L22 6c0-1.1-.9-2-2-2H4c-1.1 0-2 .9-2 2v10c0 1.1.9 2 2 2H0v2h24v-2h-4zM4 6h16v10H4V6z">
        </path>
      </g>
      <g id="create">
        <path
          d="M3 17.25V21h3.75L17.81 9.94l-3.75-3.75L3 17.25zM20.71 7.04c.39-.39.39-1.02 0-1.41l-2.34-2.34c-.39-.39-1.02-.39-1.41 0l-1.83 1.83 3.75 3.75 1.83-1.83z">
        </path>
      </g>
      <g id="delete" viewBox="0 -960 960 960">
        <path
          d="M309.37-135.87q-34.48 0-58.74-24.26-24.26-24.26-24.26-58.74v-474.5h-53.5v-83H378.5v-53.5h202.52v53.5h206.11v83h-53.5v474.07q0 35.21-24.26 59.32t-58.74 24.11H309.37Zm341.26-557.5H309.37v474.5h341.26v-474.5ZM379.7-288.24h77.5v-336h-77.5v336Zm123.1 0h77.5v-336h-77.5v336ZM309.37-693.37v474.5-474.5Z">
        </path>
      </g>
      <g id="domain">
        <path
          d="M12 7V3H2v18h20V7H12zM6 19H4v-2h2v2zm0-4H4v-2h2v2zm0-4H4V9h2v2zm0-4H4V5h2v2zm4 12H8v-2h2v2zm0-4H8v-2h2v2zm0-4H8V9h2v2zm0-4H8V5h2v2zm10 12h-8v-2h2v-2h-2v-2h2v-2h-2V9h8v10zm-2-8h-2v2h2v-2zm0 4h-2v2h2v-2z">
        </path>
      </g>
      <!-- source: https://fonts.google.com/icons?selected=Material+Symbols+Outlined:family_link:FILL@0;wght@0;GRAD@0;opsz@24&icon.size=24&icon.color=%23e8eaed -->
      <g id="kite" viewBox="0 -960 960 960">
        <path
          d="M390-40q-51 0-90.5-30.5T246-149q-6-23-25-37t-43-14q-16 0-30 6.5T124-175l-61-51q21-26 51.5-40t63.5-14q51 0 91 30t54 79q6 23 25 37t42 14q19 0 34-10t26-25l1-2-276-381q-8-11-11.5-23t-3.5-24q0-16 6-30.5t18-26.5l260-255q11-11 26-17t30-6q15 0 30 6t26 17l260 255q12 12 18 26.5t6 30.5q0 12-3.5 24T825-538L500-88q-18 25-48 36.5T390-40Zm110-185 260-360-260-255-259 256 259 359Zm1-308Z"/>
        </path>
      </g>
      <g id="error">
        <path
          d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-2h2v2zm0-4h-2V7h2v6z">
        </path>
      </g>
      <g id="error-outline">
        <path
          d="M11 15h2v2h-2zm0-8h2v6h-2zm.99-5C6.47 2 2 6.48 2 12s4.47 10 9.99 10C17.52 22 22 17.52 22 12S17.52 2 11.99 2zM12 20c-4.42 0-8-3.58-8-8s3.58-8 8-8 8 3.58 8 8-3.58 8-8 8z">
        </path>
      </g>
      <g id="expand-less">
        <path d="M12 8l-6 6 1.41 1.41L12 10.83l4.59 4.58L18 14z"></path>
      </g>
      <g id="expand-more">
        <path d="M16.59 8.59L12 13.17 7.41 8.59 6 10l6 6 6-6z"></path>
      </g>
      <g id="extension">
        <path
          d="M20.5 11H19V7c0-1.1-.9-2-2-2h-4V3.5C13 2.12 11.88 1 10.5 1S8 2.12 8 3.5V5H4c-1.1 0-1.99.9-1.99 2v3.8H3.5c1.49 0 2.7 1.21 2.7 2.7s-1.21 2.7-2.7 2.7H2V20c0 1.1.9 2 2 2h3.8v-1.5c0-1.49 1.21-2.7 2.7-2.7 1.49 0 2.7 1.21 2.7 2.7V22H17c1.1 0 2-.9 2-2v-4h1.5c1.38 0 2.5-1.12 2.5-2.5S21.88 11 20.5 11z">
        </path>
      </g>
      <g id="file-download" viewBox="0 -960 960 960">
        <path d="M480-336 288-528l51-51 105 105v-342h72v342l105-105 51 51-192 192ZM263.72-192Q234-192 213-213.15T192-264v-72h72v72h432v-72h72v72q0 29.7-21.16 50.85Q725.68-192 695.96-192H263.72Z"></path>
      </g>
      <g id="fullscreen">
        <path
          d="M7 14H5v5h5v-2H7v-3zm-2-4h2V7h3V5H5v5zm12 7h-3v2h5v-5h-2v3zM14 5v2h3v3h2V5h-5z">
        </path>
      </g>
      <g id="group">
        <path
          d="M16 11c1.66 0 2.99-1.34 2.99-3S17.66 5 16 5c-1.66 0-3 1.34-3 3s1.34 3 3 3zm-8 0c1.66 0 2.99-1.34 2.99-3S9.66 5 8 5C6.34 5 5 6.34 5 8s1.34 3 3 3zm0 2c-2.33 0-7 1.17-7 3.5V19h14v-2.5c0-2.33-4.67-3.5-7-3.5zm8 0c-.29 0-.62.02-.97.05 1.16.84 1.97 1.97 1.97 3.45V19h6v-2.5c0-2.33-4.67-3.5-7-3.5z">
        </path>
      </g>
      <g id="help-outline">
        <path
          d="M11 18h2v-2h-2v2zm1-16C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 18c-4.41 0-8-3.59-8-8s3.59-8 8-8 8 3.59 8 8-3.59 8-8 8zm0-14c-2.21 0-4 1.79-4 4h2c0-1.1.9-2 2-2s2 .9 2 2c0 2-3 1.75-3 5h2c0-2.25 3-2.5 3-5 0-2.21-1.79-4-4-4z">
        </path>
      </g>
      <g id="history">
        <path
          d="M12.945312 22.75 C 10.320312 22.75 8.074219 21.839844 6.207031 20.019531 C 4.335938 18.199219 3.359375 15.972656 3.269531 13.34375 L 5.089844 13.34375 C 5.175781 15.472656 5.972656 17.273438 7.480469 18.742188 C 8.988281 20.210938 10.808594 20.945312 12.945312 20.945312 C 15.179688 20.945312 17.070312 20.164062 18.621094 18.601562 C 20.167969 17.039062 20.945312 15.144531 20.945312 12.910156 C 20.945312 10.714844 20.164062 8.855469 18.601562 7.335938 C 17.039062 5.816406 15.15625 5.054688 12.945312 5.054688 C 11.710938 5.054688 10.554688 5.339844 9.480469 5.902344 C 8.402344 6.46875 7.476562 7.226562 6.699219 8.179688 L 9.585938 8.179688 L 9.585938 9.984375 L 3.648438 9.984375 L 3.648438 4.0625 L 5.453125 4.0625 L 5.453125 6.824219 C 6.386719 5.707031 7.503906 4.828125 8.804688 4.199219 C 10.109375 3.566406 11.488281 3.25 12.945312 3.25 C 14.300781 3.25 15.570312 3.503906 16.761719 4.011719 C 17.949219 4.519531 18.988281 5.214844 19.875 6.089844 C 20.761719 6.964844 21.464844 7.992188 21.976562 9.167969 C 22.492188 10.34375 22.75 11.609375 22.75 12.964844 C 22.75 14.316406 22.492188 15.589844 21.976562 16.777344 C 21.464844 17.964844 20.761719 19.003906 19.875 19.882812 C 18.988281 20.765625 17.949219 21.464844 16.761719 21.976562 C 15.570312 22.492188 14.300781 22.75 12.945312 22.75 Z M 16.269531 17.460938 L 12.117188 13.34375 L 12.117188 7.527344 L 13.921875 7.527344 L 13.921875 12.601562 L 17.550781 16.179688 Z M 16.269531 17.460938">
        </path>
      </g>
      <g id="info">
        <path
          d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-6h2v6zm0-8h-2V7h2v2z">
        </path>
      </g>
      <g id="info-outline">
        <path
          d="M11 17h2v-6h-2v6zm1-15C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 18c-4.41 0-8-3.59-8-8s3.59-8 8-8 8 3.59 8 8-3.59 8-8 8zM11 9h2V7h-2v2z">
        </path>
      </g>
      <g id="insert-drive-file">
        <path
          d="M6 2c-1.1 0-1.99.9-1.99 2L4 20c0 1.1.89 2 1.99 2H18c1.1 0 2-.9 2-2V8l-6-6H6zm7 7V3.5L18.5 9H13z">
        </path>
      </g>
      <g id="location-on">
        <path
          d="M12 2C8.13 2 5 5.13 5 9c0 5.25 7 13 7 13s7-7.75 7-13c0-3.87-3.13-7-7-7zm0 9.5c-1.38 0-2.5-1.12-2.5-2.5s1.12-2.5 2.5-2.5 2.5 1.12 2.5 2.5-1.12 2.5-2.5 2.5z">
        </path>
      </g>
      <g id="mic">
        <path
          d="M12 14c1.66 0 2.99-1.34 2.99-3L15 5c0-1.66-1.34-3-3-3S9 3.34 9 5v6c0 1.66 1.34 3 3 3zm5.3-3c0 3-2.54 5.1-5.3 5.1S6.7 14 6.7 11H5c0 3.41 2.72 6.23 6 6.72V21h2v-3.28c3.28-.48 6-3.3 6-6.72h-1.7z">
        </path>
      </g>
      <g id="more-vert">
        <path
          d="M12 8c1.1 0 2-.9 2-2s-.9-2-2-2-2 .9-2 2 .9 2 2 2zm0 2c-1.1 0-2 .9-2 2s.9 2 2 2 2-.9 2-2-.9-2-2-2zm0 6c-1.1 0-2 .9-2 2s.9 2 2 2 2-.9 2-2-.9-2-2-2z">
        </path>
      </g>
      <g id="open-in-new" viewBox="0 -960 960 960">
        <path
          d="M216-144q-29.7 0-50.85-21.15Q144-186.3 144-216v-528q0-29.7 21.15-50.85Q186.3-816 216-816h264v72H216v528h528v-264h72v264q0 29.7-21.15 50.85Q773.7-144 744-144H216Zm171-192-51-51 357-357H576v-72h240v240h-72v-117L387-336Z">
        </path>
      </g>
      <g id="person">
        <path
          d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z">
        </path>
      </g>
      <g id="phonelink">
        <path
          d="M4 6h18V4H4c-1.1 0-2 .9-2 2v11H0v3h14v-3H4V6zm19 2h-6c-.55 0-1 .45-1 1v10c0 .55.45 1 1 1h6c.55 0 1-.45 1-1V9c0-.55-.45-1-1-1zm-1 9h-4v-7h4v7z">
        </path>
      </g>
      <g id="print">
        <path
          d="M19 8H5c-1.66 0-3 1.34-3 3v6h4v4h12v-4h4v-6c0-1.66-1.34-3-3-3zm-3 11H8v-5h8v5zm3-7c-.55 0-1-.45-1-1s.45-1 1-1 1 .45 1 1-.45 1-1 1zm-1-9H6v4h12V3z">
        </path>
      </g>
      <g id="schedule">
        <path
          d="M11.99 2C6.47 2 2 6.48 2 12s4.47 10 9.99 10C17.52 22 22 17.52 22 12S17.52 2 11.99 2zM12 20c-4.42 0-8-3.58-8-8s3.58-8 8-8 8 3.58 8 8-3.58 8-8 8zm.5-13H11v6l5.25 3.15.75-1.23-4.5-2.67z">
        </path>
      </g>
      <g id="search">
        <path
          d="M15.5 14h-.79l-.28-.27C15.41 12.59 16 11.11 16 9.5 16 5.91 13.09 3 9.5 3S3 5.91 3 9.5 5.91 16 9.5 16c1.61 0 3.09-.59 4.23-1.57l.27.28v.79l5 4.99L20.49 19l-4.99-5zm-6 0C7.01 14 5 11.99 5 9.5S7.01 5 9.5 5 14 7.01 14 9.5 11.99 14 9.5 14z">
        </path>
      </g>
      <g id="security">
        <path
          d="M12 1L3 5v6c0 5.55 3.84 10.74 9 12 5.16-1.26 9-6.45 9-12V5l-9-4zm0 10.99h7c-.53 4.12-3.28 7.79-7 8.94V12H5V6.3l7-3.11v8.8z">
        </path>
      </g>
      <!-- The <g> IDs are exposed as global variables in Vulcanized mode, which
        conflicts with the "settings" namespace of MD Settings. Using an "_icon"
        suffix prevents the naming conflict. -->
      <g id="settings_icon">
        <path
          d="M19.43 12.98c.04-.32.07-.64.07-.98s-.03-.66-.07-.98l2.11-1.65c.19-.15.24-.42.12-.64l-2-3.46c-.12-.22-.39-.3-.61-.22l-2.49 1c-.52-.4-1.08-.73-1.69-.98l-.38-2.65C14.46 2.18 14.25 2 14 2h-4c-.25 0-.46.18-.49.42l-.38 2.65c-.61.25-1.17.59-1.69.98l-2.49-1c-.23-.09-.49 0-.61.22l-2 3.46c-.13.22-.07.49.12.64l2.11 1.65c-.04.32-.07.65-.07.98s.03.66.07.98l-2.11 1.65c-.19.15-.24.42-.12.64l2 3.46c.12.22.39.3.61.22l2.49-1c.52.4 1.08.73 1.69.98l.38 2.65c.03.24.24.42.49.42h4c.25 0 .46-.18.49-.42l.38-2.65c.61-.25 1.17-.59 1.69-.98l2.49 1c.23.09.49 0 .61-.22l2-3.46c.12-.22.07-.49-.12-.64l-2.11-1.65zM12 15.5c-1.93 0-3.5-1.57-3.5-3.5s1.57-3.5 3.5-3.5 3.5 1.57 3.5 3.5-1.57 3.5-3.5 3.5z">
        </path>
      </g>
      <g id="star">
        <path
          d="M12 17.27L18.18 21l-1.64-7.03L22 9.24l-7.19-.61L12 2 9.19 8.63 2 9.24l5.46 4.73L5.82 21z">
        </path>
      </g>
      <g id="sync" viewBox="0 -960 960 960">
        <path
          d="M216-192v-72h74q-45-40-71.5-95.5T192-480q0-101 61-177.5T408-758v75q-63 23-103.5 77.5T264-480q0 48 19.5 89t52.5 70v-63h72v192H216Zm336-10v-75q63-23 103.5-77.5T696-480q0-48-19.5-89T624-639v63h-72v-192h192v72h-74q45 40 71.5 95.5T768-480q0 101-61 177.5T552-202Z">
        </path>
      </g>
      <g id="thumbs-down">
        <path
            d="M6 3h11v13l-7 7-1.25-1.25a1.454 1.454 0 0 1-.3-.475c-.067-.2-.1-.392-.1-.575v-.35L9.45 16H3c-.533 0-1-.2-1.4-.6-.4-.4-.6-.867-.6-1.4v-2c0-.117.017-.242.05-.375s.067-.258.1-.375l3-7.05c.15-.333.4-.617.75-.85C5.25 3.117 5.617 3 6 3Zm9 2H6l-3 7v2h9l-1.35 5.5L15 15.15V5Zm0 10.15V5v10.15Zm2 .85v-2h3V5h-3V3h5v13h-5Z">
        </path>
      </g>
      <g id="thumbs-down-filled">
        <path
            d="M6 3h10v13l-7 7-1.25-1.25a1.336 1.336 0 0 1-.29-.477 1.66 1.66 0 0 1-.108-.574v-.347L8.449 16H3c-.535 0-1-.2-1.398-.602C1.199 15 1 14.535 1 14v-2c0-.117.012-.242.04-.375.022-.133.062-.258.108-.375l3-7.05c.153-.333.403-.618.75-.848A1.957 1.957 0 0 1 6 3Zm12 13V3h4v13Zm0 0">
        </path>
      </g>
      <g id="thumbs-up">
        <path
            d="M18 21H7V8l7-7 1.25 1.25c.117.117.208.275.275.475.083.2.125.392.125.575v.35L14.55 8H21c.533 0 1 .2 1.4.6.4.4.6.867.6 1.4v2c0 .117-.017.242-.05.375s-.067.258-.1.375l-3 7.05c-.15.333-.4.617-.75.85-.35.233-.717.35-1.1.35Zm-9-2h9l3-7v-2h-9l1.35-5.5L9 8.85V19ZM9 8.85V19 8.85ZM7 8v2H4v9h3v2H2V8h5Z">
        </path>
      </g>
      <g id="thumbs-up-filled">
        <path
            d="M18 21H8V8l7-7 1.25 1.25c.117.117.21.273.29.477.073.199.108.39.108.574v.347L15.551 8H21c.535 0 1 .2 1.398.602C22.801 9 23 9.465 23 10v2c0 .117-.012.242-.04.375a1.897 1.897 0 0 1-.108.375l-3 7.05a2.037 2.037 0 0 1-.75.848A1.957 1.957 0 0 1 18 21ZM6 8v13H2V8Zm0 0">
      </g>
      <g id="videocam" viewBox="0 -960 960 960">
        <path
          d="M216-192q-29 0-50.5-21.5T144-264v-432q0-29.7 21.5-50.85Q187-768 216-768h432q29.7 0 50.85 21.15Q720-725.7 720-696v168l144-144v384L720-432v168q0 29-21.15 50.5T648-192H216Zm0-72h432v-432H216v432Zm0 0v-432 432Z">
        </path>
      </g>
      <g id="warning">
        <path d="M1 21h22L12 2 1 21zm12-3h-2v-2h2v2zm0-4h-2v-4h2v4z"></path>
      </g>
    </defs>
  </svg>
</cr-iconset>`;
const iconsets = div.querySelectorAll('cr-iconset');
for (const iconset of iconsets) {
    document.head.appendChild(iconset);
}

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Action links are elements that are used to perform an in-page navigation or
// action (e.g. showing a dialog).
//
// They look like normal anchor (<a>) tags as their text color is blue. However,
// they're subtly different as they're not initially underlined (giving users a
// clue that underlined links navigate while action links don't).
//
// Action links look very similar to normal links when hovered (hand cursor,
// underlined). This gives the user an idea that clicking this link will do
// something similar to navigation but in the same page.
//
// They can be created in JavaScript like this (note second arg):
//
//   var link = document.createElement('a', {is: 'action-link'});
//
// or with a constructor like this:
//
//   var link = new ActionLink();
//
// They can be used easily from HTML as well, like so:
//
//   <a is="action-link">Click me!</a>
//
// NOTE: <action-link> and document.createElement('action-link') don't work.
class ActionLink extends HTMLAnchorElement {
    boundOnKeyDown_ = null;
    boundOnMouseDown_ = null;
    boundOnBlur_ = null;
    connectedCallback() {
        // Action links can start disabled (e.g. <a is="action-link" disabled>).
        this.tabIndex = this.disabled ? -1 : 0;
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'link');
        }
        this.boundOnKeyDown_ = (e) => {
            if (!this.disabled && e.key === 'Enter' && !this.href) {
                // Schedule a click asynchronously because other 'keydown' handlers
                // may still run later (e.g. document.addEventListener('keydown')).
                // Specifically options dialogs break when this timeout isn't here.
                // NOTE: this affects the "trusted" state of the ensuing click. I
                // haven't found anything that breaks because of this (yet).
                window.setTimeout(() => this.click(), 0);
            }
        };
        this.addEventListener('keydown', this.boundOnKeyDown_);
        function preventDefault(e) {
            e.preventDefault();
        }
        function removePreventDefault() {
            document.removeEventListener('selectstart', preventDefault);
            document.removeEventListener('mouseup', removePreventDefault);
        }
        this.boundOnMouseDown_ = () => {
            // This handlers strives to match the behavior of <a href="...">.
            // While the mouse is down, prevent text selection from dragging.
            document.addEventListener('selectstart', preventDefault);
            document.addEventListener('mouseup', removePreventDefault);
            // If focus started via mouse press, don't show an outline.
            if (document.activeElement !== this) {
                this.classList.add('no-outline');
            }
        };
        this.addEventListener('mousedown', this.boundOnMouseDown_);
        this.boundOnBlur_ = () => this.classList.remove('no-outline');
        this.addEventListener('blur', this.boundOnBlur_);
    }
    disconnectedCallback() {
        this.removeEventListener('keydown', this.boundOnKeyDown_);
        this.boundOnKeyDown_ = null;
        this.removeEventListener('mousedown', this.boundOnMouseDown_);
        this.boundOnMouseDown_ = null;
        this.removeEventListener('blur', this.boundOnBlur_);
        this.boundOnBlur_ = null;
    }
    set disabled(disabled) {
        if (disabled) {
            HTMLAnchorElement.prototype.setAttribute.call(this, 'disabled', '');
        }
        else {
            HTMLAnchorElement.prototype.removeAttribute.call(this, 'disabled');
        }
        this.tabIndex = disabled ? -1 : 0;
    }
    get disabled() {
        return this.hasAttribute('disabled');
    }
    setAttribute(attr, val) {
        if (attr.toLowerCase() === 'disabled') {
            this.disabled = true;
        }
        else {
            super.setAttribute(attr, val);
        }
    }
    removeAttribute(attr) {
        if (attr.toLowerCase() === 'disabled') {
            this.disabled = false;
        }
        else {
            super.removeAttribute(attr);
        }
    }
}
customElements.define('action-link', ActionLink, { extends: 'a' });

let instance$g = null;
function getCss$d() {
    return instance$g || (instance$g = [...[], css `:host{--cr-toast-background:var(--color-toast-background,var(--cr-fallback-color-inverse-surface));--cr-toast-button-color:var(--color-toast-button,var(--cr-fallback-color-inverse-primary));--cr-toast-text-color:var(--color-toast-foreground,var(--cr-fallback-color-inverse-on-surface));--cr-focus-outline-color:var(--cr-focus-outline-inverse-color)}:host{align-items:center;background:var(--cr-toast-background);border-radius:8px;bottom:0;box-shadow:0 2px 4px 0 rgba(0,0,0,0.28);box-sizing:border-box;display:flex;line-height:20px;margin:24px;max-width:var(--cr-toast-max-width,568px);min-height:52px;min-width:288px;opacity:0;padding:0 16px;position:fixed;transform:translateY(100px);transition:opacity 300ms,transform 300ms;visibility:hidden;z-index:1}:host-context([dir=ltr]){left:0}:host-context([dir=rtl]){right:0}:host([open]){opacity:1;transform:translateY(0);visibility:visible}:host(:not([open])) ::slotted(*){display:none}:host ::slotted(*){color:var(--cr-toast-text-color)}:host ::slotted(cr-button){background-color:transparent !important;border:none !important;color:var(--cr-toast-button-color) !important;margin-inline-start:32px !important;min-width:52px !important;padding:8px !important}:host ::slotted(cr-button:hover){background-color:transparent !important}::slotted(cr-button:last-of-type){margin-inline-end:-8px}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$7() {
    return html `<slot></slot>`;
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview A lightweight toast.
 */
class CrToastElement extends CrLitElement {
    static get is() {
        return 'cr-toast';
    }
    static get styles() {
        return getCss$d();
    }
    render() {
        return getHtml$7.bind(this)();
    }
    static get properties() {
        return {
            duration: {
                type: Number,
            },
            open: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #duration_accessor_storage = 0;
    get duration() { return this.#duration_accessor_storage; }
    set duration(value) { this.#duration_accessor_storage = value; }
    #open_accessor_storage = false;
    get open() { return this.#open_accessor_storage; }
    set open(value) { this.#open_accessor_storage = value; }
    hideTimeoutId_ = null;
    constructor() {
        super();
        this.addEventListener('focusin', this.clearTimeout_);
        this.addEventListener('focusout', this.resetAutoHide_);
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('duration') || changedProperties.has('open')) {
            this.resetAutoHide_();
        }
    }
    clearTimeout_() {
        if (this.hideTimeoutId_ !== null) {
            window.clearTimeout(this.hideTimeoutId_);
            this.hideTimeoutId_ = null;
        }
    }
    /**
     * Cancels existing auto-hide, and sets up new auto-hide.
     */
    resetAutoHide_() {
        this.clearTimeout_();
        if (this.open && this.duration !== 0) {
            this.hideTimeoutId_ = window.setTimeout(() => {
                this.hide();
            }, this.duration);
        }
    }
    /**
     * Shows the toast and auto-hides after |this.duration| milliseconds has
     * passed. If the toast is currently being shown, any preexisting auto-hide
     * is cancelled and replaced with a new auto-hide.
     */
    async show() {
        // Force autohide to reset if calling show on an already shown toast.
        const shouldResetAutohide = this.open;
        // The role attribute is removed first so that screen readers to better
        // ensure that screen readers will read out the content inside the toast.
        // If the role is not removed and re-added back in, certain screen readers
        // do not read out the contents, especially if the text remains exactly
        // the same as a previous toast.
        this.removeAttribute('role');
        this.open = true;
        await this.updateComplete;
        this.setAttribute('role', 'alert');
        if (shouldResetAutohide) {
            this.resetAutoHide_();
        }
    }
    /**
     * Hides the toast and ensures that its contents can not be focused while
     * hidden.
     */
    async hide() {
        this.open = false;
        await this.updateComplete;
    }
}
customElements.define(CrToastElement.is, CrToastElement);

let instance$f = null;
function getCss$c() {
    return instance$f || (instance$f = [...[getCss$o()], css `#content{display:flex;flex:1}.collapsible{overflow:hidden;text-overflow:ellipsis}span{white-space:pre}.elided-text{overflow:hidden;text-overflow:ellipsis;white-space:nowrap}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$6() {
    return html `
<cr-toast id="toast" .duration="${this.duration}">
  <div id="content" class="elided-text"></div>
  <slot id="slotted"></slot>
</cr-toast>`;
}

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/** @fileoverview Element which shows toasts with optional undo button. */
let toastManagerInstance = null;
function getToastManager() {
    assert(toastManagerInstance);
    return toastManagerInstance;
}
function setInstance(instance) {
    assert(!instance || !toastManagerInstance);
    toastManagerInstance = instance;
}
class CrToastManagerElement extends CrLitElement {
    static get is() {
        return 'cr-toast-manager';
    }
    static get styles() {
        return getCss$c();
    }
    render() {
        return getHtml$6.bind(this)();
    }
    static get properties() {
        return {
            duration: {
                type: Number,
            },
        };
    }
    #duration_accessor_storage = 0;
    get duration() { return this.#duration_accessor_storage; }
    set duration(value) { this.#duration_accessor_storage = value; }
    get isToastOpen() {
        return this.$.toast.open;
    }
    get slottedHidden() {
        return this.$.slotted.hidden;
    }
    connectedCallback() {
        super.connectedCallback();
        setInstance(this);
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        setInstance(null);
    }
    /**
     * @param label The label to display inside the toast.
     */
    show(label, hideSlotted = false) {
        this.$.content.textContent = label;
        this.showInternal_(hideSlotted);
    }
    /**
     * Shows the toast, making certain text fragments collapsible.
     */
    showForStringPieces(pieces, hideSlotted = false) {
        const content = this.$.content;
        content.textContent = '';
        pieces.forEach(function (p) {
            if (p.value.length === 0) {
                return;
            }
            const span = document.createElement('span');
            span.textContent = p.value;
            if (p.collapsible) {
                span.classList.add('collapsible');
            }
            content.appendChild(span);
        });
        this.showInternal_(hideSlotted);
    }
    showInternal_(hideSlotted) {
        this.$.slotted.hidden = hideSlotted;
        this.$.toast.show();
    }
    hide() {
        this.$.toast.hide();
    }
}
customElements.define(CrToastManagerElement.is, CrToastManagerElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class FocusRowMixinDelegate {
    listItem_;
    constructor(listItem) {
        this.listItem_ = listItem;
    }
    /**
     * This function gets called when the [focus-row-control] element receives
     * the focus event.
     */
    onFocus(_row, e) {
        const element = e.composedPath()[0];
        const focusableElement = FocusRow.getFocusableElement(element);
        if (element !== focusableElement) {
            focusableElement.focus();
        }
        this.listItem_.lastFocused = focusableElement;
    }
    /**
     * @param row The row that detected a keydown.
     * @return Whether the event was handled.
     */
    onKeydown(_row, e) {
        // Prevent iron-list from changing the focus on enter.
        if (e.key === 'Enter') {
            e.stopPropagation();
        }
        return false;
    }
    getCustomEquivalent(sampleElement) {
        return this.listItem_.overrideCustomEquivalent ?
            this.listItem_.getCustomEquivalent(sampleElement) :
            null;
    }
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const FocusRowMixinLit = (superClass) => {
    class FocusRowMixinLit extends superClass {
        static get properties() {
            return {
                row_: { type: Object },
                mouseFocused_: { type: Boolean },
                // Will be updated when |index| is set, unless specified elsewhere.
                id: {
                    type: String,
                    reflect: true,
                },
                isFocused: {
                    type: Boolean,
                    notify: true,
                },
                focusRowIndex: { type: Number },
                lastFocused: {
                    type: Object,
                    notify: true,
                },
                listTabIndex: { type: Number },
                listBlurred: {
                    type: Boolean,
                    notify: true,
                },
            };
        }
        #row__accessor_storage = null;
        get row_() { return this.#row__accessor_storage; }
        set row_(value) { this.#row__accessor_storage = value; }
        #mouseFocused__accessor_storage = false;
        get mouseFocused_() { return this.#mouseFocused__accessor_storage; }
        set mouseFocused_(value) { this.#mouseFocused__accessor_storage = value; }
        #isFocused_accessor_storage = false;
        // For notifying when the row is in focus.
        get isFocused() { return this.#isFocused_accessor_storage; }
        set isFocused(value) { this.#isFocused_accessor_storage = value; }
        #focusRowIndex_accessor_storage;
        // Should be bound to the index of the item from the iron-list or
        // infinite-list.
        get focusRowIndex() { return this.#focusRowIndex_accessor_storage; }
        set focusRowIndex(value) { this.#focusRowIndex_accessor_storage = value; }
        #lastFocused_accessor_storage = null;
        get lastFocused() { return this.#lastFocused_accessor_storage; }
        set lastFocused(value) { this.#lastFocused_accessor_storage = value; }
        #listTabIndex_accessor_storage;
        /**
         * This is different from tabIndex, since the template only does a
         * one-way binding on both attributes, and the mixin makes use of this.
         * For example, when a control within a row is focused, it will have
         * tabIndex = -1 and listTabIndex = 0.
         */
        get listTabIndex() { return this.#listTabIndex_accessor_storage; }
        set listTabIndex(value) { this.#listTabIndex_accessor_storage = value; }
        #listBlurred_accessor_storage = false;
        get listBlurred() { return this.#listBlurred_accessor_storage; }
        set listBlurred(value) { this.#listBlurred_accessor_storage = value; }
        firstControl_ = null;
        controlObservers_ = [];
        subtreeObserver_ = null;
        boundOnFirstControlKeydown_ = null;
        connectedCallback() {
            super.connectedCallback();
            this.classList.add('no-outline');
            this.boundOnFirstControlKeydown_ = this.onFirstControlKeydown_.bind(this);
            this.updateComplete.then(() => {
                const rowContainer = this.shadowRoot.querySelector('[focus-row-container]');
                assert(rowContainer);
                this.row_ =
                    new VirtualFocusRow(rowContainer, new FocusRowMixinDelegate(this));
                this.addItems_();
                // Adding listeners asynchronously to reduce blocking time, since
                // this behavior will be used by items in potentially long lists.
                this.addEventListener('focus', this.onFocus_);
                this.subtreeObserver_ = new MutationObserver(() => this.addItems_());
                this.subtreeObserver_.observe(this.shadowRoot, { childList: true, subtree: true });
                this.addEventListener('mousedown', this.onMouseDown_);
                this.addEventListener('blur', this.onBlur_);
            });
        }
        disconnectedCallback() {
            super.disconnectedCallback();
            this.removeEventListener('focus', this.onFocus_);
            if (this.subtreeObserver_) {
                this.subtreeObserver_.disconnect();
                this.subtreeObserver_ = null;
            }
            this.removeEventListener('mousedown', this.onMouseDown_);
            this.removeEventListener('blur', this.onBlur_);
            this.removeObservers_();
            if (this.firstControl_ && this.boundOnFirstControlKeydown_) {
                this.firstControl_.removeEventListener('keydown', this.boundOnFirstControlKeydown_);
                this.boundOnFirstControlKeydown_ = null;
            }
            if (this.row_) {
                this.row_.destroy();
            }
        }
        willUpdate(changedProperties) {
            super.willUpdate(changedProperties);
            if (changedProperties.has('focusRowIndex') &&
                this.focusRowIndex !== undefined) {
                // focusRowIndex is 0-based where aria-rowindex is 1-based.
                this.setAttribute('aria-rowindex', (this.focusRowIndex + 1).toString());
                // Only set ID if it matches what was previously set. This prevents
                // overriding the ID value if it's set elsewhere.
                const oldIndex = changedProperties.get('focusRowIndex');
                if (this.id === this.computeId_(oldIndex)) {
                    this.id = this.computeId_(this.focusRowIndex) || '';
                }
            }
        }
        updated(changedProperties) {
            super.updated(changedProperties);
            if (changedProperties.has('listTabIndex')) {
                this.listTabIndexChanged_();
            }
        }
        /**
         * Returns an ID based on the index that was passed in.
         */
        computeId_(index) {
            return index !== undefined ? `frb${index}` : undefined;
        }
        getFocusRow() {
            assert(this.row_);
            return this.row_;
        }
        updateFirstControl_() {
            assert(this.row_);
            const newFirstControl = this.row_.getFirstFocusable();
            if (newFirstControl === this.firstControl_) {
                return;
            }
            if (this.firstControl_) {
                this.firstControl_.removeEventListener('keydown', this.boundOnFirstControlKeydown_);
            }
            this.firstControl_ = newFirstControl;
            if (this.firstControl_) {
                this.firstControl_.addEventListener('keydown', this.boundOnFirstControlKeydown_);
            }
        }
        removeObservers_() {
            if (this.controlObservers_.length > 0) {
                this.controlObservers_.forEach(observer => {
                    observer.disconnect();
                });
            }
            this.controlObservers_ = [];
        }
        addItems_() {
            this.listTabIndexChanged_();
            if (this.row_) {
                this.removeObservers_();
                this.row_.destroy();
                const controls = this.shadowRoot.querySelectorAll('[focus-row-control]');
                controls.forEach(control => {
                    assert(control);
                    assert(this.row_);
                    this.row_.addItem(control.getAttribute('focus-type'), FocusRow.getFocusableElement(control));
                    this.addMutationObservers_(control);
                });
                this.updateFirstControl_();
            }
        }
        createObserver_() {
            return new MutationObserver(mutations => {
                const mutation = mutations[0];
                if (mutation.attributeName === 'style' && mutation.oldValue) {
                    const newStyle = window.getComputedStyle(mutation.target);
                    const oldDisplayValue = mutation.oldValue.match(/^display:(.*)(?=;)/);
                    const oldVisibilityValue = mutation.oldValue.match(/^visibility:(.*)(?=;)/);
                    // Return early if display and visibility have not changed.
                    if (oldDisplayValue &&
                        newStyle.display === oldDisplayValue[1].trim() &&
                        oldVisibilityValue &&
                        newStyle.visibility === oldVisibilityValue[1].trim()) {
                        return;
                    }
                }
                this.updateFirstControl_();
            });
        }
        /**
         * The first focusable control changes if hidden, disabled, or
         * style.display changes for the control or any of its ancestors. Add
         * mutation observers to watch for these changes in order to ensure the
         * first control keydown listener is always on the correct element.
         */
        addMutationObservers_(control) {
            let current = control;
            while (current && current !== this.shadowRoot) {
                const currentObserver = this.createObserver_();
                currentObserver.observe(current, {
                    attributes: true,
                    attributeFilter: ['hidden', 'disabled', 'style'],
                    attributeOldValue: true,
                });
                this.controlObservers_.push(currentObserver);
                current = current.parentNode;
            }
        }
        /**
         * This function gets called when the row itself receives the focus
         * event.
         */
        onFocus_(e) {
            if (this.mouseFocused_) {
                this.mouseFocused_ = false; // Consume and reset flag.
                return;
            }
            // If focus is being restored from outside the item and the event is
            // fired by the list item itself, focus the first control so that the
            // user can tab through all the controls. When the user shift-tabs
            // back to the row, or focus is restored to the row from a dropdown on
            // the last item, the last child item will be focused before the row
            // itself. Since this is the desired behavior, do not shift focus to
            // the first item in these cases.
            const restoreFocusToFirst = this.listBlurred && e.composedPath()[0] === this;
            if (this.lastFocused && !restoreFocusToFirst) {
                assert(this.row_);
                focusWithoutInk(this.row_.getEquivalentElement(this.lastFocused));
            }
            else {
                assert(this.firstControl_);
                const firstFocusable = this.firstControl_;
                focusWithoutInk(firstFocusable);
            }
            this.listBlurred = false;
            this.isFocused = true;
        }
        onFirstControlKeydown_(e) {
            const keyEvent = e;
            if (keyEvent.shiftKey && keyEvent.key === 'Tab') {
                this.focus();
            }
        }
        listTabIndexChanged_() {
            if (this.row_) {
                this.row_.makeActive(this.listTabIndex === 0);
            }
            // If a new row is being focused, reset listBlurred. This means an
            // item has been removed and iron-list is about to focus the next
            // item.
            if (this.listTabIndex === 0) {
                this.listBlurred = false;
            }
        }
        onMouseDown_() {
            this.mouseFocused_ = true; // Set flag to not do any control-focusing.
        }
        onBlur_(e) {
            // Reset focused flags since it's not active anymore.
            this.mouseFocused_ = false;
            this.isFocused = false;
            const node = e.relatedTarget ? e.relatedTarget : null;
            if (!this.parentNode.contains(node)) {
                this.listBlurred = true;
            }
        }
    }
    return FocusRowMixinLit;
};

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Make a string safe for Polymer bindings that are inner-h-t-m-l or other
 * innerHTML use.
 * @param rawString The unsanitized string
 * @param opts Optional additional allowed tags and attributes.
 */
function sanitizeInnerHtmlInternal(rawString, opts) {
    opts = opts || {};
    const html = parseHtmlSubset(`<b>${rawString}</b>`, opts.tags, opts.attrs)
        .firstElementChild;
    return html.innerHTML;
}
// 
let sanitizedPolicy = null;
/**
 * Same as |sanitizeInnerHtmlInternal|, but it passes through sanitizedPolicy
 * to create a TrustedHTML.
 */
function sanitizeInnerHtml(rawString, opts) {
    assert(window.trustedTypes);
    if (sanitizedPolicy === null) {
        // Initialize |sanitizedPolicy| lazily.
        sanitizedPolicy = window.trustedTypes.createPolicy('sanitize-inner-html', {
            createHTML: sanitizeInnerHtmlInternal,
            createScript: () => assertNotReached(),
            createScriptURL: () => assertNotReached(),
        });
    }
    return sanitizedPolicy.createHTML(rawString, opts);
}
const allowAttribute = (_node, _value) => true;
/** Allow-list of attributes in parseHtmlSubset. */
const allowedAttributes = new Map([
    [
        'href',
        (node, value) => {
            // Only allow a[href] starting with chrome:// or https:// or equaling
            // to #.
            return node.tagName === 'A' &&
                (value.startsWith('chrome://') || value.startsWith('https://') ||
                    value === '#');
        },
    ],
    [
        'target',
        (node, value) => {
            // Only allow a[target='_blank'].
            // TODO(dbeam): are there valid use cases for target !== '_blank'?
            return node.tagName === 'A' && value === '_blank';
        },
    ],
]);
/** Allow-list of optional attributes in parseHtmlSubset. */
const allowedOptionalAttributes = new Map([
    ['class', allowAttribute],
    ['id', allowAttribute],
    ['is', (_node, value) => value === 'action-link' || value === ''],
    ['role', (_node, value) => value === 'link'],
    [
        'src',
        (node, value) => {
            // Only allow img[src] starting with chrome://
            return node.tagName === 'IMG' &&
                value.startsWith('chrome://');
        },
    ],
    ['tabindex', allowAttribute],
    ['aria-description', allowAttribute],
    ['aria-hidden', allowAttribute],
    ['aria-label', allowAttribute],
    ['aria-labelledby', allowAttribute],
]);
/** Allow-list of tag names in parseHtmlSubset. */
const allowedTags = new Set(['A', 'B', 'I', 'BR', 'DIV', 'EM', 'KBD', 'P', 'PRE', 'SPAN', 'STRONG']);
/** Allow-list of optional tag names in parseHtmlSubset. */
const allowedOptionalTags = new Set(['IMG', 'LI', 'UL']);
/**
 * This policy maps a given string to a `TrustedHTML` object
 * without performing any validation. Callsites must ensure
 * that the resulting object will only be used in inert
 * documents. Initialized lazily.
 */
let unsanitizedPolicy;
/**
 * @param optTags an Array to merge.
 * @return Set of allowed tags.
 */
function mergeTags(optTags) {
    const clone = new Set(allowedTags);
    optTags.forEach(str => {
        const tag = str.toUpperCase();
        if (allowedOptionalTags.has(tag)) {
            clone.add(tag);
        }
    });
    return clone;
}
/**
 * @param optAttrs an Array to merge.
 * @return Map of allowed attributes.
 */
function mergeAttrs(optAttrs) {
    const clone = new Map(allowedAttributes);
    optAttrs.forEach(key => {
        if (allowedOptionalAttributes.has(key)) {
            clone.set(key, allowedOptionalAttributes.get(key));
        }
    });
    return clone;
}
function walk(n, f) {
    f(n);
    for (let i = 0; i < n.childNodes.length; i++) {
        walk(n.childNodes[i], f);
    }
}
function assertElement(tags, node) {
    if (!tags.has(node.tagName)) {
        throw Error(node.tagName + ' is not supported');
    }
}
function assertAttribute(attrs, attrNode, node) {
    const n = attrNode.nodeName;
    const v = attrNode.nodeValue || '';
    if (!attrs.has(n) || !attrs.get(n)(node, v)) {
        throw Error(node.tagName + '[' + n + '="' + v +
            '"] is not supported');
    }
}
/**
 * Parses a very small subset of HTML. This ensures that insecure HTML /
 * javascript cannot be injected into WebUI.
 * @param s The string to parse.
 * @param extraTags Optional extra allowed tags.
 * @param extraAttrs
 *     Optional extra allowed attributes (all tags are run through these).
 * @throws an Error in case of non supported markup.
 * @return A document fragment containing the DOM tree.
 */
function parseHtmlSubset(s, extraTags, extraAttrs) {
    const tags = extraTags ? mergeTags(extraTags) : allowedTags;
    const attrs = extraAttrs ? mergeAttrs(extraAttrs) : allowedAttributes;
    const doc = document.implementation.createHTMLDocument('');
    const r = doc.createRange();
    r.selectNode(doc.body);
    if (window.trustedTypes) {
        if (!unsanitizedPolicy) {
            unsanitizedPolicy =
                window.trustedTypes.createPolicy('parse-html-subset', {
                    createHTML: (untrustedHTML) => untrustedHTML,
                    createScript: () => assertNotReached(),
                    createScriptURL: () => assertNotReached(),
                });
        }
        s = unsanitizedPolicy.createHTML(s);
    }
    // This does not execute any scripts because the document has no view.
    const df = r.createContextualFragment(s);
    walk(df, function (node) {
        switch (node.nodeType) {
            case Node.ELEMENT_NODE:
                assertElement(tags, node);
                const nodeAttrs = node.attributes;
                for (let i = 0; i < nodeAttrs.length; ++i) {
                    assertAttribute(attrs, nodeAttrs[i], node);
                }
                break;
            case Node.COMMENT_NODE:
            case Node.DOCUMENT_FRAGMENT_NODE:
            case Node.TEXT_NODE:
                break;
            default:
                throw Error('Node type ' + node.nodeType + ' is not supported');
        }
    });
    return df;
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * 'I18nMixinLit' is a Mixin offering loading of internationalization
 * strings. Typically it is used as [[i18n('someString')]] computed bindings or
 * for this.i18n('foo'). It is not needed for HTML $i18n{otherString}, which is
 * handled by a C++ templatizer.
 */
const I18nMixinLit = (superClass) => {
    class I18nMixinLit extends superClass {
        /**
         * Returns a translated string where $1 to $9 are replaced by the given
         * values.
         * @param id The ID of the string to translate.
         * @param varArgs Values to replace the placeholders $1 to $9 in the
         *     string.
         * @return A translated, substituted string.
         */
        i18nRaw_(id, ...varArgs) {
            return varArgs.length === 0 ? loadTimeData.getString(id) :
                loadTimeData.getStringF(id, ...varArgs);
        }
        /**
         * Returns a translated string where $1 to $9 are replaced by the given
         * values. Also sanitizes the output to filter out dangerous HTML/JS.
         * Use with Lit bindings that are *not* innerHTML.
         * NOTE: This is not related to $i18n{foo} in HTML, see file overview.
         * @param id The ID of the string to translate.
         * @param varArgs Values to replace the placeholders $1 to $9 in the
         *     string.
         * @return A translated, sanitized, substituted string.
         */
        i18n(id, ...varArgs) {
            const rawString = this.i18nRaw_(id, ...varArgs);
            return parseHtmlSubset(`<b>${rawString}</b>`).firstChild.textContent;
        }
        /**
         * Similar to 'i18n', returns a translated, sanitized, substituted
         * string. It receives the string ID and a dictionary containing the
         * substitutions as well as optional additional allowed tags and
         * attributes. Use with Lit bindings that are innerHTML.
         * @param id The ID of the string to translate.
         */
        i18nAdvanced(id, opts) {
            opts = opts || {};
            const rawString = this.i18nRaw_(id, ...(opts.substitutions || []));
            return sanitizeInnerHtml(rawString, opts);
        }
        /**
         * Similar to 'i18n', with an unused |locale| parameter used to trigger
         * updates when the locale changes.
         * @param locale The UI language used.
         * @param id The ID of the string to translate.
         * @param varArgs Values to replace the placeholders $1 to $9 in the
         *     string.
         * @return A translated, sanitized, substituted string.
         */
        i18nDynamic(_locale, id, ...varArgs) {
            return this.i18n(id, ...varArgs);
        }
        /**
         * Similar to 'i18nDynamic', but varArgs values are interpreted as keys
         * in loadTimeData. This allows generation of strings that take other
         * localized strings as parameters.
         * @param locale The UI language used.
         * @param id The ID of the string to translate.
         * @param varArgs Values to replace the placeholders $1 to $9
         *     in the string. Values are interpreted as strings IDs if found in
         * the list of localized strings.
         * @return A translated, sanitized, substituted string.
         */
        i18nRecursive(locale, id, ...varArgs) {
            let args = varArgs;
            if (args.length > 0) {
                // Try to replace IDs with localized values.
                args = args.map(str => {
                    return this.i18nExists(str) ? loadTimeData.getString(str) : str;
                });
            }
            return this.i18nDynamic(locale, id, ...args);
        }
        /**
         * Returns true if a translation exists for |id|.
         */
        i18nExists(id) {
            return loadTimeData.valueExists(id);
        }
    }
    return I18nMixinLit;
};

// mojo/public/mojom/base/big_buffer.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const BigBufferSharedMemoryRegionSpec = { $: {} };
const BigBufferSpec = { $: {} };
mojo.internal.Struct(BigBufferSharedMemoryRegionSpec.$, 'BigBufferSharedMemoryRegion', [
    mojo.internal.StructField('bufferHandle', 0, 0, mojo.internal.Handle, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('size', 4, 0, mojo.internal.Uint32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Union(BigBufferSpec.$, 'BigBuffer', {
    'bytes': {
        'ordinal': 0,
        'type': mojo.internal.Array(mojo.internal.Uint8, false),
    },
    'sharedMemory': {
        'ordinal': 1,
        'type': BigBufferSharedMemoryRegionSpec.$,
    },
    'invalidBuffer': {
        'ordinal': 2,
        'type': mojo.internal.Bool,
    },
});
var BigBufferFieldTags;
(function (BigBufferFieldTags) {
    BigBufferFieldTags[BigBufferFieldTags["BYTES"] = 0] = "BYTES";
    BigBufferFieldTags[BigBufferFieldTags["SHARED_MEMORY"] = 1] = "SHARED_MEMORY";
    BigBufferFieldTags[BigBufferFieldTags["INVALID_BUFFER"] = 2] = "INVALID_BUFFER";
})(BigBufferFieldTags || (BigBufferFieldTags = {}));

// mojo/public/mojom/base/string16.mojom-converters.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class String16DataView {
    decoder_;
    version_;
    fieldSpecs_;
    constructor(decoder, version, fieldSpecs) {
        this.decoder_ = decoder;
        this.version_ = version;
        this.fieldSpecs_ = fieldSpecs;
    }
    get data() {
        const field = this.fieldSpecs_[0];
        return mojo.internal.decodeStructField(this.decoder_, field, this.version_);
    }
}

class String16Converter {
    data(str) {
        const arr = [];
        for (let i = 0; i < str.length; ++i) {
            arr.push(str.charCodeAt(i));
        }
        return arr;
    }
    convert(view) {
        const data = view.data;
        return this.convertImpl(data);
    }
    // Exported for testing.
    // TODO(crbug.com/448737199): we should not have to expose a separate method
    // just to facilitate testing.
    convertImpl(data) {
        // Taken from chunk size used in goog.crypt.byteArrayToBinaryString in
        // Closure Library. The value is equal to 2^13.
        const CHUNK_SIZE = 8192;
        if (data.length < CHUNK_SIZE) {
            return String.fromCharCode(...data);
        }
        // Convert the array to a string in chunks, to avoid passing too many
        // arguments to String.fromCharCode() at once, which can exceed the max call
        // stack size (c.f. crbug.com/1509792).
        let str = '';
        for (let i = 0; i < data.length; i += CHUNK_SIZE) {
            const chunk = data.slice(i, i + CHUNK_SIZE);
            str += String.fromCharCode(...chunk);
        }
        return str;
    }
}

// mojo/public/mojom/base/string16.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const String16Spec = { $: {} };
const BigString16Spec = { $: {} };
const converterForString16 = new String16Converter();
mojo.internal.TypemappedStruct(String16Spec.$, 'String16', String16DataView, converterForString16, [
    mojo.internal.StructField('data', 0, 0, mojo.internal.Array(mojo.internal.Uint16, false), null, false /* nullable */, 0, undefined, ((value) => converterForString16.data(value))),
], [[0, 16],]);
mojo.internal.Struct(BigString16Spec.$, 'BigString16', [
    mojo.internal.StructField('data', 0, 0, BigBufferSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);

// url/mojom/url.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const UrlSpec = { $: {} };
mojo.internal.Struct(UrlSpec.$, 'Url', [
    mojo.internal.StructField('url', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);

// chrome/browser/ui/webui/downloads/downloads.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const DangerTypeSpec = { $: mojo.internal.Enum() };
var DangerType;
(function (DangerType) {
    DangerType[DangerType["MIN_VALUE"] = 0] = "MIN_VALUE";
    DangerType[DangerType["MAX_VALUE"] = 17] = "MAX_VALUE";
    DangerType[DangerType["kNoApplicableDangerType"] = 0] = "kNoApplicableDangerType";
    DangerType[DangerType["kDangerousFile"] = 1] = "kDangerousFile";
    DangerType[DangerType["kDangerousUrl"] = 2] = "kDangerousUrl";
    DangerType[DangerType["kDangerousContent"] = 3] = "kDangerousContent";
    DangerType[DangerType["kCookieTheft"] = 4] = "kCookieTheft";
    DangerType[DangerType["kUncommonContent"] = 5] = "kUncommonContent";
    DangerType[DangerType["kDangerousHost"] = 6] = "kDangerousHost";
    DangerType[DangerType["kPotentiallyUnwanted"] = 7] = "kPotentiallyUnwanted";
    DangerType[DangerType["kAsyncScanning"] = 8] = "kAsyncScanning";
    DangerType[DangerType["kAsyncLocalPasswordScanning"] = 9] = "kAsyncLocalPasswordScanning";
    DangerType[DangerType["kBlockedPasswordProtected"] = 10] = "kBlockedPasswordProtected";
    DangerType[DangerType["kBlockedTooLarge"] = 11] = "kBlockedTooLarge";
    DangerType[DangerType["kSensitiveContentWarning"] = 12] = "kSensitiveContentWarning";
    DangerType[DangerType["kSensitiveContentBlock"] = 13] = "kSensitiveContentBlock";
    DangerType[DangerType["kDeepScannedFailed"] = 14] = "kDeepScannedFailed";
    DangerType[DangerType["kDeepScannedSafe"] = 15] = "kDeepScannedSafe";
    DangerType[DangerType["kDeepScannedOpenedDangerous"] = 16] = "kDeepScannedOpenedDangerous";
    DangerType[DangerType["kBlockedScanFailed"] = 17] = "kBlockedScanFailed";
})(DangerType || (DangerType = {}));
const TailoredWarningTypeSpec = { $: mojo.internal.Enum() };
var TailoredWarningType;
(function (TailoredWarningType) {
    TailoredWarningType[TailoredWarningType["MIN_VALUE"] = 0] = "MIN_VALUE";
    TailoredWarningType[TailoredWarningType["MAX_VALUE"] = 2] = "MAX_VALUE";
    TailoredWarningType[TailoredWarningType["kNoApplicableTailoredWarningType"] = 0] = "kNoApplicableTailoredWarningType";
    TailoredWarningType[TailoredWarningType["kCookieTheft"] = 1] = "kCookieTheft";
    TailoredWarningType[TailoredWarningType["kSuspiciousArchive"] = 2] = "kSuspiciousArchive";
})(TailoredWarningType || (TailoredWarningType = {}));
const StateSpec = { $: mojo.internal.Enum() };
var State;
(function (State) {
    State[State["MIN_VALUE"] = 0] = "MIN_VALUE";
    State[State["MAX_VALUE"] = 9] = "MAX_VALUE";
    State[State["kInProgress"] = 0] = "kInProgress";
    State[State["kCancelled"] = 1] = "kCancelled";
    State[State["kComplete"] = 2] = "kComplete";
    State[State["kPaused"] = 3] = "kPaused";
    State[State["kDangerous"] = 4] = "kDangerous";
    State[State["kInterrupted"] = 5] = "kInterrupted";
    State[State["kInsecure"] = 6] = "kInsecure";
    State[State["kAsyncScanning"] = 7] = "kAsyncScanning";
    State[State["kPromptForScanning"] = 8] = "kPromptForScanning";
    State[State["kPromptForLocalPasswordScanning"] = 9] = "kPromptForLocalPasswordScanning";
})(State || (State = {}));
const SafeBrowsingStateSpec = { $: mojo.internal.Enum() };
var SafeBrowsingState;
(function (SafeBrowsingState) {
    SafeBrowsingState[SafeBrowsingState["MIN_VALUE"] = 0] = "MIN_VALUE";
    SafeBrowsingState[SafeBrowsingState["MAX_VALUE"] = 2] = "MAX_VALUE";
    SafeBrowsingState[SafeBrowsingState["kNoSafeBrowsing"] = 0] = "kNoSafeBrowsing";
    SafeBrowsingState[SafeBrowsingState["kStandardProtection"] = 1] = "kStandardProtection";
    SafeBrowsingState[SafeBrowsingState["kEnhancedProtection"] = 2] = "kEnhancedProtection";
})(SafeBrowsingState || (SafeBrowsingState = {}));
class PageHandlerFactoryPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'downloads.mojom.PageHandlerFactory', scope);
    }
}
class PageHandlerFactoryRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(PageHandlerFactoryPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    createPageHandler(page, handler) {
        this.proxy.sendMessage(0, PageHandlerFactory_CreatePageHandler_ParamsSpec.$, null, [
            page,
            handler
        ], false);
    }
}
class PageHandlerFactory {
    static get $interfaceName() {
        return "downloads.mojom.PageHandlerFactory";
    }
    /**
     * Returns a remote for this interface which sends messages to the browser.
     * The browser must have an interface request binder registered for this
     * interface and accessible to the calling document's frame.
     */
    static getRemote() {
        let remote = new PageHandlerFactoryRemote;
        remote.$.bindNewPipeAndPassReceiver().bindInBrowser();
        return remote;
    }
}
class PageHandlerPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'downloads.mojom.PageHandler', scope);
    }
}
class PageHandlerRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(PageHandlerPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    getDownloads(searchTerms) {
        this.proxy.sendMessage(0, PageHandler_GetDownloads_ParamsSpec.$, null, [
            searchTerms
        ], false);
    }
    openFileRequiringGesture(id) {
        this.proxy.sendMessage(1, PageHandler_OpenFileRequiringGesture_ParamsSpec.$, null, [
            id
        ], false);
    }
    drag(id) {
        this.proxy.sendMessage(2, PageHandler_Drag_ParamsSpec.$, null, [
            id
        ], false);
    }
    saveSuspiciousRequiringGesture(id) {
        this.proxy.sendMessage(3, PageHandler_SaveSuspiciousRequiringGesture_ParamsSpec.$, null, [
            id
        ], false);
    }
    recordOpenBypassWarningDialog(id) {
        this.proxy.sendMessage(4, PageHandler_RecordOpenBypassWarningDialog_ParamsSpec.$, null, [
            id
        ], false);
    }
    saveDangerousFromDialogRequiringGesture(id) {
        this.proxy.sendMessage(5, PageHandler_SaveDangerousFromDialogRequiringGesture_ParamsSpec.$, null, [
            id
        ], false);
    }
    recordCancelBypassWarningDialog(id) {
        this.proxy.sendMessage(6, PageHandler_RecordCancelBypassWarningDialog_ParamsSpec.$, null, [
            id
        ], false);
    }
    discardDangerous(id) {
        this.proxy.sendMessage(7, PageHandler_DiscardDangerous_ParamsSpec.$, null, [
            id
        ], false);
    }
    retryDownload(id) {
        this.proxy.sendMessage(8, PageHandler_RetryDownload_ParamsSpec.$, null, [
            id
        ], false);
    }
    show(id) {
        this.proxy.sendMessage(9, PageHandler_Show_ParamsSpec.$, null, [
            id
        ], false);
    }
    pause(id) {
        this.proxy.sendMessage(10, PageHandler_Pause_ParamsSpec.$, null, [
            id
        ], false);
    }
    resume(id) {
        this.proxy.sendMessage(11, PageHandler_Resume_ParamsSpec.$, null, [
            id
        ], false);
    }
    remove(id) {
        this.proxy.sendMessage(12, PageHandler_Remove_ParamsSpec.$, null, [
            id
        ], false);
    }
    undo() {
        this.proxy.sendMessage(13, PageHandler_Undo_ParamsSpec.$, null, [], false);
    }
    cancel(id) {
        this.proxy.sendMessage(14, PageHandler_Cancel_ParamsSpec.$, null, [
            id
        ], false);
    }
    clearAll() {
        this.proxy.sendMessage(15, PageHandler_ClearAll_ParamsSpec.$, null, [], false);
    }
    openDownloadsFolderRequiringGesture() {
        this.proxy.sendMessage(16, PageHandler_OpenDownloadsFolderRequiringGesture_ParamsSpec.$, null, [], false);
    }
    openEsbSettings() {
        this.proxy.sendMessage(17, PageHandler_OpenEsbSettings_ParamsSpec.$, null, [], false);
    }
    logEsbPromotionRowViewed() {
        this.proxy.sendMessage(18, PageHandler_LogEsbPromotionRowViewed_ParamsSpec.$, null, [], false);
    }
    openDuringScanningRequiringGesture(id) {
        this.proxy.sendMessage(19, PageHandler_OpenDuringScanningRequiringGesture_ParamsSpec.$, null, [
            id
        ], false);
    }
    reviewDangerousRequiringGesture(id) {
        this.proxy.sendMessage(20, PageHandler_ReviewDangerousRequiringGesture_ParamsSpec.$, null, [
            id
        ], false);
    }
    deepScan(id) {
        this.proxy.sendMessage(21, PageHandler_DeepScan_ParamsSpec.$, null, [
            id
        ], false);
    }
    bypassDeepScanRequiringGesture(id) {
        this.proxy.sendMessage(22, PageHandler_BypassDeepScanRequiringGesture_ParamsSpec.$, null, [
            id
        ], false);
    }
    isEligibleForEsbPromo() {
        return this.proxy.sendMessage(23, PageHandler_IsEligibleForEsbPromo_ParamsSpec.$, PageHandler_IsEligibleForEsbPromo_ResponseParamsSpec.$, [], false);
    }
}
class PagePendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'downloads.mojom.Page', scope);
    }
}
class PageRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(PagePendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    removeItem(index) {
        this.proxy.sendMessage(0, Page_RemoveItem_ParamsSpec.$, null, [
            index
        ], false);
    }
    updateItem(index, data) {
        this.proxy.sendMessage(1, Page_UpdateItem_ParamsSpec.$, null, [
            index,
            data
        ], false);
    }
    insertItems(index, items) {
        this.proxy.sendMessage(2, Page_InsertItems_ParamsSpec.$, null, [
            index,
            items
        ], false);
    }
    clearAll() {
        this.proxy.sendMessage(3, Page_ClearAll_ParamsSpec.$, null, [], false);
    }
}
/**
 * An object which receives request messages for the Page
 * mojom interface and dispatches them as callbacks. One callback receiver exists
 * on this object for each message defined in the mojom interface, and each
 * receiver can have any number of listeners added to it.
 */
class PageCallbackRouter {
    helper_internal_;
    $;
    router_;
    removeItem;
    updateItem;
    insertItems;
    clearAll;
    onConnectionError;
    constructor() {
        this.helper_internal_ = new mojo.internal.interfaceSupport.InterfaceReceiverHelperInternal(PageRemote);
        this.$ = new mojo.internal.interfaceSupport.InterfaceReceiverHelper(this.helper_internal_);
        this.router_ = new mojo.internal.interfaceSupport.CallbackRouter;
        this.removeItem =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(0, Page_RemoveItem_ParamsSpec.$, null, this.removeItem.createReceiverHandler(false /* expectsResponse */), false);
        this.updateItem =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(1, Page_UpdateItem_ParamsSpec.$, null, this.updateItem.createReceiverHandler(false /* expectsResponse */), false);
        this.insertItems =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(2, Page_InsertItems_ParamsSpec.$, null, this.insertItems.createReceiverHandler(false /* expectsResponse */), false);
        this.clearAll =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(3, Page_ClearAll_ParamsSpec.$, null, this.clearAll.createReceiverHandler(false /* expectsResponse */), false);
        this.onConnectionError = this.helper_internal_.getConnectionErrorEventRouter();
    }
    /**
     * @param id An ID returned by a prior call to addListener.
     * @return True iff the identified listener was found and removed.
     */
    removeListener(id) {
        return this.router_.removeListener(id);
    }
}
const DataSpec = { $: {} };
const PageHandlerFactory_CreatePageHandler_ParamsSpec = { $: {} };
const PageHandler_GetDownloads_ParamsSpec = { $: {} };
const PageHandler_OpenFileRequiringGesture_ParamsSpec = { $: {} };
const PageHandler_Drag_ParamsSpec = { $: {} };
const PageHandler_SaveSuspiciousRequiringGesture_ParamsSpec = { $: {} };
const PageHandler_RecordOpenBypassWarningDialog_ParamsSpec = { $: {} };
const PageHandler_SaveDangerousFromDialogRequiringGesture_ParamsSpec = { $: {} };
const PageHandler_RecordCancelBypassWarningDialog_ParamsSpec = { $: {} };
const PageHandler_DiscardDangerous_ParamsSpec = { $: {} };
const PageHandler_RetryDownload_ParamsSpec = { $: {} };
const PageHandler_Show_ParamsSpec = { $: {} };
const PageHandler_Pause_ParamsSpec = { $: {} };
const PageHandler_Resume_ParamsSpec = { $: {} };
const PageHandler_Remove_ParamsSpec = { $: {} };
const PageHandler_Undo_ParamsSpec = { $: {} };
const PageHandler_Cancel_ParamsSpec = { $: {} };
const PageHandler_ClearAll_ParamsSpec = { $: {} };
const PageHandler_OpenDownloadsFolderRequiringGesture_ParamsSpec = { $: {} };
const PageHandler_OpenEsbSettings_ParamsSpec = { $: {} };
const PageHandler_LogEsbPromotionRowViewed_ParamsSpec = { $: {} };
const PageHandler_OpenDuringScanningRequiringGesture_ParamsSpec = { $: {} };
const PageHandler_ReviewDangerousRequiringGesture_ParamsSpec = { $: {} };
const PageHandler_DeepScan_ParamsSpec = { $: {} };
const PageHandler_BypassDeepScanRequiringGesture_ParamsSpec = { $: {} };
const PageHandler_IsEligibleForEsbPromo_ParamsSpec = { $: {} };
const PageHandler_IsEligibleForEsbPromo_ResponseParamsSpec = { $: {} };
const Page_RemoveItem_ParamsSpec = { $: {} };
const Page_UpdateItem_ParamsSpec = { $: {} };
const Page_InsertItems_ParamsSpec = { $: {} };
const Page_ClearAll_ParamsSpec = { $: {} };
mojo.internal.Struct(DataSpec.$, 'Data', [
    mojo.internal.StructField('fileExternallyRemoved', 0, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('isDangerous', 0, 1, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('isInsecure', 0, 2, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('isReviewable', 0, 3, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('otr', 0, 4, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('resume', 0, 5, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('retry', 0, 6, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('percent', 4, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('started', 8, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('total', 16, 0, mojo.internal.Int64, BigInt(0), false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('byExtId', 24, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('byExtName', 32, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('dangerType', 12, 0, DangerTypeSpec.$, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('tailoredWarningType', 40, 0, TailoredWarningTypeSpec.$, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('dateString', 48, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('fileName', 56, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('filePath', 64, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('fileUrl', 72, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('id', 80, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('lastReasonText', 88, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('progressStatusText', 96, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('showInFolderText', 104, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('sinceString', 112, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('state', 44, 0, StateSpec.$, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('displayInitiatorOrigin', 120, 0, String16Spec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('url', 128, 0, UrlSpec.$, null, true /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('displayUrl', 136, 0, String16Spec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('safeBrowsingState', 144, 0, SafeBrowsingStateSpec.$, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('hasSafeBrowsingVerdict', 0, 7, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 160],]);
mojo.internal.Struct(PageHandlerFactory_CreatePageHandler_ParamsSpec.$, 'PageHandlerFactory_CreatePageHandler_Params', [
    mojo.internal.StructField('page', 0, 0, mojo.internal.InterfaceProxy(PageRemote), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('handler', 8, 0, mojo.internal.InterfaceRequest(PageHandlerPendingReceiver), null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(PageHandler_GetDownloads_ParamsSpec.$, 'PageHandler_GetDownloads_Params', [
    mojo.internal.StructField('searchTerms', 0, 0, mojo.internal.Array(mojo.internal.String, false), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_OpenFileRequiringGesture_ParamsSpec.$, 'PageHandler_OpenFileRequiringGesture_Params', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_Drag_ParamsSpec.$, 'PageHandler_Drag_Params', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_SaveSuspiciousRequiringGesture_ParamsSpec.$, 'PageHandler_SaveSuspiciousRequiringGesture_Params', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_RecordOpenBypassWarningDialog_ParamsSpec.$, 'PageHandler_RecordOpenBypassWarningDialog_Params', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_SaveDangerousFromDialogRequiringGesture_ParamsSpec.$, 'PageHandler_SaveDangerousFromDialogRequiringGesture_Params', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_RecordCancelBypassWarningDialog_ParamsSpec.$, 'PageHandler_RecordCancelBypassWarningDialog_Params', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_DiscardDangerous_ParamsSpec.$, 'PageHandler_DiscardDangerous_Params', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_RetryDownload_ParamsSpec.$, 'PageHandler_RetryDownload_Params', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_Show_ParamsSpec.$, 'PageHandler_Show_Params', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_Pause_ParamsSpec.$, 'PageHandler_Pause_Params', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_Resume_ParamsSpec.$, 'PageHandler_Resume_Params', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_Remove_ParamsSpec.$, 'PageHandler_Remove_Params', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_Undo_ParamsSpec.$, 'PageHandler_Undo_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_Cancel_ParamsSpec.$, 'PageHandler_Cancel_Params', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_ClearAll_ParamsSpec.$, 'PageHandler_ClearAll_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_OpenDownloadsFolderRequiringGesture_ParamsSpec.$, 'PageHandler_OpenDownloadsFolderRequiringGesture_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_OpenEsbSettings_ParamsSpec.$, 'PageHandler_OpenEsbSettings_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_LogEsbPromotionRowViewed_ParamsSpec.$, 'PageHandler_LogEsbPromotionRowViewed_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_OpenDuringScanningRequiringGesture_ParamsSpec.$, 'PageHandler_OpenDuringScanningRequiringGesture_Params', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_ReviewDangerousRequiringGesture_ParamsSpec.$, 'PageHandler_ReviewDangerousRequiringGesture_Params', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_DeepScan_ParamsSpec.$, 'PageHandler_DeepScan_Params', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_BypassDeepScanRequiringGesture_ParamsSpec.$, 'PageHandler_BypassDeepScanRequiringGesture_Params', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_IsEligibleForEsbPromo_ParamsSpec.$, 'PageHandler_IsEligibleForEsbPromo_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_IsEligibleForEsbPromo_ResponseParamsSpec.$, 'PageHandler_IsEligibleForEsbPromo_ResponseParams', [
    mojo.internal.StructField('result', 0, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Page_RemoveItem_ParamsSpec.$, 'Page_RemoveItem_Params', [
    mojo.internal.StructField('index', 0, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Page_UpdateItem_ParamsSpec.$, 'Page_UpdateItem_Params', [
    mojo.internal.StructField('index', 0, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('data', 8, 0, DataSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(Page_InsertItems_ParamsSpec.$, 'Page_InsertItems_Params', [
    mojo.internal.StructField('index', 0, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('items', 8, 0, mojo.internal.Array(DataSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(Page_ClearAll_ParamsSpec.$, 'Page_ClearAll_Params', [], [[0, 8],]);

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class BrowserProxy {
    callbackRouter;
    handler;
    constructor() {
        this.callbackRouter = new PageCallbackRouter();
        this.handler = new PageHandlerRemote();
        const factory = PageHandlerFactory.getRemote();
        factory.createPageHandler(this.callbackRouter.$.bindNewPipeAndPassRemote(), this.handler.$.bindNewPipeAndPassReceiver());
    }
    static getInstance() {
        return instance$e || (instance$e = new BrowserProxy());
    }
    static setInstance(obj) {
        instance$e = obj;
    }
}
let instance$e = null;

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * A URL for the filetype icon for |filePath|. OS and theme dependent.
 */
function getFileIconUrl(filePath) {
    const url = new URL('chrome://fileicon/');
    url.searchParams.set('path', filePath);
    url.searchParams.set('scale', window.devicePixelRatio + 'x');
    return url.toString();
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview PromiseResolver is a helper class that allows creating a
 * Promise that will be fulfilled (resolved or rejected) some time later.
 *
 * Example:
 *  const resolver = new PromiseResolver();
 *  resolver.promise.then(function(result) {
 *    console.log('resolved with', result);
 *  });
 *  ...
 *  ...
 *  resolver.resolve({hello: 'world'});
 */
class PromiseResolver {
    resolve_ = () => { };
    reject_ = () => { };
    isFulfilled_ = false;
    promise_;
    constructor() {
        this.promise_ = new Promise((resolve, reject) => {
            this.resolve_ = (resolution) => {
                resolve(resolution);
                this.isFulfilled_ = true;
            };
            this.reject_ = (reason) => {
                reject(reason);
                this.isFulfilled_ = true;
            };
        });
    }
    /** Whether this resolver has been resolved or rejected. */
    get isFulfilled() {
        return this.isFulfilled_;
    }
    get promise() {
        return this.promise_;
    }
    get resolve() {
        return this.resolve_;
    }
    get reject() {
        return this.reject_;
    }
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class IconLoaderImpl {
    iconResolvers_;
    listeningImages_;
    constructor() {
        this.iconResolvers_ = new Map();
        this.listeningImages_ = new Set();
    }
    /**
     * @return Whether or not the icon loaded successfully.
     */
    loadIcon(imageEl, filePath) {
        const url = getFileIconUrl(filePath);
        if (!this.iconResolvers_.has(url)) {
            this.iconResolvers_.set(url, new PromiseResolver());
        }
        if (!this.listeningImages_.has(imageEl)) {
            imageEl.addEventListener('load', this.finishedLoading_.bind(this));
            imageEl.addEventListener('error', this.finishedLoading_.bind(this));
            this.listeningImages_.add(imageEl);
        }
        imageEl.src = url;
        return this.iconResolvers_.get(url).promise;
    }
    finishedLoading_(e) {
        const resolver = this.iconResolvers_.get(e.currentTarget.src);
        if (!resolver.isFulfilled) {
            resolver.resolve(e.type === 'load');
        }
    }
    static getInstance() {
        return instance$d || (instance$d = new IconLoaderImpl());
    }
    static setInstance(obj) {
        instance$d = obj;
    }
}
let instance$d = null;

let instance$c = null;
function getCss$b() {
    return instance$c || (instance$c = [...[], css `[is='action-link']{cursor:pointer;display:inline-block;text-decoration:underline}[is='action-link'],[is='action-link']:active,[is='action-link']:hover,[is='action-link']:visited{color:var(--cr-link-color)}[is='action-link'][disabled]{color:var(--cr-fallback-color-disabled-foreground);cursor:default;pointer-events:none}[is='action-link'].no-outline{outline:none}`]);
}

let instance$b = null;
function getCss$a() {
    return instance$b || (instance$b = [...[getCss$o(), getCss$b(), getCss$k()], css `:host{--controlled-by-active-color:#333;--controlled-by-active-link-color:var(--google-blue-600);--controlled-by-inactive-color:#5a5a5a;display:flex;flex-direction:column;outline:none}@media (prefers-color-scheme:dark){:host{--controlled-by-active-color:inherit;--controlled-by-active-link-color:var(--cr-link-color);--controlled-by-inactive-color:inherit}}cr-button{font-weight:500;margin:0;min-width:auto}#date{font-size:0.875rem;font-weight:400;letter-spacing:.25px;margin:21px auto 6px;padding-bottom:4px;padding-top:8px;width:var(--downloads-card-width)}#date:empty{display:none}#content{border-radius:var(--cr-card-border-radius);margin:6px auto;min-height:75px;width:var(--downloads-card-width);max-width:calc(100% - 2 * var(--downloads-card-margin))}#main-content{display:flex;flex:none}#content.is-active{background-color:var(--cr-card-background-color);box-shadow:var(--cr-card-shadow)}@media (prefers-color-scheme:light){#content:not(.is-active){background-color:rgb(221,227,234)}}@media (prefers-color-scheme:dark){#content:not(.is-active){background-color:rgb(51,53,55);border-color:var(--google-grey-800)}}#details{display:flex;flex:1;flex-direction:column;min-width:0;padding-bottom:16px;padding-inline-end:16px;padding-inline-start:var(--downloads-card-margin);padding-top:18px}@media (prefers-color-scheme:dark){#details{border-color:rgba(var(--google-grey-800-rgb),.8)}}#content:not(.is-active) #name{text-decoration:line-through}@media (prefers-color-scheme:dark){#content:not(.is-active) :-webkit-any(#name,#tag){color:var(--google-grey-500)}}.item-tray{flex:none;width:var(--downloads-card-width)}.icon-wrapper{align-self:start;flex:none;justify-content:center;margin:18px 24px}.icon,#file-icon-wrapper{height:32px;width:32px}#file-icon-wrapper{overflow:hidden}#content:not(.is-active) .icon{filter:grayscale(100%);opacity:.5}#esb-download-row-promo{background-color:var(--cr-hover-background-color)}#file-icon-wrapper cr-icon[icon-color='light-grey']{color:var(--google-grey-400)}#file-icon-wrapper cr-icon[icon-color='red']{color:var(--google-red-700)}#file-icon-wrapper cr-icon[icon-color='yellow']{color:var(--google-yellow-500)}@media (prefers-color-scheme:dark){#file-icon-wrapper cr-icon[icon-color='red']{color:var(--google-red-300)}}#file-icon-wrapper cr-icon[icon-color='grey']{color:var(--google-grey-700)}@media (prefers-color-scheme:dark){#file-icon-wrapper cr-icon[icon-color='grey']{color:var(--google-grey-500)}}.description[description-color='red']{color:var(--google-red-700)}@media (prefers-color-scheme:dark){.description[description-color='red']{color:var(--google-red-300)}}.description[description-color='grey']{color:var(--google-grey-700)}@media (prefers-color-scheme:dark){.description[description-color='grey']{color:var(--google-grey-500)}}#title-area{display:flex;flex-direction:row}#name,#file-link,#url{max-width:100%;overflow:hidden;text-overflow:ellipsis;white-space:nowrap}#name,#file-link{font-weight:500;word-break:break-all;text-align:start}@media (prefers-color-scheme:light){.is-active :-webkit-any(#file-link){color:var(--google-blue-600)}.is-active #name{color:var(--cr-primary-text-color)}}#tag{color:#5a5a5a;font-weight:500;margin-inline-start:12px}#url{color:inherit;display:block;margin-top:6px;min-height:0;text-decoration:none;direction:rtl;text-align:end}#initiator-origin{text-overflow:ellipsis;white-space:nowrap;display:flex;flex-wrap:nowrap;margin-top:6px}:host-context([dir=rtl]) #url{direction:ltr}:host-context([dir=rtl]) #name{direction:ltr}:host-context([dir=rtl]) #file-link{direction:ltr}.is-active #url{color:var(--cr-secondary-text-color)}#progress,.description:not(:empty),.controls{margin-top:16px}@media (prefers-color-scheme:light){.is-active .description[description-color='']{color:#616161}}#progress{--cr-progress-active-color:var(--google-blue-600);--cr-progress-container-color:rgb(223,222,223);width:auto}@media (prefers-color-scheme:dark){#progress{--cr-progress-active-color:var(--google-blue-300);--cr-progress-container-color:var(--google-grey-800)}}#controlled-by,#controlled-by a{color:var(--controlled-by-inactive-color)}.is-active #controlled-by{color:var(--controlled-by-active-color)}.is-active #controlled-by a{color:var(--controlled-by-active-link-color)}.more-options{display:flex;flex-direction:column;align-items:center}#remove-old{--cr-icon-button-icon-size:20px;--cr-icon-button-margin-end:12px;--cr-icon-button-margin-start:12px;margin-top:12px}#action-icon-buttons{display:flex;flex-direction:row;--cr-icon-button-icon-size:20px;--cr-icon-button-margin-end:0;--cr-icon-button-margin-start:0;gap:12px;margin-top:12px;margin-inline-start:12px;margin-inline-end:12px}#incognito{-webkit-mask-image:url(images/incognito_marker.svg);background-color:var(--cr-secondary-text-color);height:16px;margin-block-end:20px;margin-block-start:auto;width:16px}#deepScan{margin-inline-end:8px}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$5() {
    // clang-format off
    return html `<!--_html_template_start_-->
<div id="date" role="heading" aria-level="2">${this.computeDate_()}</div>

<div id="content" @dragstart="${this.onDragStart_}"
    class="${this.computeClass_()}" focus-row-container>
  <div id="main-content">
    <div id="file-icon-wrapper" class="icon-wrapper" role="img"
        aria-label="${this.computeIconAriaLabel_()}"
        aria-hidden="${this.computeIconAriaHidden_()}">
      <img class="icon" id="file-icon" alt="" ?hidden="${!this.useFileIcon_}"
          icon-color="${this.computeIconColor_()}">
      <cr-icon class="icon" ?hidden="${this.useFileIcon_}"
          .icon="${this.computeIcon_()}"
          icon-color="${this.computeIconColor_()}">
      </cr-icon>
    </div>

    <div id="details">
      <div id="title-area" role="gridcell"><!--
        Can't have any line breaks.
        --><a is="action-link" id="file-link"
            href="${this.data?.url?.url || ''}"
            @click="${this.onFileLinkClick_}" focus-row-control
            focus-type="fileLink"
            title="${this.data?.fileName || ''}"
            ?hidden="${!this.shouldLinkFilename_}"><!-- No line break
          -->${this.data?.fileName || ''}<!-- No line break
        --></a><!--
        Before #name.
        --><span id="name"
            title="${this.data?.fileName || ''}"
            ?hidden="${this.shouldLinkFilename_}"><!-- No line break
          -->${this.data?.fileName || ''}</span>
        <span id="tag">${this.computeTag_()}</span>
      </div>

      <div role="gridcell">
        <div id="initiator-origin" ?hidden="${!this.showInitiatorOrigin_}">
          ${this.computeInitiatorOriginText_()}
        </div>
        <a id="url" ?hidden="${this.showInitiatorOrigin_}" target="_blank"
          @click="${this.onUrlClick_}" focus-row-control
          focus-type="url">${this.getDisplayUrlStr_()}</a>
      </div>

      <div class="description" role="gridcell"
          description-color="${this.iconAndDescriptionColor_()}"
          ?hidden="${!this.computeDescriptionVisible_()}">
        ${this.computeDescription_()}
      </div>

      <div class="description" role="gridcell"
          ?hidden="${!this.computeSecondLineVisible_()}">
        $i18n{asyncScanningDownloadDescSecond}
      </div>

      ${this.showProgress_ ? html `
        <div role="gridcell">
          <cr-progress id="progress"
              .indeterminate="${this.isIndeterminate_()}"
              .value="${this.data?.percent || 0}">
          </cr-progress>
        </div>` : ''}

      <div id="safe" class="controls" ?hidden="${this.isDangerous_}">
        <span role="gridcell" ?hidden="${!this.showDeepScan_}">
          <cr-button @click="${this.onDeepScanClick_}" id="deepScan"
              class="action-button" focus-row-control focus-type="open">
            ${this.computeDeepScanControlText_()}
          </cr-button>
        </span>
      </div>
      <div id="controlled-by" ?hidden="${this.isDangerous_}"><!--
        Text populated dynamically.
      --></div>
    </div>
    <div class="more-options">
      <!-- Menu and/or quick action(s). -->
      <div role="gridcell" id="action-icon-buttons">
        <cr-icon-button id="copy-download-link" iron-icon="downloads:link"
            ?hidden="${!this.computeShowCopyDownloadLink_()}"
            title="$i18n{controlCopyDownloadLink}"
            aria-label="$i18n{controlCopyDownloadLink}"
            @click="${this.onCopyDownloadLinkClick_}"
            focus-row-control focus-type="copyDownloadLink">
        </cr-icon-button>
        <cr-icon-button id="more-actions" iron-icon="cr:more-vert"
            ?hidden="${!this.computeShowActionMenu_()}"
            class="dropdown-trigger" title="$i18n{moreActions}"
            @click="${this.onMoreActionsClick_}" aria-haspopup="menu"
            focus-row-control focus-type="actionMenuButton">
        </cr-icon-button>
        <cr-icon-button id="quick-show-in-folder" class="icon-folder-open"
            ?hidden="${!this.computeShowQuickShow_()}"
            title="${this.data?.showInFolderText || ''}"
            aria-label="${this.data?.showInFolderText || ''}"
            @click="${this.onShowClick_}"
            focus-row-control focus-type="quickShow">
        </cr-icon-button>
        <cr-icon-button id="quick-remove" class="icon-clear"
            ?hidden="${!this.computeShowQuickRemove_()}"
            title="$i18n{controlDeleteFromHistory}"
            aria-label="$i18n{controlDeleteFromHistory}"
            @click="${this.onQuickRemoveClick_}"
            focus-row-control focus-type="quickRemove">
        </cr-icon-button>
      </div>
      <cr-action-menu id="more-actions-menu"
          role-description="$i18n{actionMenuDescription}">
        <button class="dropdown-item" @click="${this.onShowClick_}"
            ?hidden="${!this.computeHasShowInFolderLink_()}"
            id="show-in-folder">
          ${this.data?.showInFolderText || ''}
        </button>
        <button class="dropdown-item" @click="${this.onPauseOrResumeClick_}"
            ?hidden="${!this.pauseOrResumeText_}" id="pause-or-resume">
          ${this.pauseOrResumeText_}
        </button>
        <button class="dropdown-item" @click="${this.onRemoveClick_}"
            ?hidden="${!this.computeShowRemove_()}" id="remove">
          $i18n{controlDeleteFromHistory}
        </button>
        <button class="dropdown-item" @click="${this.onDiscardDangerousClick_}"
            ?hidden="${!this.computeShowControlsForDangerous_()}"
            id="discard-dangerous">
          $i18n{controlDeleteFromHistory}
        </button>
        <button class="dropdown-item" @click="${this.onRetryClick_}"
            ?hidden="${!this.data?.retry}" id="retry">
          $i18n{controlRetry}
        </button>
        <button class="dropdown-item" @click="${this.onDeepScanClick_}"
            ?hidden="${!this.showDeepScan_}" id="deep-scan">
          $i18n{controlDeepScan}
        </button>
        <button class="dropdown-item" @click="${this.onBypassDeepScanClick_}"
            ?hidden="${!this.showDeepScan_}" id="bypass-deep-scan">
          ${this.computeSaveDangerousLabel_()}
        </button>
        <button class="dropdown-item" @click="${this.onCancelClick_}"
            ?hidden="${!this.showCancel_}" id="cancel">
          $i18n{controlCancel}
        </button>
        <button class="dropdown-item" @click="${this.onOpenAnywayClick_}"
            ?hidden="${!this.showOpenAnyway_}" id="open-anyway">
          $i18n{controlOpenAnyway}
        </button>
        <button class="dropdown-item" @click="${this.onDiscardDangerousClick_}"
            ?hidden="${!this.isReviewable_}" id="reviewable-discard-dangerous">
          $i18n{dangerDiscard}
        </button>
        <button class="dropdown-item" @click="${this.onReviewDangerousClick_}"
            ?hidden="${!this.isReviewable_}" id="review-dangerous">
          $i18n{dangerReview}
        </button>
        <button class="dropdown-item" @click="${this.onSaveDangerousClick_}"
            ?hidden="${!this.computeShowControlsForDangerous_()}"
            id="save-dangerous">
          ${this.computeSaveDangerousLabel_()}
        </button>
      </cr-action-menu>
      <div id="incognito" title="$i18n{inIncognito}"
          ?hidden="${!this.data?.otr}">
      </div>
    </div>
  </div>
  
</div>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const DownloadsItemElementBase = I18nMixinLit(FocusRowMixinLit(CrLitElement));
/**
 * The UI pattern for displaying a download. Computed from DangerType and other
 * properties of the download and user's profile.
 */
var DisplayType;
(function (DisplayType) {
    DisplayType[DisplayType["NORMAL"] = 0] = "NORMAL";
    DisplayType[DisplayType["DANGEROUS"] = 1] = "DANGEROUS";
    DisplayType[DisplayType["SUSPICIOUS"] = 2] = "SUSPICIOUS";
    DisplayType[DisplayType["UNVERIFIED"] = 3] = "UNVERIFIED";
    DisplayType[DisplayType["INSECURE"] = 4] = "INSECURE";
    DisplayType[DisplayType["ERROR"] = 5] = "ERROR";
})(DisplayType || (DisplayType = {}));
class DownloadsItemElement extends DownloadsItemElementBase {
    static get is() {
        return 'downloads-item';
    }
    static get styles() {
        return getCss$a();
    }
    render() {
        return getHtml$5.bind(this)();
    }
    static get properties() {
        return {
            data: { type: Object },
            completelyOnDisk_: { type: Boolean },
            shouldLinkFilename_: { type: Boolean },
            isDangerous_: { type: Boolean },
            isReviewable_: { type: Boolean },
            pauseOrResumeText_: { type: String },
            showCancel_: { type: Boolean },
            showProgress_: { type: Boolean },
            showDeepScan_: { type: Boolean },
            showOpenAnyway_: { type: Boolean },
            displayType_: {
                type: Number,
                state: true,
            },
            // 
            useFileIcon_: { type: Boolean },
            showInitiatorOrigin_: { type: Boolean },
        };
    }
    #data_accessor_storage;
    get data() { return this.#data_accessor_storage; }
    set data(value) { this.#data_accessor_storage = value; }
    // 
    mojoHandler_ = null;
    #isDangerous__accessor_storage = false;
    get isDangerous_() { return this.#isDangerous__accessor_storage; }
    set isDangerous_(value) { this.#isDangerous__accessor_storage = value; }
    #isReviewable__accessor_storage = false;
    get isReviewable_() { return this.#isReviewable__accessor_storage; }
    set isReviewable_(value) { this.#isReviewable__accessor_storage = value; }
    #pauseOrResumeText__accessor_storage = '';
    get pauseOrResumeText_() { return this.#pauseOrResumeText__accessor_storage; }
    set pauseOrResumeText_(value) { this.#pauseOrResumeText__accessor_storage = value; }
    #showCancel__accessor_storage = false;
    get showCancel_() { return this.#showCancel__accessor_storage; }
    set showCancel_(value) { this.#showCancel__accessor_storage = value; }
    #showProgress__accessor_storage = false;
    get showProgress_() { return this.#showProgress__accessor_storage; }
    set showProgress_(value) { this.#showProgress__accessor_storage = value; }
    #showDeepScan__accessor_storage = false;
    get showDeepScan_() { return this.#showDeepScan__accessor_storage; }
    set showDeepScan_(value) { this.#showDeepScan__accessor_storage = value; }
    #showOpenAnyway__accessor_storage = false;
    get showOpenAnyway_() { return this.#showOpenAnyway__accessor_storage; }
    set showOpenAnyway_(value) { this.#showOpenAnyway__accessor_storage = value; }
    #useFileIcon__accessor_storage = false;
    get useFileIcon_() { return this.#useFileIcon__accessor_storage; }
    set useFileIcon_(value) { this.#useFileIcon__accessor_storage = value; }
    #showInitiatorOrigin__accessor_storage = loadTimeData.getBoolean('showInitiatorOrigin');
    get showInitiatorOrigin_() { return this.#showInitiatorOrigin__accessor_storage; }
    set showInitiatorOrigin_(value) { this.#showInitiatorOrigin__accessor_storage = value; }
    restoreFocusAfterCancel_ = false;
    #displayType__accessor_storage = DisplayType.NORMAL;
    get displayType_() { return this.#displayType__accessor_storage; }
    set displayType_(value) { this.#displayType__accessor_storage = value; }
    #completelyOnDisk__accessor_storage = true;
    get completelyOnDisk_() { return this.#completelyOnDisk__accessor_storage; }
    set completelyOnDisk_(value) { this.#completelyOnDisk__accessor_storage = value; }
    #shouldLinkFilename__accessor_storage = true;
    get shouldLinkFilename_() { return this.#shouldLinkFilename__accessor_storage; }
    set shouldLinkFilename_(value) { this.#shouldLinkFilename__accessor_storage = value; }
    overrideCustomEquivalent = true;
    firstUpdated() {
        this.setAttribute('role', 'row');
        this.mojoHandler_ = BrowserProxy.getInstance().handler;
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('data')) {
            this.completelyOnDisk_ = this.computeCompletelyOnDisk_();
            this.shouldLinkFilename_ = this.computeShouldLinkFilename_();
            this.isDangerous_ = this.computeIsDangerous_();
            this.isReviewable_ = this.computeIsReviewable_();
            this.pauseOrResumeText_ = this.computePauseOrResumeText_();
            this.showCancel_ = this.computeShowCancel_();
            // Compute showProgress_ after showCancel_ since it uses showCancel_.
            this.showProgress_ = this.computeShowProgress_();
            this.showDeepScan_ = this.computeShowDeepScan_();
            this.showOpenAnyway_ = this.computeShowOpenAnyway_();
            this.displayType_ = this.computeDisplayType_();
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('data')) {
            this.updateControlledBy_();
            this.updateUiForStateChange_();
            this.restoreFocusAfterCancelIfNeeded_();
            // Since this item's height may have changed due to the updated data,
            // we send an iron-resize event to iron-list to allow it to resize.
            this.fire('iron-resize');
        }
    }
    /** Overrides FocusRowMixin. */
    getCustomEquivalent(sampleElement) {
        if (sampleElement.getAttribute('focus-type') === 'cancel') {
            return this.shadowRoot.querySelector('[focus-type="retry"]');
        }
        if (sampleElement.getAttribute('focus-type') === 'retry') {
            return this.shadowRoot.querySelector('[focus-type="pauseOrResume"]');
        }
        return null;
    }
    getFileIcon() {
        return this.$['file-icon'];
    }
    getMoreActionsButton() {
        const button = this.shadowRoot.querySelector('#more-actions');
        return button || null;
    }
    getMoreActionsMenu() {
        const menu = this.shadowRoot.querySelector('#more-actions-menu');
        assert(!!menu);
        return menu;
    }
    /**
     * @return A JS string of the display URL.
     */
    getDisplayUrlStr_() {
        return this.data ? this.data.displayUrl : '';
    }
    computeClass_() {
        const classes = [];
        if (this.computeIsActive_()) {
            classes.push('is-active');
        }
        if (this.isDangerous_) {
            classes.push('dangerous');
        }
        if (this.showProgress_) {
            classes.push('show-progress');
        }
        return classes.join(' ');
    }
    computeCompletelyOnDisk_() {
        if (this.data === undefined) {
            return false;
        }
        if (this.data.fileExternallyRemoved) {
            return false;
        }
        switch (this.data.state) {
            case State.kComplete:
                return true;
            case State.kInProgress:
            case State.kCancelled:
            case State.kPaused:
            case State.kDangerous:
            case State.kInterrupted:
            case State.kInsecure:
            case State.kAsyncScanning:
            case State.kPromptForScanning:
            case State.kPromptForLocalPasswordScanning:
                return false;
            default:
                assertNotReached('Unhandled State encountered');
        }
    }
    computeShouldLinkFilename_() {
        if (this.data === undefined) {
            return false;
        }
        if (!this.completelyOnDisk_) {
            return false;
        }
        switch (this.data.dangerType) {
            case DangerType.kDeepScannedFailed:
                return false;
            case DangerType.kNoApplicableDangerType:
            case DangerType.kDangerousFile:
            case DangerType.kDangerousUrl:
            case DangerType.kDangerousContent:
            case DangerType.kCookieTheft:
            case DangerType.kUncommonContent:
            case DangerType.kDangerousHost:
            case DangerType.kPotentiallyUnwanted:
            case DangerType.kAsyncScanning:
            case DangerType.kAsyncLocalPasswordScanning:
            case DangerType.kBlockedPasswordProtected:
            case DangerType.kBlockedTooLarge:
            case DangerType.kSensitiveContentWarning:
            case DangerType.kSensitiveContentBlock:
            case DangerType.kDeepScannedSafe:
            case DangerType.kDeepScannedOpenedDangerous:
            case DangerType.kBlockedScanFailed:
                return true;
            default:
                assertNotReached('Unhandled DangerType encountered');
        }
    }
    computeHasShowInFolderLink_() {
        if (this.data === undefined) {
            return false;
        }
        if (!this.computeCompletelyOnDisk_()) {
            return false;
        }
        switch (this.data.dangerType) {
            case DangerType.kDeepScannedFailed:
                return false;
            case DangerType.kNoApplicableDangerType:
            case DangerType.kDangerousFile:
            case DangerType.kDangerousUrl:
            case DangerType.kDangerousContent:
            case DangerType.kCookieTheft:
            case DangerType.kUncommonContent:
            case DangerType.kDangerousHost:
            case DangerType.kPotentiallyUnwanted:
            case DangerType.kAsyncScanning:
            case DangerType.kAsyncLocalPasswordScanning:
            case DangerType.kBlockedPasswordProtected:
            case DangerType.kBlockedTooLarge:
            case DangerType.kSensitiveContentWarning:
            case DangerType.kSensitiveContentBlock:
            case DangerType.kDeepScannedSafe:
            case DangerType.kDeepScannedOpenedDangerous:
            case DangerType.kBlockedScanFailed:
                return true;
            default:
                assertNotReached('Unhandled DangerType encountered');
        }
    }
    computeControlledBy_() {
        if (!this.data || !this.data.byExtId || !this.data.byExtName) {
            return '';
        }
        const url = `chrome://extensions/?id=${this.data.byExtId}`;
        const name = this.data.byExtName;
        return loadTimeData.getStringF('controlledByUrl', url, htmlEscape(name));
    }
    computeDate_() {
        if (!this.data) {
            return '';
        }
        assert(typeof this.data.hideDate === 'boolean');
        if (this.data.hideDate) {
            return '';
        }
        return this.data.sinceString || this.data.dateString;
    }
    computeDescriptionVisible_() {
        return this.computeDescription_() !== '';
    }
    computeSecondLineVisible_() {
        if (!this.data) {
            return false;
        }
        switch (this.data.state) {
            case State.kAsyncScanning:
                return true;
            case State.kInProgress:
            case State.kCancelled:
            case State.kComplete:
            case State.kPaused:
            case State.kDangerous:
            case State.kInterrupted:
            case State.kInsecure:
            case State.kPromptForScanning:
            case State.kPromptForLocalPasswordScanning:
                return false;
            default:
                assertNotReached('Unhandled State encountered');
        }
    }
    isSuspiciousEnterpriseApVerdict_(requestsApVerdicts, dangerType) {
        switch (dangerType) {
            case DangerType.kUncommonContent:
                return requestsApVerdicts;
            case DangerType.kSensitiveContentWarning:
                return true;
            case DangerType.kNoApplicableDangerType:
            case DangerType.kDangerousFile:
            case DangerType.kDangerousUrl:
            case DangerType.kDangerousContent:
            case DangerType.kCookieTheft:
            case DangerType.kDangerousHost:
            case DangerType.kPotentiallyUnwanted:
            case DangerType.kAsyncScanning:
            case DangerType.kAsyncLocalPasswordScanning:
            case DangerType.kBlockedPasswordProtected:
            case DangerType.kBlockedTooLarge:
            case DangerType.kSensitiveContentBlock:
            case DangerType.kDeepScannedFailed:
            case DangerType.kDeepScannedSafe:
            case DangerType.kDeepScannedOpenedDangerous:
            case DangerType.kBlockedScanFailed:
                return false;
            default:
                assertNotReached('Unhandled DangerType encountered');
        }
    }
    computeDisplayType_() {
        // Most downloads are normal. If we don't have data, don't assume danger.
        if (!this.data) {
            return DisplayType.NORMAL;
        }
        if (this.data.isInsecure) {
            return DisplayType.INSECURE;
        }
        switch (this.data.state) {
            case State.kAsyncScanning:
            case State.kPromptForScanning:
            case State.kPromptForLocalPasswordScanning:
                return DisplayType.SUSPICIOUS;
            case State.kInsecure:
                return DisplayType.INSECURE;
            case State.kInProgress:
            case State.kCancelled:
            case State.kComplete:
            case State.kPaused:
            case State.kDangerous:
            case State.kInterrupted:
                break;
            default:
                assertNotReached('Unhandled State encountered');
        }
        // Enterprise AP verdicts.
        if (this.isSuspiciousEnterpriseApVerdict_(loadTimeData.getBoolean('requestsApVerdicts'), this.data.dangerType)) {
            return DisplayType.SUSPICIOUS;
        }
        switch (this.data.dangerType) {
            // Mimics logic in download_ui_model.cc for downloads with danger_type
            // DOWNLOAD_DANGER_TYPE_DANGEROUS_FILE.
            case DangerType.kDangerousFile:
                return this.data.hasSafeBrowsingVerdict ? DisplayType.SUSPICIOUS :
                    DisplayType.UNVERIFIED;
            case DangerType.kDangerousUrl:
            case DangerType.kDangerousContent:
            case DangerType.kCookieTheft:
            case DangerType.kDangerousHost:
            case DangerType.kPotentiallyUnwanted:
            case DangerType.kDeepScannedOpenedDangerous:
                return DisplayType.DANGEROUS;
            case DangerType.kUncommonContent:
            case DangerType.kDeepScannedFailed:
                return DisplayType.SUSPICIOUS;
            case DangerType.kNoApplicableDangerType:
            case DangerType.kAsyncScanning:
            case DangerType.kAsyncLocalPasswordScanning:
            case DangerType.kSensitiveContentWarning:
            case DangerType.kDeepScannedSafe:
            case DangerType.kBlockedScanFailed:
                return DisplayType.NORMAL;
            case DangerType.kBlockedPasswordProtected:
            case DangerType.kBlockedTooLarge:
            case DangerType.kSensitiveContentBlock:
                return DisplayType.ERROR;
            default:
                assertNotReached('Unhandled DangerType encountered');
        }
    }
    computeDeepScanControlText_() {
        if (!this.data) {
            return '';
        }
        switch (this.data.state) {
            case State.kPromptForScanning:
                return loadTimeData.getString('controlDeepScan');
            case State.kPromptForLocalPasswordScanning:
                return loadTimeData.getString('controlLocalPasswordScan');
            case State.kInProgress:
            case State.kCancelled:
            case State.kComplete:
            case State.kPaused:
            case State.kDangerous:
            case State.kInterrupted:
            case State.kInsecure:
            case State.kAsyncScanning:
                return '';
            default:
                assertNotReached('Unhandled State encountered');
        }
    }
    computeSaveDangerousLabel_() {
        switch (this.displayType_) {
            case DisplayType.DANGEROUS:
                return this.i18n('controlKeepDangerous');
            case DisplayType.SUSPICIOUS:
                return this.i18n('controlKeepSuspicious');
            case DisplayType.UNVERIFIED:
                return this.i18n('controlKeepUnverified');
            case DisplayType.INSECURE:
                return this.i18n('controlKeepInsecure');
            case DisplayType.NORMAL:
            case DisplayType.ERROR:
                return '';
            default:
                assertNotReached('Unhandled DisplayType encountered');
        }
    }
    computeDescription_() {
        if (!this.data) {
            return '';
        }
        const data = this.data;
        switch (data.state) {
            case State.kComplete:
                switch (data.dangerType) {
                    case DangerType.kDeepScannedOpenedDangerous:
                        return loadTimeData.getString('deepScannedOpenedDangerousDesc');
                    case DangerType.kDeepScannedFailed:
                        return loadTimeData.getString('deepScannedFailedDesc');
                    case DangerType.kNoApplicableDangerType:
                    case DangerType.kDangerousFile:
                    case DangerType.kDangerousUrl:
                    case DangerType.kDangerousContent:
                    case DangerType.kCookieTheft:
                    case DangerType.kUncommonContent:
                    case DangerType.kDangerousHost:
                    case DangerType.kPotentiallyUnwanted:
                    case DangerType.kAsyncScanning:
                    case DangerType.kAsyncLocalPasswordScanning:
                    case DangerType.kBlockedPasswordProtected:
                    case DangerType.kBlockedTooLarge:
                    case DangerType.kSensitiveContentWarning:
                    case DangerType.kSensitiveContentBlock:
                    case DangerType.kDeepScannedSafe:
                    case DangerType.kBlockedScanFailed:
                        return '';
                    default:
                        assertNotReached('Unhandled DangerType encountered');
                }
            case State.kInsecure:
                return loadTimeData.getString('insecureDownloadDesc');
            case State.kDangerous:
                switch (data.dangerType) {
                    case DangerType.kNoApplicableDangerType:
                        return '';
                    case DangerType.kDangerousFile:
                        return data.safeBrowsingState ===
                            SafeBrowsingState.kNoSafeBrowsing ?
                            loadTimeData.getString('noSafeBrowsingDesc') :
                            loadTimeData.getString('dangerFileDesc');
                    case DangerType.kDangerousUrl:
                    case DangerType.kDangerousContent:
                    case DangerType.kDangerousHost:
                        return loadTimeData.getString('dangerDownloadDesc');
                    case DangerType.kCookieTheft:
                        switch (data.tailoredWarningType) {
                            case TailoredWarningType.kCookieTheft:
                                return loadTimeData.getString('dangerDownloadCookieTheft');
                            case TailoredWarningType.kSuspiciousArchive:
                            case TailoredWarningType.kNoApplicableTailoredWarningType:
                                return loadTimeData.getString('dangerDownloadDesc');
                            default:
                                assertNotReached('Unhandled TailoredWarningType encountered');
                        }
                    case DangerType.kUncommonContent:
                        switch (data.tailoredWarningType) {
                            case TailoredWarningType.kSuspiciousArchive:
                                return loadTimeData.getString('dangerUncommonSuspiciousArchiveDesc');
                            case TailoredWarningType.kCookieTheft:
                            case TailoredWarningType.kNoApplicableTailoredWarningType:
                                return loadTimeData.getString('dangerUncommonDesc');
                            default:
                                assertNotReached('Unhandled TailoredWarningType encountered');
                        }
                    case DangerType.kPotentiallyUnwanted:
                        return loadTimeData.getString('dangerSettingsDesc');
                    case DangerType.kAsyncScanning:
                    case DangerType.kAsyncLocalPasswordScanning:
                    case DangerType.kBlockedPasswordProtected:
                    case DangerType.kBlockedTooLarge:
                        return '';
                    case DangerType.kSensitiveContentWarning:
                        return loadTimeData.getString('sensitiveContentWarningDesc');
                    case DangerType.kSensitiveContentBlock:
                    case DangerType.kDeepScannedFailed:
                    case DangerType.kDeepScannedSafe:
                    case DangerType.kDeepScannedOpenedDangerous:
                    case DangerType.kBlockedScanFailed:
                        return '';
                    default:
                        assertNotReached('Unhandled DangerType encountered');
                }
            case State.kAsyncScanning:
                return loadTimeData.getString('asyncScanningDownloadDesc');
            case State.kPromptForScanning:
                return loadTimeData.getString('promptForScanningDesc');
            case State.kPromptForLocalPasswordScanning:
                return loadTimeData.getString('promptForLocalPasswordScanningDesc');
            case State.kInProgress:
            case State.kPaused: // Fallthrough.
                return data.progressStatusText;
            case State.kInterrupted:
                switch (data.dangerType) {
                    case DangerType.kNoApplicableDangerType:
                    case DangerType.kDangerousFile:
                    case DangerType.kDangerousUrl:
                    case DangerType.kDangerousContent:
                    case DangerType.kDangerousHost:
                    case DangerType.kCookieTheft:
                    case DangerType.kUncommonContent:
                    case DangerType.kPotentiallyUnwanted:
                    case DangerType.kAsyncScanning:
                    case DangerType.kAsyncLocalPasswordScanning:
                        return '';
                    case DangerType.kBlockedPasswordProtected:
                        return loadTimeData.getString('blockedPasswordProtectedDesc');
                    case DangerType.kBlockedTooLarge:
                        return loadTimeData.getString('blockedTooLargeDesc');
                    case DangerType.kSensitiveContentWarning:
                        return '';
                    case DangerType.kSensitiveContentBlock:
                        return loadTimeData.getString('sensitiveContentBlockedDesc');
                    case DangerType.kDeepScannedFailed:
                    case DangerType.kDeepScannedSafe:
                    case DangerType.kDeepScannedOpenedDangerous:
                    case DangerType.kBlockedScanFailed:
                        return '';
                    default:
                        assertNotReached('Unhandled DangerType encountered');
                }
            case State.kCancelled:
                return '';
            default:
                assertNotReached('Unhandled State encountered');
        }
    }
    computeIconAriaHidden_() {
        return (this.displayType_ === DisplayType.NORMAL ||
            this.displayType_ === DisplayType.ERROR) ?
            'true' :
            'false';
    }
    computeIconAriaLabel_() {
        switch (this.displayType_) {
            case DisplayType.DANGEROUS:
                return this.i18n('accessibleLabelDangerous');
            case DisplayType.INSECURE:
                return this.i18n('accessibleLabelInsecure');
            case DisplayType.UNVERIFIED:
                return this.i18n('accessibleLabelUnverified');
            case DisplayType.SUSPICIOUS:
                return this.i18n('accessibleLabelSuspicious');
            case DisplayType.NORMAL:
            case DisplayType.ERROR:
                return '';
            default:
                assertNotReached('Unhandled DisplayType encountered');
        }
    }
    iconAndDescriptionColor_() {
        switch (this.displayType_) {
            case DisplayType.DANGEROUS:
            case DisplayType.ERROR:
                return 'red';
            case DisplayType.INSECURE:
            case DisplayType.UNVERIFIED:
            case DisplayType.SUSPICIOUS:
                return 'grey';
            case DisplayType.NORMAL:
                return '';
            default:
                assertNotReached('Unhandled DisplayType encountered');
        }
    }
    computeIcon_() {
        if (this.data) {
            switch (this.displayType_) {
                case DisplayType.DANGEROUS:
                    return 'downloads:dangerous';
                case DisplayType.INSECURE:
                case DisplayType.UNVERIFIED:
                case DisplayType.SUSPICIOUS:
                    return 'cr:warning';
                case DisplayType.ERROR:
                    return 'cr:error';
                case DisplayType.NORMAL:
                    break;
                default:
                    assertNotReached('Unhandled DisplayType encountered');
            }
            assert(this.displayType_ === DisplayType.NORMAL);
            const dangerType = this.data.dangerType;
            if (this.isSuspiciousEnterpriseApVerdict_(loadTimeData.getBoolean('requestsApVerdicts'), dangerType)) {
                return 'cr:warning';
            }
            switch (dangerType) {
                case DangerType.kDeepScannedFailed:
                    return 'cr:info';
                case DangerType.kSensitiveContentBlock:
                case DangerType.kBlockedTooLarge:
                case DangerType.kBlockedPasswordProtected:
                    return 'cr:error';
                case DangerType.kNoApplicableDangerType:
                case DangerType.kDangerousFile:
                case DangerType.kDangerousUrl:
                case DangerType.kDangerousContent:
                case DangerType.kCookieTheft:
                case DangerType.kUncommonContent:
                case DangerType.kDangerousHost:
                case DangerType.kPotentiallyUnwanted:
                case DangerType.kAsyncScanning:
                case DangerType.kAsyncLocalPasswordScanning:
                case DangerType.kSensitiveContentWarning:
                case DangerType.kDeepScannedSafe:
                case DangerType.kDeepScannedOpenedDangerous:
                case DangerType.kBlockedScanFailed:
                    break;
                default:
                    assertNotReached('Unhandled DangerType encountered');
            }
            switch (this.data.state) {
                case State.kAsyncScanning:
                case State.kPromptForScanning:
                case State.kPromptForLocalPasswordScanning:
                    return 'cr:warning';
                case State.kInProgress:
                case State.kCancelled:
                case State.kComplete:
                case State.kPaused:
                case State.kDangerous:
                case State.kInterrupted:
                case State.kInsecure:
                    break;
                default:
                    assertNotReached('Unhandled State encountered');
            }
        }
        if (this.isDangerous_) {
            return 'downloads:dangerous';
        }
        if (!this.useFileIcon_) {
            return 'cr:insert-drive-file';
        }
        return '';
    }
    computeIconColor_() {
        if (this.data) {
            return this.iconAndDescriptionColor_();
        }
        if (this.isDangerous_) {
            return 'red';
        }
        if (!this.useFileIcon_) {
            return 'light-grey';
        }
        return '';
    }
    computeIsActive_() {
        if (!this.data) {
            return true;
        }
        if (this.data.fileExternallyRemoved) {
            return false;
        }
        switch (this.data.state) {
            case State.kComplete:
            case State.kInProgress:
                return true;
            case State.kCancelled:
            case State.kInterrupted:
            case State.kPaused:
            case State.kDangerous:
            case State.kInsecure:
            case State.kAsyncScanning:
            case State.kPromptForScanning:
            case State.kPromptForLocalPasswordScanning:
                return false;
            default:
                assertNotReached('Unhandled State encountered');
        }
    }
    computeIsDangerous_() {
        if (!this.data) {
            return false;
        }
        switch (this.data.state) {
            case State.kDangerous:
            case State.kInsecure:
                return true;
            case State.kInProgress:
            case State.kCancelled:
            case State.kComplete:
            case State.kPaused:
            case State.kInterrupted:
            case State.kAsyncScanning:
            case State.kPromptForScanning:
            case State.kPromptForLocalPasswordScanning:
                return false;
            default:
                assertNotReached('Unhandled State encountered');
        }
    }
    computeIsReviewable_() {
        return !!this.data && this.data.isReviewable;
    }
    computePauseOrResumeText_() {
        if (this.data === undefined) {
            return '';
        }
        if (this.data.state === State.kInProgress) {
            return loadTimeData.getString('controlPause');
        }
        if (this.data.resume) {
            return loadTimeData.getString('controlResume');
        }
        return '';
    }
    computeShowRemove_() {
        const canDelete = loadTimeData.getBoolean('allowDeletingHistory');
        const hideRemove = this.isDangerous_ || this.showCancel_ || !canDelete;
        return !hideRemove;
    }
    computeRemoveStyle_() {
        return this.computeShowRemove_() ? '' : 'visibility: hidden';
    }
    computeShowControlsForDangerous_() {
        return !this.isReviewable_ && this.isDangerous_;
    }
    computeShowCancel_() {
        if (!this.data) {
            return false;
        }
        switch (this.data.state) {
            case State.kInProgress:
            case State.kPaused:
                return true;
            case State.kCancelled:
            case State.kComplete:
            case State.kDangerous:
            case State.kInterrupted:
            case State.kInsecure:
            case State.kAsyncScanning:
            case State.kPromptForScanning:
            case State.kPromptForLocalPasswordScanning:
                return false;
            default:
                assertNotReached('Unhandled State encountered');
        }
    }
    computeShowProgress_() {
        if (!this.data) {
            return false;
        }
        switch (this.data.state) {
            case State.kInProgress:
            case State.kCancelled:
            case State.kComplete:
            case State.kPaused:
            case State.kDangerous:
            case State.kInterrupted:
            case State.kInsecure:
                return this.showCancel_ && this.data.percent >= -1;
            case State.kAsyncScanning:
                return true;
            case State.kPromptForScanning:
            case State.kPromptForLocalPasswordScanning:
                return false;
            default:
                assertNotReached('Unhandled State encountered');
        }
    }
    computeShowDeepScan_() {
        if (!this.data) {
            return false;
        }
        switch (this.data.state) {
            case State.kPromptForScanning:
            case State.kPromptForLocalPasswordScanning:
                return true;
            case State.kInProgress:
            case State.kCancelled:
            case State.kComplete:
            case State.kPaused:
            case State.kDangerous:
            case State.kInterrupted:
            case State.kInsecure:
            case State.kAsyncScanning:
                return false;
            default:
                assertNotReached('Unhandled State encountered');
        }
    }
    computeShowOpenAnyway_() {
        if (!this.data) {
            return false;
        }
        switch (this.data.dangerType) {
            case DangerType.kDeepScannedFailed:
                return true;
            case DangerType.kNoApplicableDangerType:
            case DangerType.kDangerousFile:
            case DangerType.kDangerousUrl:
            case DangerType.kDangerousContent:
            case DangerType.kCookieTheft:
            case DangerType.kUncommonContent:
            case DangerType.kDangerousHost:
            case DangerType.kPotentiallyUnwanted:
            case DangerType.kAsyncScanning:
            case DangerType.kAsyncLocalPasswordScanning:
            case DangerType.kBlockedPasswordProtected:
            case DangerType.kBlockedTooLarge:
            case DangerType.kSensitiveContentWarning:
            case DangerType.kSensitiveContentBlock:
            case DangerType.kDeepScannedSafe:
            case DangerType.kDeepScannedOpenedDangerous:
            case DangerType.kBlockedScanFailed:
                return false;
            default:
                assertNotReached('Unhandled DangerType encountered');
        }
    }
    computeShowActionMenu_() {
        if (!this.data) {
            return false;
        }
        // If any of these actions are available, the action menu must be shown
        // because they don't have corresponding "quick actions".
        return !!this.pauseOrResumeText_ || // pause-or-resume
            this.computeShowControlsForDangerous_() || // save-dangerous
            this.data.retry || // retry
            this.showDeepScan_ || // deep-scan and bypass-deep-scan
            this.showCancel_ || // cancel
            this.showOpenAnyway_ || // open-anyway
            this.isReviewable_; // review-dangerous
    }
    computeShowCopyDownloadLink_() {
        return !!(this.data && this.data.url);
    }
    computeShowQuickRemove_() {
        return this.isReviewable_ || this.computeShowRemove_() ||
            this.computeShowControlsForDangerous_();
    }
    computeShowQuickShow_() {
        // Only show the quick "show in folder" button if the full action menu
        // is hidden. If the action menu is shown, hide the quick "show in folder"
        // button to save space since this action will have an entry in the menu
        // anyway.
        return this.computeHasShowInFolderLink_() && !this.computeShowActionMenu_();
    }
    computeTag_() {
        if (!this.data) {
            return '';
        }
        switch (this.data.state) {
            case State.kCancelled:
                return loadTimeData.getString('statusCancelled');
            case State.kInterrupted:
                return this.data.lastReasonText;
            case State.kComplete:
                return this.data.fileExternallyRemoved ?
                    loadTimeData.getString('statusRemoved') :
                    '';
            case State.kInProgress:
            case State.kPaused:
            case State.kDangerous:
            case State.kInsecure:
            case State.kAsyncScanning:
            case State.kPromptForScanning:
            case State.kPromptForLocalPasswordScanning:
                return '';
            default:
                assertNotReached('Unhandled State encountered');
        }
    }
    isIndeterminate_() {
        if (!this.data || this.data.percent === -1) {
            return true;
        }
        switch (this.data.state) {
            case State.kAsyncScanning:
                return true;
            case State.kInProgress:
            case State.kCancelled:
            case State.kComplete:
            case State.kPaused:
            case State.kDangerous:
            case State.kInterrupted:
            case State.kInsecure:
            case State.kPromptForScanning:
            case State.kPromptForLocalPasswordScanning:
                return false;
            default:
                assertNotReached('Unhandled State encountered');
        }
    }
    updateControlledBy_() {
        const controlledBy = this.computeControlledBy_();
        this.$['controlled-by'].innerHTML = sanitizeInnerHtml(controlledBy);
        if (controlledBy) {
            const link = this.shadowRoot.querySelector('#controlled-by a');
            link.setAttribute('focus-row-control', '');
            link.setAttribute('focus-type', 'controlledBy');
        }
    }
    computeInitiatorOriginText_() {
        if (!this.data || this.data.displayInitiatorOrigin.length === 0) {
            return '';
        }
        return loadTimeData.getStringF('initiatorLine', this.data.displayInitiatorOrigin);
    }
    updateUiForStateChange_() {
        const removeFileUrlLinks = () => {
            this.$.url.removeAttribute('href');
            this.$['file-link'].removeAttribute('href');
        };
        if (!this.data) {
            return;
        }
        // Returns whether to use the file icon, and additionally clears file url
        // links if necessary.
        const mayUseFileIcon = () => {
            const use = this.displayType_ === DisplayType.NORMAL;
            if (!use) {
                removeFileUrlLinks();
            }
            return use;
        };
        this.useFileIcon_ = mayUseFileIcon();
        if (!this.useFileIcon_) {
            return;
        }
        // The file is not dangerous. Link the url if supplied.
        if (this.data.url) {
            this.$.url.href = this.data.url.url;
        }
        else {
            removeFileUrlLinks();
        }
        const path = this.data.filePath;
        IconLoaderImpl.getInstance()
            .loadIcon(this.$['file-icon'], path)
            .then(success => {
            if (!!this.data && path === this.data.filePath &&
                this.data.state !== State.kAsyncScanning) {
                // Check again if we may use the file icon, to avoid a race between
                // loading the icon and determining the proper danger type.
                this.useFileIcon_ = mayUseFileIcon() && success;
            }
        });
    }
    // 
    onCopyDownloadLinkClick_(e) {
        if (!this.data || !this.data.url) {
            return;
        }
        let copied = true;
        navigator.clipboard.writeText(this.data.url.url)
            .catch(error => {
            console.error('Unable to copy to clipboard:', error);
            copied = false;
        })
            .finally(() => this.displayCopyToast_(e, copied));
    }
    onMoreActionsClick_() {
        const button = this.getMoreActionsButton();
        // The menu button is not always shown, but if this handler is invoked, then
        // it must be.
        assert(!!button);
        this.getMoreActionsMenu().showAt(button);
    }
    // Handles the "x" remove button which can be different actions depending on
    // the state of the download.
    onQuickRemoveClick_(e) {
        if (this.isReviewable_ || this.computeShowControlsForDangerous_()) {
            this.onDiscardDangerousClick_(e);
            return;
        }
        assert(this.computeShowRemove_());
        this.onRemoveClick_(e);
    }
    onCancelClick_() {
        this.restoreFocusAfterCancel_ = true;
        assert(!!this.mojoHandler_);
        this.mojoHandler_.cancel(this.dataId_());
        getInstance().announce(loadTimeData.getString('screenreaderCanceled'));
        this.getMoreActionsMenu().close();
    }
    onDiscardDangerousClick_(e) {
        assert(!!this.mojoHandler_);
        this.mojoHandler_.discardDangerous(this.dataId_());
        this.displayRemovedToast_(/*canUndo=*/ false, e);
        this.getMoreActionsMenu().close();
    }
    dataId_() {
        return this.data ? this.data.id : '';
    }
    onDeepScanClick_() {
        this.mojoHandler_.deepScan(this.dataId_());
        this.getMoreActionsMenu().close();
    }
    onBypassDeepScanClick_() {
        this.mojoHandler_.bypassDeepScanRequiringGesture(this.dataId_());
        this.getMoreActionsMenu().close();
    }
    onReviewDangerousClick_() {
        this.mojoHandler_.reviewDangerousRequiringGesture(this.dataId_());
        this.getMoreActionsMenu().close();
    }
    onOpenAnywayClick_() {
        this.mojoHandler_.openFileRequiringGesture(this.dataId_());
        this.getMoreActionsMenu().close();
    }
    onDragStart_(e) {
        e.preventDefault();
        this.mojoHandler_.drag(this.dataId_());
    }
    onFileLinkClick_(e) {
        e.preventDefault();
        this.mojoHandler_.openFileRequiringGesture(this.dataId_());
    }
    onUrlClick_() {
        if (!this.data || !this.data.url) {
            return;
        }
        chrome.send('metricsHandler:recordAction', ['Downloads_OpenUrlOfDownloadedItem']);
    }
    doPause_() {
        assert(!!this.mojoHandler_);
        this.mojoHandler_.pause(this.dataId_());
        getInstance().announce(loadTimeData.getString('screenreaderPaused'));
    }
    doResume_() {
        assert(!!this.mojoHandler_);
        this.mojoHandler_.resume(this.dataId_());
        getInstance().announce(loadTimeData.getString('screenreaderResumed'));
    }
    onPauseOrResumeClick_() {
        if (this.data && this.data.state === State.kInProgress) {
            this.doPause_();
        }
        else {
            this.doResume_();
        }
        this.getMoreActionsMenu().close();
    }
    displayCopyToast_(e, copied) {
        if (!this.data || !this.data.url) {
            return;
        }
        if (copied) {
            const pieces = loadTimeData.getSubstitutedStringPieces(loadTimeData.getString('toastCopiedDownloadLink'), this.data.url.url);
            pieces.forEach(p => {
                p.collapsible = !!p.arg;
            });
            getToastManager().showForStringPieces(pieces, /*hideSlotted=*/ true);
        }
        else {
            getToastManager().show(loadTimeData.getString('toastCopyDownloadLinkFailed'), 
            /*hideSlotted=*/ true);
        }
        e.stopPropagation();
        e.preventDefault();
    }
    displayRemovedToast_(canUndo, e) {
        const templateStringId = (this.displayType_ === DisplayType.NORMAL && this.completelyOnDisk_) ?
            'toastDeletedFromHistoryStillOnDevice' :
            'toastDeletedFromHistory';
        const filename = this.data ? this.data.fileName : '';
        const pieces = loadTimeData.getSubstitutedStringPieces(loadTimeData.getString(templateStringId), filename);
        pieces.forEach(p => {
            // Make the file name collapsible.
            p.collapsible = !!p.arg;
        });
        getToastManager().showForStringPieces(pieces, /*hideSlotted=*/ !canUndo);
        // Stop propagating a click to the document to remove toast.
        e.stopPropagation();
        e.preventDefault();
    }
    onRemoveClick_(e) {
        assert(!!this.mojoHandler_);
        assert(this.data);
        this.mojoHandler_.remove(this.data.id);
        const canUndo = !this.data.isDangerous && !this.data.isInsecure;
        this.displayRemovedToast_(canUndo, e);
        this.getMoreActionsMenu().close();
    }
    onRetryClick_() {
        this.mojoHandler_.retryDownload(this.dataId_());
        this.getMoreActionsMenu().close();
    }
    notifySaveDangerousClick_() {
        this.dispatchEvent(new CustomEvent('save-dangerous-click', {
            bubbles: true,
            composed: true,
            detail: { id: this.dataId_() },
        }));
    }
    onSaveDangerousClick_() {
        this.getMoreActionsMenu().close();
        if (this.displayType_ === DisplayType.DANGEROUS) {
            this.notifySaveDangerousClick_();
            return;
        }
        // "Suspicious" types which show up in grey can be validated directly.
        // This maps each such display type to its applicable screenreader
        // announcement string id.
        const SAVED_FROM_PAGE_TYPES_ANNOUNCEMENTS = new Map([
            [DisplayType.SUSPICIOUS, 'screenreaderSavedSuspicious'],
            [DisplayType.UNVERIFIED, 'screenreaderSavedUnverified'],
            [DisplayType.INSECURE, 'screenreaderSavedInsecure'],
        ]);
        assert(SAVED_FROM_PAGE_TYPES_ANNOUNCEMENTS.has(this.displayType_));
        assert(this.data);
        assert(!!this.mojoHandler_);
        this.mojoHandler_.saveSuspiciousRequiringGesture(this.data.id);
        const announcement = loadTimeData.getString(SAVED_FROM_PAGE_TYPES_ANNOUNCEMENTS.get(this.displayType_));
        getInstance().announce(announcement);
    }
    onShowClick_() {
        assert(this.data);
        this.mojoHandler_.show(this.data.id);
        this.getMoreActionsMenu().close();
    }
    restoreFocusAfterCancelIfNeeded_() {
        if (!this.restoreFocusAfterCancel_) {
            return;
        }
        this.restoreFocusAfterCancel_ = false;
        setTimeout(() => {
            const element = this.getFocusRow().getFirstFocusable('retry');
            if (element) {
                element.focus();
            }
        });
    }
}
customElements.define(DownloadsItemElement.is, DownloadsItemElement);

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Helper functions for implementing an incremental search field. See
 * <settings-subpage-search> for a simple implementation.
 */
const CrSearchFieldMixinLit = (superClass) => {
    class CrSearchFieldMixinLit extends superClass {
        static get properties() {
            return {
                // Prompt text to display in the search field.
                label: {
                    type: String,
                },
                // Tooltip to display on the clear search button.
                clearLabel: {
                    type: String,
                },
                hasSearchText: {
                    type: Boolean,
                    reflect: true,
                },
            };
        }
        #label_accessor_storage = '';
        get label() { return this.#label_accessor_storage; }
        set label(value) { this.#label_accessor_storage = value; }
        #clearLabel_accessor_storage = '';
        get clearLabel() { return this.#clearLabel_accessor_storage; }
        set clearLabel(value) { this.#clearLabel_accessor_storage = value; }
        #hasSearchText_accessor_storage = false;
        get hasSearchText() { return this.#hasSearchText_accessor_storage; }
        set hasSearchText(value) { this.#hasSearchText_accessor_storage = value; }
        effectiveValue_ = '';
        searchDelayTimer_ = -1;
        /**
         * @return The input field element the behavior should use.
         */
        getSearchInput() {
            assertNotReached();
        }
        /**
         * @return The value of the search field.
         */
        getValue() {
            return this.getSearchInput().value;
        }
        /**
         * Sets the value of the search field.
         * @param noEvent Whether to prevent a 'search-changed' event
         *     firing for this change.
         */
        setValue(value, noEvent) {
            const updated = this.updateEffectiveValue_(value);
            this.getSearchInput().value = this.effectiveValue_;
            if (!updated) {
                // If the input is only whitespace and value is empty,
                // |hasSearchText| needs to be updated.
                if (value === '' && this.hasSearchText) {
                    this.hasSearchText = false;
                }
                return;
            }
            this.onSearchTermInput();
            if (!noEvent) {
                this.fire('search-changed', this.effectiveValue_);
            }
        }
        scheduleSearch_() {
            if (this.searchDelayTimer_ >= 0) {
                clearTimeout(this.searchDelayTimer_);
            }
            // Dispatch 'search' event after:
            //    0ms if the value is empty
            //  500ms if the value length is 1
            //  400ms if the value length is 2
            //  300ms if the value length is 3
            //  200ms if the value length is 4 or greater.
            // The logic here was copied from WebKit's native 'search' event.
            const length = this.getValue().length;
            const timeoutMs = length > 0 ? (500 - 100 * (Math.min(length, 4) - 1)) : 0;
            this.searchDelayTimer_ = setTimeout(() => {
                this.getSearchInput().dispatchEvent(new CustomEvent('search', { composed: true, detail: this.getValue() }));
                this.searchDelayTimer_ = -1;
            }, timeoutMs);
        }
        onSearchTermSearch() {
            this.onValueChanged_(this.getValue(), false);
        }
        /**
         * Update the state of the search field whenever the underlying input
         * value changes. Unlike onsearch or onkeypress, this is reliably called
         * immediately after any change, whether the result of user input or JS
         * modification.
         */
        onSearchTermInput() {
            this.hasSearchText = this.getSearchInput().value !== '';
            this.scheduleSearch_();
        }
        /**
         * Updates the internal state of the search field based on a change that
         * has already happened.
         * @param noEvent Whether to prevent a 'search-changed' event
         *     firing for this change.
         */
        onValueChanged_(newValue, noEvent) {
            const updated = this.updateEffectiveValue_(newValue);
            if (updated && !noEvent) {
                this.fire('search-changed', this.effectiveValue_);
            }
        }
        /**
         * Trim leading whitespace and replace consecutive whitespace with
         * single space. This will prevent empty string searches and searches
         * for effectively the same query.
         */
        updateEffectiveValue_(value) {
            const effectiveValue = value.replace(/\s+/g, ' ').replace(/^\s/, '');
            if (effectiveValue === this.effectiveValue_) {
                return false;
            }
            this.effectiveValue_ = effectiveValue;
            return true;
        }
    }
    return CrSearchFieldMixinLit;
};

let instance$a = null;
function getCss$9() {
    return instance$a || (instance$a = [...[getCss$o(), getCss$k()], css `[actionable]{cursor:pointer}.hr{border-top:var(--cr-separator-line)}iron-list.cr-separators>*:not([first]){border-top:var(--cr-separator-line)}[scrollable]{border-color:transparent;border-style:solid;border-width:1px 0;overflow-y:auto}[scrollable].is-scrolled{border-top-color:var(--cr-scrollable-border-color)}[scrollable].can-scroll:not(.scrolled-to-bottom){border-bottom-color:var(--cr-scrollable-border-color)}[scrollable] iron-list>:not(.no-outline):focus-visible,[selectable]:focus-visible,[selectable]>:focus-visible{outline:solid 2px var(--cr-focus-outline-color);outline-offset:-2px}.scroll-container{display:flex;flex-direction:column;min-height:1px}[selectable]>*{cursor:pointer}.cr-centered-card-container{box-sizing:border-box;display:block;height:inherit;margin:0 auto;max-width:var(--cr-centered-card-max-width);min-width:550px;position:relative;width:calc(100% * var(--cr-centered-card-width-percentage))}.cr-container-shadow{box-shadow:inset 0 5px 6px -3px rgba(0,0,0,.4);height:var(--cr-container-shadow-height);left:0;margin:0 0 var(--cr-container-shadow-margin);opacity:0;pointer-events:none;position:relative;right:0;top:0;transition:opacity 500ms;z-index:1}#cr-container-shadow-bottom{margin-bottom:0;margin-top:var(--cr-container-shadow-margin);transform:scaleY(-1)}#cr-container-shadow-top:has(+#container.can-scroll:not(.scrolled-to-top)),#container.can-scroll:not(.scrolled-to-bottom)+#cr-container-shadow-bottom,#cr-container-shadow-bottom.force-shadow,#cr-container-shadow-top.force-shadow{opacity:var(--cr-container-shadow-max-opacity)}.cr-row{align-items:center;border-top:var(--cr-separator-line);display:flex;min-height:var(--cr-section-min-height);padding:0 var(--cr-section-padding)}.cr-row.first,.cr-row.continuation{border-top:none}.cr-row-gap{padding-inline-start:16px}.cr-button-gap{margin-inline-start:8px}paper-tooltip::part(tooltip),cr-tooltip::part(tooltip){border-radius:var(--paper-tooltip-border-radius,2px);font-size:92.31%;font-weight:500;max-width:330px;min-width:var(--paper-tooltip-min-width,200px);padding:var(--paper-tooltip-padding,10px 8px)}.cr-padded-text{padding-block-end:var(--cr-section-vertical-padding);padding-block-start:var(--cr-section-vertical-padding)}.cr-title-text{color:var(--cr-title-text-color);font-size:107.6923%;font-weight:500}.cr-secondary-text{color:var(--cr-secondary-text-color);font-weight:400}.cr-form-field-label{color:var(--cr-form-field-label-color);display:block;font-size:var(--cr-form-field-label-font-size);font-weight:500;letter-spacing:.4px;line-height:var(--cr-form-field-label-line-height);margin-bottom:8px}.cr-vertical-tab{align-items:center;display:flex}.cr-vertical-tab::before{border-radius:0 3px 3px 0;content:'';display:block;flex-shrink:0;height:var(--cr-vertical-tab-height,100%);width:4px}.cr-vertical-tab.selected::before{background:var(--cr-vertical-tab-selected-color,var(--cr-checked-color))}:host-context([dir=rtl]) .cr-vertical-tab::before{transform:scaleX(-1)}.iph-anchor-highlight{background-color:var(--cr-iph-anchor-highlight-color)}`]);
}

let instance$9 = null;
function getCss$8() {
    return instance$9 || (instance$9 = [...[], css `.spinner{--cr-spinner-size:28px;mask-image:url(//resources/images/throbber_small.svg);mask-position:center;mask-repeat:no-repeat;mask-size:var(--cr-spinner-size) var(--cr-spinner-size);background-color:var(--cr-spinner-color,var(--google-blue-500));height:var(--cr-spinner-size);width:var(--cr-spinner-size)}@media (prefers-color-scheme:dark){.spinner{background-color:var(--cr-spinner-color,var(--google-blue-300))}}`]);
}

let instance$8 = null;
function getCss$7() {
    return instance$8 || (instance$8 = [...[getCss$9(), getCss$k(), getCss$8()], css `:host{display:block;height:40px;isolation:isolate;transition:background-color 150ms cubic-bezier(0.4,0,0.2,1),width 150ms cubic-bezier(0.4,0,0.2,1);width:44px}:host([disabled]){opacity:var(--cr-disabled-opacity)}[hidden]{display:none !important}@media (prefers-color-scheme:light){cr-icon-button{--cr-icon-button-fill-color:var(--cr-toolbar-search-field-input-icon-color,var(--google-grey-700));--cr-icon-button-focus-outline-color:var(--cr-toolbar-icon-button-focus-outline-color,var(--cr-focus-outline-color))}}@media (prefers-color-scheme:dark){cr-icon-button{--cr-icon-button-fill-color:var(--cr-toolbar-search-field-input-icon-color,var(--google-grey-500))}}cr-icon-button{--cr-icon-button-fill-color:var(--cr-toolbar-search-field-icon-color,var(--color-toolbar-search-field-icon,var(--cr-secondary-text-color)));--cr-icon-button-size:var(--cr-toolbar-icon-container-size,28px);--cr-icon-button-icon-size:20px;margin:var(--cr-toolbar-icon-margin,0)}#icon{transition:margin 150ms,opacity 200ms}#prompt{color:var(--cr-toolbar-search-field-prompt-color,var(--color-toolbar-search-field-foreground-placeholder,var(--cr-secondary-text-color)));opacity:0}@media (prefers-color-scheme:dark){#prompt{color:var(--cr-toolbar-search-field-prompt-color,white)}}@media (prefers-color-scheme:dark){#prompt{--cr-toolbar-search-field-prompt-opacity:1;color:var(--cr-secondary-text-color,white)}}.spinner{--cr-spinner-color:var(--cr-toolbar-search-field-input-icon-color,var(--google-grey-700));--cr-spinner-size:var(--cr-icon-size);margin:0;opacity:1;padding:2px;position:absolute}@media (prefers-color-scheme:dark){.spinner{--cr-spinner-color:var(--cr-toolbar-search-field-input-icon-color,white)}}#prompt{transition:opacity 200ms}#searchTerm{-webkit-font-smoothing:antialiased;flex:1;font-size:12px;font-weight:500;line-height:185%;margin:var(--cr-toolbar-search-field-term-margin,0);position:relative}label{bottom:0;cursor:var(--cr-toolbar-search-field-cursor,text);left:0;overflow:hidden;position:absolute;right:0;top:0;white-space:nowrap}:host([has-search-text]) label{visibility:hidden}input{-webkit-appearance:none;background:transparent;border:none;caret-color:var(--cr-toolbar-search-field-input-caret-color,currentColor);color:var(--cr-toolbar-search-field-input-text-color,var(--color-toolbar-search-field-foreground,var(--cr-fallback-color-on-surface)));font:inherit;font-size:12px;font-weight:500;outline:none;padding:0;position:relative;width:100%}@media (prefers-color-scheme:dark){input{color:var(--cr-toolbar-search-field-input-text-color,white)}}input[type='search']::-webkit-search-cancel-button{display:none}:host([narrow]){border-radius:var(--cr-toolbar-search-field-border-radius,0)}:host(:not([narrow])){background:none;border-radius:var(--cr-toolbar-search-field-border-radius,46px);cursor:var(--cr-toolbar-search-field-cursor,default);height:36px;max-width:var(--cr-toolbar-field-max-width,none);overflow:hidden;padding:0 6px;position:relative;width:var(--cr-toolbar-field-width,680px);--cr-toolbar-search-field-border-radius:100px}@media (prefers-color-scheme:dark){:host(:not([narrow])){background:var(--cr-toolbar-search-field-background,rgba(0,0,0,0.22))}}#background,#stateBackground{display:none}:host(:not([narrow])) #background{background:var(--cr-toolbar-search-field-background,var(--color-toolbar-search-field-background,var(--cr-fallback-color-base-container)));border-radius:inherit;display:block;inset:0;pointer-events:none;position:absolute;z-index:0}:host([search-focused_]:not([narrow])){outline:2px solid var(--cr-focus-outline-color);outline-offset:2px}:host(:not([narrow])) #stateBackground{display:block;inset:0;pointer-events:none;position:absolute}:host(:hover:not([search-focused_],[narrow])) #stateBackground{background:var(--color-toolbar-search-field-background-hover,var(--cr-hover-background-color));z-index:1}:host(:not([narrow]):not([showing-search])) #icon{opacity:var(--cr-toolbar-search-field-icon-opacity,1)}:host(:not([narrow])) #prompt{opacity:var(--cr-toolbar-search-field-prompt-opacity,1)}:host([narrow]) #prompt{opacity:var(--cr-toolbar-search-field-narrow-mode-prompt-opacity,0)}:host([narrow]:not([showing-search])) #searchTerm{display:none}:host([showing-search][spinner-active]) #icon{opacity:0}:host([narrow][showing-search]){width:100%}:host([narrow][showing-search]) #icon,:host([narrow][showing-search]) .spinner{margin-inline-start:var(--cr-toolbar-search-icon-margin-inline-start,18px)}#content{align-items:center;display:flex;height:100%;position:relative;z-index:2}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$4() {
    // clang-format off
    return html `
<div id="background"></div>
<div id="stateBackground"></div>
<div id="content">
  ${this.shouldShowSpinner_() ? html `
    <div class="spinner"></div>` : ''}
    <cr-icon-button id="icon" iron-icon="${this.iconOverride || 'cr:search'}"
        title="${this.label}" tabindex="${this.getIconTabIndex_()}"
        aria-hidden="${this.getIconAriaHidden_()}" suppress-rtl-flip
        @click="${this.onSearchIconClicked_}" ?disabled="${this.disabled}">
  </cr-icon-button>
  <div id="searchTerm">
    <label id="prompt" for="searchInput" aria-hidden="true">
      ${this.label}
    </label>
    <input id="searchInput"
        aria-labelledby="prompt"
        aria-description="${this.inputAriaDescription}"
        autocapitalize="off"
        autocomplete="off"
        type="search"
        @beforeinput="${this.onSearchTermNativeBeforeInput}"
        @input="${this.onSearchTermNativeInput}"
        @search="${this.onSearchTermSearch}"
        @keydown="${this.onSearchTermKeydown_}"
        @focus="${this.onInputFocus_}"
        @blur="${this.onInputBlur_}"
        ?autofocus="${this.autofocus}"
        spellcheck="false"
        ?disabled="${this.disabled}">
  </div>
  ${this.hasSearchText ? html `
    <cr-icon-button id="clearSearch" iron-icon="cr:cancel"
        title="${this.clearLabel}" @click="${this.clearSearch_}"
        ?disabled="${this.disabled}"></cr-icon-button>` : ''}
</div>`;
    // clang-format on
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const CrToolbarSearchFieldElementBase = CrSearchFieldMixinLit(CrLitElement);
class CrToolbarSearchFieldElement extends CrToolbarSearchFieldElementBase {
    static get is() {
        return 'cr-toolbar-search-field';
    }
    static get styles() {
        return getCss$7();
    }
    render() {
        return getHtml$4.bind(this)();
    }
    static get properties() {
        return {
            narrow: {
                type: Boolean,
                reflect: true,
            },
            showingSearch: {
                type: Boolean,
                notify: true,
                reflect: true,
            },
            disabled: {
                type: Boolean,
                reflect: true,
            },
            autofocus: {
                type: Boolean,
                reflect: true,
            },
            // When true, show a loading spinner to indicate that the backend is
            // processing the search. Will only show if the search field is open.
            spinnerActive: {
                type: Boolean,
                reflect: true,
            },
            searchFocused_: {
                type: Boolean,
                reflect: true,
            },
            iconOverride: { type: String },
            inputAriaDescription: { type: String },
        };
    }
    #narrow_accessor_storage = false;
    get narrow() { return this.#narrow_accessor_storage; }
    set narrow(value) { this.#narrow_accessor_storage = value; }
    #showingSearch_accessor_storage = false;
    get showingSearch() { return this.#showingSearch_accessor_storage; }
    set showingSearch(value) { this.#showingSearch_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #autofocus_accessor_storage = false;
    get autofocus() { return this.#autofocus_accessor_storage; }
    set autofocus(value) { this.#autofocus_accessor_storage = value; }
    #spinnerActive_accessor_storage = false;
    get spinnerActive() { return this.#spinnerActive_accessor_storage; }
    set spinnerActive(value) { this.#spinnerActive_accessor_storage = value; }
    #searchFocused__accessor_storage = false;
    get searchFocused_() { return this.#searchFocused__accessor_storage; }
    set searchFocused_(value) { this.#searchFocused__accessor_storage = value; }
    #iconOverride_accessor_storage;
    get iconOverride() { return this.#iconOverride_accessor_storage; }
    set iconOverride(value) { this.#iconOverride_accessor_storage = value; }
    #inputAriaDescription_accessor_storage = '';
    get inputAriaDescription() { return this.#inputAriaDescription_accessor_storage; }
    set inputAriaDescription(value) { this.#inputAriaDescription_accessor_storage = value; }
    firstUpdated() {
        this.addEventListener('click', e => this.showSearch_(e));
    }
    getSearchInput() {
        return this.$.searchInput;
    }
    isSearchFocused() {
        return this.searchFocused_;
    }
    async showAndFocus() {
        this.showingSearch = true;
        await this.updateComplete;
        this.focus_();
    }
    onSearchTermNativeBeforeInput(e) {
        this.fire('search-term-native-before-input', { e });
    }
    onSearchTermInput() {
        super.onSearchTermInput();
        this.showingSearch = this.hasSearchText || this.isSearchFocused();
    }
    onSearchTermNativeInput(e) {
        this.onSearchTermInput();
        this.fire('search-term-native-input', { e, inputValue: this.getValue() });
    }
    getIconTabIndex_() {
        return this.narrow && !this.hasSearchText ? 0 : -1;
    }
    getIconAriaHidden_() {
        return Boolean(!this.narrow || this.hasSearchText).toString();
    }
    shouldShowSpinner_() {
        return this.spinnerActive && this.showingSearch;
    }
    onSearchIconClicked_() {
        this.fire('search-icon-clicked');
    }
    focus_() {
        this.getSearchInput().focus();
    }
    onInputFocus_() {
        this.searchFocused_ = true;
    }
    onInputBlur_() {
        this.searchFocused_ = false;
        if (!this.hasSearchText) {
            this.showingSearch = false;
        }
    }
    onSearchTermKeydown_(e) {
        if (e.key === 'Escape') {
            this.showingSearch = false;
            this.setValue('');
            this.getSearchInput().blur();
        }
    }
    async showSearch_(e) {
        if (e.target !== this.shadowRoot.querySelector('#clearSearch')) {
            this.showingSearch = true;
        }
        if (this.narrow) {
            await this.updateComplete; // Wait for input to become focusable.
            this.focus_();
        }
    }
    clearSearch_() {
        this.setValue('');
        this.focus_();
        this.spinnerActive = false;
        this.fire('search-term-cleared');
    }
}
customElements.define(CrToolbarSearchFieldElement.is, CrToolbarSearchFieldElement);

let instance$7 = null;
function getCss$6() {
    return instance$7 || (instance$7 = [...[getCss$o(), getCss$k()], css `:host{align-items:center;box-sizing:border-box;color:var(--google-grey-900);display:flex;height:var(--cr-toolbar-height)}@media (prefers-color-scheme:dark){:host{color:var(--cr-secondary-text-color)}}h1{flex:1;font-size:170%;font-weight:var(--cr-toolbar-header-font-weight,500);letter-spacing:.25px;line-height:normal;margin-inline-start:6px;padding-inline-end:12px;white-space:var(--cr-toolbar-header-white-space,normal)}@media (prefers-color-scheme:dark){h1{color:var(--cr-primary-text-color)}}#leftContent{position:relative;transition:opacity 100ms}#leftSpacer{align-items:center;box-sizing:border-box;display:flex;padding-inline-start:calc(12px + 6px);width:var(--cr-toolbar-left-spacer-width,auto)}cr-icon-button{--cr-icon-button-size:32px;min-width:32px}@media (prefers-color-scheme:light){cr-icon-button{--cr-icon-button-fill-color:currentColor;--cr-icon-button-focus-outline-color:var(--cr-focus-outline-color)}}#centeredContent{display:flex;flex:1 1 0;justify-content:center}#rightSpacer{padding-inline-end:12px}:host([narrow]) #centeredContent{justify-content:flex-end}:host([has-overlay]){transition:visibility var(--cr-toolbar-overlay-animation-duration);visibility:hidden}:host([narrow][showing-search_]) #leftContent{opacity:0;position:absolute}:host(:not([narrow])) #leftContent{flex:1 1 var(--cr-toolbar-field-margin,0)}:host(:not([narrow])) #centeredContent{flex-basis:var(--cr-toolbar-center-basis,0)}:host(:not([narrow])[disable-right-content-grow]) #centeredContent{justify-content:start;padding-inline-start:12px}:host(:not([narrow])) #rightContent{flex:1 1 0;text-align:end}:host(:not([narrow])[disable-right-content-grow]) #rightContent{flex:0 1 0}picture{display:none}#menuButton{margin-inline-end:9px}#menuButton~h1{margin-inline-start:0}:host([always-show-logo]) picture,:host(:not([narrow])) picture{display:initial;margin-inline-end:16px}:host([always-show-logo]) #leftSpacer,:host(:not([narrow])) #leftSpacer{padding-inline-start:calc(12px + 9px)}:host([always-show-logo]) :is(picture,#product-logo),:host(:not([narrow])) :is(picture,#product-logo){height:24px;width:24px}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$3() {
    // clang-format off
    return html `
<div id="leftContent">
  <div id="leftSpacer">
    ${this.showMenu ? html `
      <cr-icon-button id="menuButton" class="no-overlap"
          iron-icon="cr20:menu" @click="${this.onMenuClick_}"
          aria-label="${this.menuLabel || nothing}"
          title="${this.menuLabel}">
      </cr-icon-button>` : ''}
    <slot name="product-logo">
      <picture>
        <source media="(prefers-color-scheme: dark)"
            srcset="//resources/images/chrome_logo_dark.svg">
        <img id="product-logo"
            srcset="chrome://theme/current-channel-logo@1x 1x,
                    chrome://theme/current-channel-logo@2x 2x"
            role="presentation">
      </picture>
    </slot>
    <h1>${this.pageName}</h1>
  </div>
</div>

<div id="centeredContent" ?hidden="${!this.showSearch}">
  <cr-toolbar-search-field id="search" ?narrow="${this.narrow}"
      label="${this.searchPrompt}" clear-label="${this.clearLabel}"
      ?spinner-active="${this.spinnerActive}"
      ?showing-search="${this.showingSearch_}"
      @showing-search-changed="${this.onShowingSearchChanged_}"
      ?autofocus="${this.autofocus}" icon-override="${this.searchIconOverride}"
      input-aria-description="${this.searchInputAriaDescription}">
  </cr-toolbar-search-field>
</div>

<div id="rightContent">
  <div id="rightSpacer">
    <slot></slot>
  </div>
</div>`;
    // clang-format on
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class CrToolbarElement extends CrLitElement {
    static get is() {
        return 'cr-toolbar';
    }
    static get styles() {
        return getCss$6();
    }
    render() {
        return getHtml$3.bind(this)();
    }
    static get properties() {
        return {
            // Name to display in the toolbar, in titlecase.
            pageName: { type: String },
            // Prompt text to display in the search field.
            searchPrompt: { type: String },
            // Tooltip to display on the clear search button.
            clearLabel: { type: String },
            // Tooltip to display on the menu button.
            menuLabel: { type: String },
            // Value is proxied through to cr-toolbar-search-field. When true,
            // the search field will show a processing spinner.
            spinnerActive: { type: Boolean },
            // Controls whether the menu button is shown at the start of the menu.
            showMenu: { type: Boolean },
            // Controls whether the search field is shown.
            showSearch: { type: Boolean },
            // Controls whether the search field is autofocused.
            autofocus: {
                type: Boolean,
                reflect: true,
            },
            // True when the toolbar is displaying in narrow mode.
            narrow: {
                type: Boolean,
                reflect: true,
                notify: true,
            },
            /**
             * The threshold at which the toolbar will change from normal to narrow
             * mode, in px.
             */
            narrowThreshold: {
                type: Number,
            },
            alwaysShowLogo: {
                type: Boolean,
                reflect: true,
            },
            showingSearch_: {
                type: Boolean,
                reflect: true,
            },
            searchIconOverride: { type: String },
            searchInputAriaDescription: { type: String },
        };
    }
    #pageName_accessor_storage = '';
    get pageName() { return this.#pageName_accessor_storage; }
    set pageName(value) { this.#pageName_accessor_storage = value; }
    #searchPrompt_accessor_storage = '';
    get searchPrompt() { return this.#searchPrompt_accessor_storage; }
    set searchPrompt(value) { this.#searchPrompt_accessor_storage = value; }
    #clearLabel_accessor_storage = '';
    get clearLabel() { return this.#clearLabel_accessor_storage; }
    set clearLabel(value) { this.#clearLabel_accessor_storage = value; }
    #menuLabel_accessor_storage;
    get menuLabel() { return this.#menuLabel_accessor_storage; }
    set menuLabel(value) { this.#menuLabel_accessor_storage = value; }
    #spinnerActive_accessor_storage = false;
    get spinnerActive() { return this.#spinnerActive_accessor_storage; }
    set spinnerActive(value) { this.#spinnerActive_accessor_storage = value; }
    #showMenu_accessor_storage = false;
    get showMenu() { return this.#showMenu_accessor_storage; }
    set showMenu(value) { this.#showMenu_accessor_storage = value; }
    #showSearch_accessor_storage = true;
    get showSearch() { return this.#showSearch_accessor_storage; }
    set showSearch(value) { this.#showSearch_accessor_storage = value; }
    #autofocus_accessor_storage = false;
    get autofocus() { return this.#autofocus_accessor_storage; }
    set autofocus(value) { this.#autofocus_accessor_storage = value; }
    #narrow_accessor_storage = false;
    get narrow() { return this.#narrow_accessor_storage; }
    set narrow(value) { this.#narrow_accessor_storage = value; }
    #narrowThreshold_accessor_storage = 900;
    get narrowThreshold() { return this.#narrowThreshold_accessor_storage; }
    set narrowThreshold(value) { this.#narrowThreshold_accessor_storage = value; }
    #alwaysShowLogo_accessor_storage = false;
    get alwaysShowLogo() { return this.#alwaysShowLogo_accessor_storage; }
    set alwaysShowLogo(value) { this.#alwaysShowLogo_accessor_storage = value; }
    #showingSearch__accessor_storage = false;
    get showingSearch_() { return this.#showingSearch__accessor_storage; }
    set showingSearch_(value) { this.#showingSearch__accessor_storage = value; }
    #searchIconOverride_accessor_storage;
    get searchIconOverride() { return this.#searchIconOverride_accessor_storage; }
    set searchIconOverride(value) { this.#searchIconOverride_accessor_storage = value; }
    #searchInputAriaDescription_accessor_storage = '';
    get searchInputAriaDescription() { return this.#searchInputAriaDescription_accessor_storage; }
    set searchInputAriaDescription(value) { this.#searchInputAriaDescription_accessor_storage = value; }
    narrowQuery_ = null;
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('narrowThreshold')) {
            this.narrowQuery_ =
                window.matchMedia(`(max-width: ${this.narrowThreshold}px)`);
            this.narrow = this.narrowQuery_.matches;
            this.narrowQuery_.addListener(() => this.onQueryChanged_());
        }
    }
    getSearchField() {
        return this.$.search;
    }
    onMenuClick_() {
        this.fire('cr-toolbar-menu-click');
    }
    async focusMenuButton() {
        assert(this.showMenu);
        // Wait for rendering to finish to ensure menuButton exists on the DOM.
        await this.updateComplete;
        const menuButton = this.shadowRoot.querySelector('#menuButton');
        assert(!!menuButton);
        menuButton.focus();
    }
    isMenuFocused() {
        return !!this.shadowRoot.activeElement &&
            this.shadowRoot.activeElement.id === 'menuButton';
    }
    onShowingSearchChanged_(e) {
        this.showingSearch_ = e.detail.value;
    }
    onQueryChanged_() {
        assert(this.narrowQuery_);
        this.narrow = this.narrowQuery_.matches;
    }
}
customElements.define(CrToolbarElement.is, CrToolbarElement);

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class SearchService {
    searchTerms_ = [];
    mojoHandler_ = BrowserProxy.getInstance().handler;
    /**
     * @param searchText Input typed by the user into a search box.
     * @return A list of terms extracted from |searchText|.
     */
    static splitTerms(searchText) {
        // Split quoted terms (e.g., 'The "lazy" dog' => ['The', 'lazy', 'dog']).
        return searchText.split(/"([^"]*)"/).map(s => s.trim()).filter(s => !!s);
    }
    /** Instructs the browser to clear all finished downloads. */
    clearAll() {
        if (loadTimeData.getBoolean('allowDeletingHistory')) {
            this.mojoHandler_.clearAll();
            this.search('');
        }
    }
    /** Loads more downloads with the current search terms. */
    loadMore() {
        this.mojoHandler_.getDownloads(this.searchTerms_);
    }
    /**
     * @return Whether the user is currently searching for downloads
     *     (i.e. has a non-empty search term).
     */
    isSearching() {
        return this.searchTerms_.length > 0;
    }
    /**
     * @param searchText What to search for.
     * @return Whether |searchText| resulted in new search terms.
     */
    search(searchText) {
        const searchTerms = SearchService.splitTerms(searchText);
        let sameTerms = searchTerms.length === this.searchTerms_.length;
        for (let i = 0; sameTerms && i < searchTerms.length; ++i) {
            if (searchTerms[i] !== this.searchTerms_[i]) {
                sameTerms = false;
            }
        }
        if (sameTerms) {
            return false;
        }
        this.searchTerms_ = searchTerms;
        this.loadMore();
        return true;
    }
    static getInstance() {
        return instance$6 || (instance$6 = new SearchService());
    }
    static setInstance(obj) {
        instance$6 = obj;
    }
}
let instance$6 = null;

let instance$5 = null;
function getCss$5() {
    return instance$5 || (instance$5 = [...[getCss$o()], css `:host{align-items:center;display:flex;min-height:56px}#toolbar{flex:1;min-width:0}#clearAll{white-space:nowrap}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$2() {
    return html `<!--_html_template_start_-->
<cr-toolbar id="toolbar" page-name="$i18n{title}" autofocus always-show-logo
    search-prompt="$i18n{search}" clear-label="$i18n{clearSearch}"
    .spinnerActive="${this.spinnerActive}"
    @search-changed="${this.onSearchChanged_}">
  <cr-button id="clearAll" @click="${this.onClearAllClick_}">
    $i18n{clearAll}
  </cr-button>
</cr-toolbar>
<!--_html_template_end_-->`;
}

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class DownloadsToolbarElement extends CrLitElement {
    static get is() {
        return 'downloads-toolbar';
    }
    static get styles() {
        return getCss$5();
    }
    render() {
        return getHtml$2.bind(this)();
    }
    static get properties() {
        return {
            hasClearableDownloads: { type: Boolean },
            items: { type: Array },
            spinnerActive: { type: Boolean },
        };
    }
    mojoHandler_ = null;
    #hasClearableDownloads_accessor_storage = false;
    get hasClearableDownloads() { return this.#hasClearableDownloads_accessor_storage; }
    set hasClearableDownloads(value) { this.#hasClearableDownloads_accessor_storage = value; }
    #spinnerActive_accessor_storage = false;
    get spinnerActive() { return this.#spinnerActive_accessor_storage; }
    set spinnerActive(value) { this.#spinnerActive_accessor_storage = value; }
    #items_accessor_storage = [];
    get items() { return this.#items_accessor_storage; }
    set items(value) { this.#items_accessor_storage = value; }
    firstUpdated() {
        this.mojoHandler_ = BrowserProxy.getInstance().handler;
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('hasClearableDownloads')) {
            this.updateClearAll_();
        }
    }
    /** @return Whether removal can be undone. */
    canUndo() {
        return !this.isSearchFocused();
    }
    /** @return Whether "Clear all" should be allowed. */
    canClearAll() {
        return this.getSearchText().length === 0 && this.hasClearableDownloads;
    }
    /** @return The full text being searched. */
    getSearchText() {
        return this.$.toolbar.getSearchField().getValue();
    }
    focusOnSearchInput() {
        this.$.toolbar.getSearchField().showAndFocus();
    }
    isSearchFocused() {
        return this.$.toolbar.getSearchField().isSearchFocused();
    }
    onClearAllClick_(e) {
        assert(this.canClearAll());
        this.mojoHandler_.clearAll();
        const canUndo = this.items.some(data => !data.isDangerous && !data.isInsecure);
        getToastManager().show(loadTimeData.getString('toastClearedAll'), 
        /* hideSlotted= */ !canUndo);
        // Stop propagating a click to the document to remove toast.
        e.stopPropagation();
        e.preventDefault();
    }
    onSearchChanged_(event) {
        const searchService = SearchService.getInstance();
        if (searchService.search(event.detail)) {
            this.spinnerActive = searchService.isSearching();
            this.dispatchEvent(new CustomEvent('spinner-active-changed', {
                detail: { value: this.spinnerActive },
                bubbles: true,
                composed: true,
            }));
        }
        this.updateClearAll_();
    }
    updateClearAll_() {
        this.$.clearAll.hidden = !this.canClearAll();
    }
}
customElements.define(DownloadsToolbarElement.is, DownloadsToolbarElement);

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const WebUiListenerMixinLit = (superClass) => {
    class WebUiListenerMixinLit extends superClass {
        /**
         * Holds WebUI listeners that need to be removed when this element is
         * destroyed.
         */
        webUiListeners_ = [];
        /**
         * Adds a WebUI listener and registers it for automatic removal when
         * this element is detached. Note: Do not use this method if you intend
         * to remove this listener manually (use addWebUiListener directly
         * instead).
         *
         * @param eventName The event to listen to.
         * @param callback The callback run when the event is fired.
         */
        addWebUiListener(eventName, callback) {
            this.webUiListeners_.push(addWebUiListener(eventName, callback));
        }
        disconnectedCallback() {
            super.disconnectedCallback();
            while (this.webUiListeners_.length > 0) {
                removeWebUiListener(this.webUiListeners_.pop());
            }
        }
    }
    return WebUiListenerMixinLit;
};

let instance$4 = null;
function getCss$4() {
    return instance$4 || (instance$4 = [...[], css `:host{align-items:center;border-top:1px solid var(--cr-separator-color);color:var(--cr-secondary-text-color);display:none;font-size:0.8125rem;justify-content:center;padding:0 24px}:host([is-managed_]){display:flex}a[href]{color:var(--cr-link-color)}cr-icon{align-self:flex-start;flex-shrink:0;height:20px;padding-inline-end:var(--managed-footnote-icon-padding,8px);width:20px}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$1() {
    return html `${this.isManaged_ ? html `
  <cr-icon .icon="${this.managedByIcon_}"></cr-icon>
  <div id="content" .innerHTML="${this.getManagementString_()}"></div>
` :
        ''}`;
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview UI element for indicating that this user is managed by
 * their organization. This component uses the |isManaged| boolean in
 * loadTimeData, and the |managedByOrg| i18n string.
 *
 * If |isManaged| is false, this component is hidden. If |isManaged| is true, it
 * becomes visible.
 */
const ManagedFootnoteElementBase = I18nMixinLit(WebUiListenerMixinLit(CrLitElement));
class ManagedFootnoteElement extends ManagedFootnoteElementBase {
    static get is() {
        return 'managed-footnote';
    }
    static get styles() {
        return getCss$4();
    }
    render() {
        return getHtml$1.bind(this)();
    }
    static get properties() {
        return {
            /**
             * Whether the user is managed by their organization through enterprise
             * policies.
             */
            isManaged_: {
                reflect: true,
                type: Boolean,
            },
            // 
            /**
             * The name of the icon to display in the footer.
             * Should only be read if isManaged_ is true.
             */
            managedByIcon_: {
                reflect: true,
                type: String,
            },
        };
    }
    #isManaged__accessor_storage = loadTimeData.getBoolean('isManaged');
    get isManaged_() { return this.#isManaged__accessor_storage; }
    set isManaged_(value) { this.#isManaged__accessor_storage = value; }
    #managedByIcon__accessor_storage = loadTimeData.getString('managedByIcon');
    get managedByIcon_() { return this.#managedByIcon__accessor_storage; }
    set managedByIcon_(value) { this.#managedByIcon__accessor_storage = value; }
    // 
    firstUpdated() {
        this.addWebUiListener('is-managed-changed', (managed) => {
            loadTimeData.overrideValues({ isManaged: managed });
            this.isManaged_ = managed;
        });
    }
    /** @return Message to display to the user. */
    getManagementString_() {
        // 
        return this.i18nAdvanced('browserManagedByOrg');
    }
}
customElements.define(ManagedFootnoteElement.is, ManagedFootnoteElement);
chrome.send('observeManagedUI');

let instance$3 = null;
function getCss$3() {
    return instance$3 || (instance$3 = [...[], css `:host{display:block;position:relative}:host([chunk-size="0"]) #container>::slotted(*){box-sizing:border-box;contain-intrinsic-size:var(--list-item-size,100px) auto;content-visibility:auto;width:100%}:host(:not([chunk-size="0"])) #container>::slotted(.chunk){box-sizing:border-box;contain-intrinsic-size:calc(var(--chunk-size) * var(--list-item-size,100px)) auto;content-visibility:auto;width:100%}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-lazy-list' is a component optimized for showing a list of
 * items that overflows the view and requires scrolling. For performance
 * reasons, the DOM items are incrementally added to the view as the user
 * scrolls through the list. The component expects a `scrollTarget` property
 * to be specified indicating the scrolling container. This container is
 * used for observing scroll events and resizes. The container should have
 * bounded height so that cr-lazy-list can determine how many HTML elements to
 * render initially.
 * If using a container that can shrink arbitrarily small to the height of the
 * contents, a 'minViewportHeight' property should also be provided specifying
 * the minimum viewport height to try to fill with items.
 * Each list item's HTML element is created using the `template` property,
 * which should be set to a function returning a TemplateResult corresponding
 * to a passed in list item and selection index.
 * Set `listItemHost` to the `this` context for any event handlers in this
 * template. If this property is not provided, cr-lazy-list is assumed to be
 * residing in a ShadowRoot, and the shadowRoot's |host| is used.
 * The `items` property specifies an array of list item data.
 * The `itemSize` property should be set to an estimate of the list item size.
 * This is used when setting contain-intrinsic-size styling for list items.
 * To restore focus to a specific item if it is focused when the items
 * array changes, set `restoreFocusItem` to that HTMLElement. If the element
 * is focused when the items array is updated, focus will be restored.
 * To set content-visibility on chunks of elements rather than on individual
 * elements, use the `chunkSize` property and specify the number of elements
 * to group. This is useful when rendering large numbers of short items, as
 * the intersection observers added by content-visibility: auto can slow down
 * the UI for very large numbers of elements.
 */
class CrLazyListElement extends CrLitElement {
    static get is() {
        return 'cr-lazy-list';
    }
    static get styles() {
        return getCss$3();
    }
    render() {
        const host = this.listItemHost === undefined ?
            this.getRootNode().host :
            this.listItemHost;
        // Render items into light DOM using the client provided template
        if (this.chunkSize === 0) {
            render(this.items.slice(0, this.numItemsDisplayed_).map((item, index) => {
                return this.template(item, index);
            }), this, { host });
        }
        else {
            const chunks = Math.ceil(this.numItemsDisplayed_ / this.chunkSize);
            const chunkArray = new Array(chunks).fill(0);
            // Render chunk divs.
            render(chunkArray.map((_item, index) => html `<div id="chunk-${index}" class="chunk">
                                     </div>`), this, { host });
            // Render items into chunk divs.
            for (let chunkIndex = 0; chunkIndex < chunks; chunkIndex++) {
                const start = chunkIndex * this.chunkSize;
                const end = Math.min(this.numItemsDisplayed_, (chunkIndex + 1) * this.chunkSize);
                const chunk = this.querySelector(`#chunk-${chunkIndex}`);
                assert(chunk);
                render(this.items.slice(start, end).map((item, index) => {
                    return this.template(item, start + index);
                }), chunk, { host });
            }
        }
        // Render container + slot into shadow DOM
        return html `<div id="container"><slot id="slot"></slot></div>`;
    }
    static get properties() {
        return {
            chunkSize: {
                type: Number,
                reflect: true,
            },
            items: { type: Array },
            itemSize: { type: Number },
            listItemHost: { type: Object },
            minViewportHeight: { type: Number },
            scrollOffset: { type: Number },
            scrollTarget: { type: Object },
            restoreFocusElement: { type: Object },
            template: { type: Object },
            numItemsDisplayed_: {
                state: true,
                type: Number,
            },
        };
    }
    #items_accessor_storage = [];
    get items() { return this.#items_accessor_storage; }
    set items(value) { this.#items_accessor_storage = value; }
    #itemSize_accessor_storage = undefined;
    get itemSize() { return this.#itemSize_accessor_storage; }
    set itemSize(value) { this.#itemSize_accessor_storage = value; }
    #listItemHost_accessor_storage;
    get listItemHost() { return this.#listItemHost_accessor_storage; }
    set listItemHost(value) { this.#listItemHost_accessor_storage = value; }
    #minViewportHeight_accessor_storage;
    get minViewportHeight() { return this.#minViewportHeight_accessor_storage; }
    set minViewportHeight(value) { this.#minViewportHeight_accessor_storage = value; }
    #scrollOffset_accessor_storage = 0;
    get scrollOffset() { return this.#scrollOffset_accessor_storage; }
    set scrollOffset(value) { this.#scrollOffset_accessor_storage = value; }
    #scrollTarget_accessor_storage = document.documentElement;
    get scrollTarget() { return this.#scrollTarget_accessor_storage; }
    set scrollTarget(value) { this.#scrollTarget_accessor_storage = value; }
    #restoreFocusElement_accessor_storage = null;
    get restoreFocusElement() { return this.#restoreFocusElement_accessor_storage; }
    set restoreFocusElement(value) { this.#restoreFocusElement_accessor_storage = value; }
    #template_accessor_storage = () => html ``;
    get template() { return this.#template_accessor_storage; }
    set template(value) { this.#template_accessor_storage = value; }
    #chunkSize_accessor_storage = 0;
    get chunkSize() { return this.#chunkSize_accessor_storage; }
    set chunkSize(value) { this.#chunkSize_accessor_storage = value; }
    #numItemsDisplayed__accessor_storage = 0;
    get numItemsDisplayed_() { return this.#numItemsDisplayed__accessor_storage; }
    set numItemsDisplayed_(value) { this.#numItemsDisplayed__accessor_storage = value; }
    // Internal state
    lastItemsLength_ = 0;
    lastRenderedHeight_ = 0;
    resizeObserver_ = null;
    scrollListener_ = () => this.onScroll_();
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('items')) {
            this.lastItemsLength_ = this.items.length;
            this.numItemsDisplayed_ = this.items.length === 0 ?
                0 :
                Math.min(this.numItemsDisplayed_, this.items.length);
        }
        else {
            assert(this.items.length === this.lastItemsLength_, 'Items array changed in place; rendered result may be incorrect.');
        }
        if (changedProperties.has('itemSize')) {
            this.style.setProperty('--list-item-size', `${this.itemSize}px`);
        }
        if (changedProperties.has('chunkSize')) {
            this.style.setProperty('--chunk-size', `${this.chunkSize}`);
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        let itemsChanged = false;
        if (changedProperties.has('items') ||
            changedProperties.has('minViewportHeight') ||
            changedProperties.has('scrollOffset')) {
            const previous = changedProperties.get('items');
            if (previous !== undefined || this.items.length !== 0) {
                this.onItemsChanged_();
                itemsChanged = true;
            }
        }
        if (changedProperties.has('scrollTarget')) {
            this.addRemoveScrollTargetListeners_(changedProperties.get('scrollTarget') || null);
            // Only re-render if there are items to display and we are not already
            // re-rendering for the items.
            if (this.scrollTarget && this.items.length > 0 && !itemsChanged) {
                this.fillCurrentViewport();
            }
        }
    }
    // Public API
    // Forces the list to fill the current viewport. Called when the viewport
    // size or position changes.
    fillCurrentViewport() {
        if (this.items.length === 0) {
            return Promise.resolve();
        }
        // Update the height if the previous height calculation was done when this
        // element was not visible or if new DOM items were added.
        return this.update_(this.$.container.style.height === '0px');
    }
    // Forces the list to render |numItems| items. If |numItems| are already
    // rendered, this is a no-op.
    async ensureItemRendered(index) {
        if (index < this.numItemsDisplayed_) {
            return this.domItems()[index];
        }
        assert(index < this.items.length);
        await this.updateNumItemsDisplayed_(index + 1);
        return this.domItems()[index];
    }
    // Private methods
    addRemoveScrollTargetListeners_(oldTarget) {
        if (oldTarget) {
            const target = oldTarget === document.documentElement ? window : oldTarget;
            target.removeEventListener('scroll', this.scrollListener_);
            assert(this.resizeObserver_);
            this.resizeObserver_.disconnect();
        }
        if (this.scrollTarget) {
            const target = this.scrollTarget === document.documentElement ?
                window :
                this.scrollTarget;
            target.addEventListener('scroll', this.scrollListener_);
            this.resizeObserver_ = new ResizeObserver(() => {
                requestAnimationFrame(() => {
                    const newHeight = this.getViewHeight_();
                    if (newHeight > 0 && newHeight !== this.lastRenderedHeight_) {
                        this.fillCurrentViewport();
                    }
                });
            });
            this.resizeObserver_.observe(this.scrollTarget);
        }
    }
    shouldRestoreFocus_() {
        if (!this.restoreFocusElement) {
            return false;
        }
        const active = getDeepActiveElement();
        return this.restoreFocusElement === active ||
            (!!this.restoreFocusElement.shadowRoot &&
                this.restoreFocusElement.shadowRoot.activeElement === active);
    }
    async onItemsChanged_() {
        if (this.items.length > 0) {
            const restoreFocus = this.shouldRestoreFocus_();
            await this.update_(true);
            if (restoreFocus) {
                // Async to allow clients to update in response to viewport-filled.
                setTimeout(() => {
                    // The element may have been removed from the DOM by the client.
                    if (!this.restoreFocusElement) {
                        return;
                    }
                    this.restoreFocusElement.focus();
                    this.fire('focus-restored-for-test');
                }, 0);
            }
        }
        else {
            // Update the container height to 0 since there are no items.
            this.$.container.style.height = '0px';
            this.fire('items-rendered');
            this.fire('viewport-filled');
        }
    }
    getScrollTop_() {
        return this.scrollTarget === document.documentElement ?
            window.pageYOffset :
            this.scrollTarget.scrollTop;
    }
    getViewHeight_() {
        const offsetHeight = this.scrollTarget === document.documentElement ?
            window.innerHeight :
            this.scrollTarget.offsetHeight;
        return this.getScrollTop_() - this.scrollOffset +
            Math.max(this.minViewportHeight || 0, offsetHeight);
    }
    async update_(forceUpdateHeight) {
        if (!this.scrollTarget) {
            return;
        }
        const height = this.getViewHeight_();
        if (height <= 0) {
            return;
        }
        const added = await this.fillViewHeight_(height);
        this.fire('items-rendered');
        if (added || forceUpdateHeight) {
            await this.updateHeight_();
            this.fire('viewport-filled');
        }
    }
    /**
     * @return Whether DOM items were created or not.
     */
    async fillViewHeight_(height) {
        this.fire('fill-height-start');
        this.lastRenderedHeight_ = height;
        // Ensure we have added enough DOM items so that we are able to estimate
        // item average height.
        assert(this.items.length);
        const initialDomItemCount = this.domItems().length;
        if (initialDomItemCount === 0) {
            await this.updateNumItemsDisplayed_(1);
        }
        const itemHeight = this.domItemAverageHeight_();
        // If this happens, the math below will be incorrect and we will render
        // all items. So return early, and correct |lastRenderedHeight_|.
        if (itemHeight === 0) {
            this.lastRenderedHeight_ = 0;
            return false;
        }
        const desiredDomItemCount = Math.min(Math.ceil(height / itemHeight), this.items.length);
        if (desiredDomItemCount > this.numItemsDisplayed_) {
            await this.updateNumItemsDisplayed_(desiredDomItemCount);
        }
        const added = initialDomItemCount !== desiredDomItemCount;
        if (added) {
            this.fire('fill-height-end');
        }
        return added;
    }
    async updateNumItemsDisplayed_(itemsToDisplay) {
        this.numItemsDisplayed_ = itemsToDisplay;
        if (this.numItemsDisplayed_ > 200 && this.chunkSize < 2) {
            console.warn(`cr-lazy-list: ${this.numItemsDisplayed_} list items rendered. ` +
                'If this is expected, consider chunking mode (chunkSize > 1) ' +
                'to improve scrolling performance.');
        }
        await this.updateComplete;
    }
    /**
     * @return The currently rendered list items, particularly useful for clients
     *     using chunking mode.
     */
    domItems() {
        return this.chunkSize === 0 ?
            this.$.slot.assignedElements() :
            Array.from(this.querySelectorAll('.chunk > *'));
    }
    /**
     * @return The average DOM item height.
     */
    domItemAverageHeight_() {
        // This logic should only be invoked if the list is non-empty and at least
        // one DOM item has been rendered so that an item average height can be
        // estimated. This is ensured by the callers.
        assert(this.items.length > 0);
        const domItems = this.domItems();
        assert(domItems.length > 0);
        const firstDomItem = domItems.at(0);
        const lastDomItem = domItems.at(-1);
        const lastDomItemHeight = lastDomItem.offsetHeight;
        if (firstDomItem === lastDomItem && lastDomItemHeight === 0) {
            // If there is only 1 item and it has a height of 0, return early. This
            // likely means the UI is still hidden or there is no content.
            return 0;
        }
        else if (this.itemSize) {
            // Once items are actually visible and have a height > 0, assume that it
            // is an accurate representation of the average item size.
            return this.itemSize;
        }
        let totalHeight = lastDomItem.offsetTop + lastDomItemHeight;
        if (this.chunkSize > 0) {
            // Add the parent's offsetTop. The offsetParent will be the chunk div.
            // Subtract the offsetTop of the first chunk div to avoid counting any
            // padding.
            totalHeight += lastDomItem.offsetParent.offsetTop -
                firstDomItem.offsetParent.offsetTop;
        }
        else {
            // Subtract the offsetTop of the first item to avoid counting any padding.
            totalHeight -= firstDomItem.offsetTop;
        }
        return totalHeight / domItems.length;
    }
    /**
     * Sets the height of the component based on an estimated average DOM item
     * height and the total number of items.
     */
    async updateHeight_() {
        // Await 1 cycle to ensure any child Lit elements have time to finish
        // rendering, or the height estimated below will be incorrect.
        await new Promise(resolve => setTimeout(resolve, 0));
        const estScrollHeight = this.items.length > 0 ?
            this.items.length * this.domItemAverageHeight_() :
            0;
        this.$.container.style.height = estScrollHeight + 'px';
    }
    /**
     * Adds additional DOM items as needed to fill the view based on user scroll
     * interactions.
     */
    async onScroll_() {
        const scrollTop = this.getScrollTop_();
        if (scrollTop <= 0 || this.numItemsDisplayed_ === this.items.length) {
            return;
        }
        await this.fillCurrentViewport();
    }
}
customElements.define(CrLazyListElement.is, CrLazyListElement);

let instance$2 = null;
function getCss$2() {
    return instance$2 || (instance$2 = [...[], css `:host{display:block;position:relative}:host([using-default-scroll-target]){overflow-y:auto}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-infinite-list' is a thin wrapper around 'cr-lazy-list' that
 * emulates some of the behavior of 'iron-list'.
 */
class CrInfiniteListElement extends CrLitElement {
    static get is() {
        return 'cr-infinite-list';
    }
    static get styles() {
        return getCss$2();
    }
    render() {
        // Render items into light DOM using the client provided template
        render(html `<cr-lazy-list id="list" .scrollTarget="${this.scrollTarget}"
          .chunkSize="${this.chunkSize}"
          .scrollOffset="${this.scrollOffset}"
          .listItemHost="${this.getRootNode().host}"
          .items="${this.items}" .itemSize="${this.itemSize}"
          .template="${(item, index) => this.template(item, index, index === this.focusedIndex ? 0 : -1)}"
          .restoreFocusElement="${this.focusedItem_}"
          @keydown="${this.onKeyDown_}"
          @focusin="${this.onItemFocus_}"
          @viewport-filled="${this.updateFocusedItem_}">
        </cr-lazy-list>`, this, {
            host: this,
        });
        return html `<slot></slot>`;
    }
    static get properties() {
        return {
            chunkSize: { type: Number },
            scrollOffset: { type: Number },
            scrollTarget: { type: Object },
            usingDefaultScrollTarget: {
                type: Boolean,
                reflect: true,
            },
            items: { type: Array },
            focusedIndex: { type: Number },
            itemSize: { type: Number },
            template: { type: Object },
            focusedItem_: { type: Object },
        };
    }
    #chunkSize_accessor_storage = 0;
    get chunkSize() { return this.#chunkSize_accessor_storage; }
    set chunkSize(value) { this.#chunkSize_accessor_storage = value; }
    #scrollOffset_accessor_storage = 0;
    get scrollOffset() { return this.#scrollOffset_accessor_storage; }
    set scrollOffset(value) { this.#scrollOffset_accessor_storage = value; }
    #scrollTarget_accessor_storage = this;
    get scrollTarget() { return this.#scrollTarget_accessor_storage; }
    set scrollTarget(value) { this.#scrollTarget_accessor_storage = value; }
    #usingDefaultScrollTarget_accessor_storage = true;
    get usingDefaultScrollTarget() { return this.#usingDefaultScrollTarget_accessor_storage; }
    set usingDefaultScrollTarget(value) { this.#usingDefaultScrollTarget_accessor_storage = value; }
    #items_accessor_storage = [];
    get items() { return this.#items_accessor_storage; }
    set items(value) { this.#items_accessor_storage = value; }
    #itemSize_accessor_storage = undefined;
    get itemSize() { return this.#itemSize_accessor_storage; }
    set itemSize(value) { this.#itemSize_accessor_storage = value; }
    #template_accessor_storage = () => html ``;
    // Unlike cr-lazy-list, cr-infinite-list provides a tabindex parameter for
    // clients as is provided by iron-list. Like iron-list, cr-infinite-list will
    // pass 0 for this parameter if the list item should be keyboard focusable,
    // and -1 otherwise.
    get template() { return this.#template_accessor_storage; }
    set template(value) { this.#template_accessor_storage = value; }
    #focusedIndex_accessor_storage = -1;
    get focusedIndex() { return this.#focusedIndex_accessor_storage; }
    set focusedIndex(value) { this.#focusedIndex_accessor_storage = value; }
    #focusedItem__accessor_storage = null;
    get focusedItem_() { return this.#focusedItem__accessor_storage; }
    set focusedItem_(value) { this.#focusedItem__accessor_storage = value; }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('scrollTarget')) {
            this.usingDefaultScrollTarget = this.scrollTarget === this;
        }
        if (changedProperties.has('items')) {
            if (this.focusedIndex >= this.items.length) {
                this.focusedIndex = this.items.length - 1;
            }
            else if (this.focusedIndex === -1 && this.items.length > 0) {
                this.focusedIndex = 0;
            }
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('focusedIndex')) {
            this.updateFocusedItem_();
        }
    }
    fillCurrentViewport() {
        const list = this.querySelector('cr-lazy-list');
        assert(list);
        return list.fillCurrentViewport();
    }
    ensureItemRendered(index) {
        const list = this.querySelector('cr-lazy-list');
        assert(list);
        return list.ensureItemRendered(index);
    }
    updateFocusedItem_() {
        if (this.focusedIndex === -1) {
            this.focusedItem_ = null;
            return;
        }
        const list = this.querySelector('cr-lazy-list');
        assert(list);
        this.focusedItem_ =
            list.domItems()[this.focusedIndex + 1] ||
                null;
    }
    onItemFocus_(e) {
        const list = this.querySelector('cr-lazy-list');
        assert(list);
        const renderedItems = list.domItems();
        const focusedIdx = Array.from(renderedItems).findIndex(item => {
            return item === e.target || item.shadowRoot?.activeElement === e.target;
        });
        if (focusedIdx !== -1) {
            this.focusedIndex = focusedIdx;
        }
    }
    /**
     * Handles key events when list item elements have focus.
     */
    async onKeyDown_(e) {
        // Do not interfere with any parent component that manages 'shift' related
        // key events.
        if (e.shiftKey || (e.key !== 'ArrowUp' && e.key !== 'ArrowDown')) {
            return;
        }
        e.stopPropagation();
        e.preventDefault();
        // Identify the new focused index.
        this.focusedIndex = e.key === 'ArrowUp' ?
            Math.max(0, this.focusedIndex - 1) :
            Math.min(this.items.length - 1, this.focusedIndex + 1);
        const list = this.querySelector('cr-lazy-list');
        assert(list);
        const element = await list.ensureItemRendered(this.focusedIndex);
        element.focus();
        element.scrollIntoViewIfNeeded();
    }
}
customElements.define(CrInfiniteListElement.is, CrInfiniteListElement);

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/** This is used to identify keyboard shortcuts. */
class KeyboardShortcut {
    useKeyCode_ = false;
    mods_ = {};
    key_ = null;
    keyCode_ = null;
    /**
     * @param shortcut The text used to describe the keys for this
     *     keyboard shortcut.
     */
    constructor(shortcut) {
        shortcut.split('|').forEach((part) => {
            const partLc = part.toLowerCase();
            switch (partLc) {
                case 'alt':
                case 'ctrl':
                case 'meta':
                case 'shift':
                    this.mods_[partLc + 'Key'] = true;
                    break;
                default:
                    if (this.key_) {
                        throw Error('Invalid shortcut');
                    }
                    this.key_ = part;
                    // For single key alpha shortcuts use event.keyCode rather than
                    // event.key to match how chrome handles shortcuts and allow
                    // non-english language input to work.
                    if (part.match(/^[a-z]$/)) {
                        this.useKeyCode_ = true;
                        this.keyCode_ = part.toUpperCase().charCodeAt(0);
                    }
            }
        });
    }
    /**
     * Whether the keyboard shortcut object matches a keyboard event.
     * @param e The keyboard event object.
     * @return Whether we found a match or not.
     */
    matchesEvent(e) {
        if ((this.useKeyCode_ && e.keyCode === this.keyCode_) ||
            e.key === this.key_) {
            // All keyboard modifiers need to match.
            const mods = this.mods_;
            return ['altKey', 'ctrlKey', 'metaKey', 'shiftKey'].every(function (k) {
                return e[k] === !!mods[k];
            });
        }
        return false;
    }
}
/** A list of keyboard shortcuts which all perform one command. */
class KeyboardShortcutList {
    shortcuts_;
    /**
     * @param shortcuts Text-based representation of one or more
     *     keyboard shortcuts, separated by spaces.
     */
    constructor(shortcuts) {
        this.shortcuts_ = shortcuts.split(/\s+/).map(function (shortcut) {
            return new KeyboardShortcut(shortcut);
        });
    }
    /**
     * Returns true if any of the keyboard shortcuts in the list matches a
     * keyboard event.
     */
    matchesEvent(e) {
        return this.shortcuts_.some(function (keyboardShortcut) {
            return keyboardShortcut.matchesEvent(e);
        });
    }
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Listens for a find keyboard shortcut (i.e. Ctrl/Cmd+f or /)
 * and keeps track of an stack of potential listeners. Only the listener at the
 * top of the stack will be notified that a find shortcut has been invoked.
 */
const FindShortcutManager = (() => {
    /**
     * Stack of listeners. Only the top listener will handle the shortcut.
     */
    const listeners = [];
    /**
     * Tracks if any modal context is open in settings. This assumes only one
     * modal can be open at a time. The modals that are being tracked include
     * cr-dialog and cr-drawer.
     * @type {boolean}
     */
    let modalContextOpen = false;
    const shortcutCtrlF = new KeyboardShortcutList(isMac ? 'meta|f' : 'ctrl|f');
    const shortcutSlash = new KeyboardShortcutList('/');
    window.addEventListener('keydown', e => {
        if (e.defaultPrevented || listeners.length === 0) {
            return;
        }
        const element = e.composedPath()[0];
        if (!shortcutCtrlF.matchesEvent(e) &&
            (element.tagName === 'INPUT' || element.tagName === 'TEXTAREA' ||
                !shortcutSlash.matchesEvent(e))) {
            return;
        }
        const focusIndex = listeners.findIndex(listener => listener.searchInputHasFocus());
        // If no listener has focus or the first (outer-most) listener has focus,
        // try the last (inner-most) listener.
        // If a listener has a search input with focus, the next listener that
        // should be called is the right before it in |listeners| such that the
        // goes from inner-most to outer-most.
        const index = focusIndex <= 0 ? listeners.length - 1 : focusIndex - 1;
        if (listeners[index].handleFindShortcut(modalContextOpen)) {
            e.preventDefault();
        }
    });
    window.addEventListener('cr-dialog-open', () => {
        modalContextOpen = true;
    });
    window.addEventListener('cr-drawer-opened', () => {
        modalContextOpen = true;
    });
    window.addEventListener('close', e => {
        if (['CR-DIALOG', 'CR-DRAWER'].includes(e.composedPath()[0].nodeName)) {
            modalContextOpen = false;
        }
    });
    return Object.freeze({ listeners: listeners });
})();

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Used to determine how to handle find shortcut invocations.
 */
const FindShortcutMixinLit = (superClass) => {
    class FindShortcutMixinLit extends superClass {
        findShortcutListenOnAttach = true;
        connectedCallback() {
            super.connectedCallback();
            if (this.findShortcutListenOnAttach) {
                this.becomeActiveFindShortcutListener();
            }
        }
        disconnectedCallback() {
            super.disconnectedCallback();
            if (this.findShortcutListenOnAttach) {
                this.removeSelfAsFindShortcutListener();
            }
        }
        becomeActiveFindShortcutListener() {
            const listeners = FindShortcutManager.listeners;
            assert(!listeners.includes(this), 'Already listening for find shortcuts.');
            listeners.push(this);
        }
        handleFindShortcutInternal_() {
            assertNotReached('Must override handleFindShortcut()');
        }
        handleFindShortcut(_modalContextOpen) {
            this.handleFindShortcutInternal_();
            return false;
        }
        removeSelfAsFindShortcutListener() {
            const listeners = FindShortcutManager.listeners;
            const index = listeners.indexOf(this);
            assert(listeners.includes(this), 'Find shortcut listener not found.');
            listeners.splice(index, 1);
        }
        searchInputHasFocusInternal_() {
            assertNotReached('Must override searchInputHasFocus()');
        }
        searchInputHasFocus() {
            this.searchInputHasFocusInternal_();
            return false;
        }
    }
    return FindShortcutMixinLit;
};

let instance$1 = null;
function getCss$1() {
    return instance$1 || (instance$1 = [...[], css `:host{color:var(--cr-primary-text-color);line-height:154%;overflow:hidden;user-select:text}`]);
}

let instance = null;
function getCss() {
    return instance || (instance = [...[getCss$o(), getCss$1(), getCss$9()], css `:host{display:flex;flex:1 0;flex-direction:column;height:100%;overflow:hidden;z-index:0}@media (prefers-color-scheme:dark){:host{color:var(--cr-secondary-text-color)}}#toolbar{z-index:1;--cr-toolbar-center-basis:680px;--cr-toolbar-field-max-width:var(--cr-toolbar-center-basis);--cr-toolbar-field-width:100%;--cr-toolbar-header-white-space:nowrap}:host([has-shadow_]) #drop-shadow{opacity:var(--cr-container-shadow-max-opacity)}downloads-item,#downloadsList,cr-toast-manager{--downloads-card-margin:24px;--downloads-card-width:clamp(550px,80%,680px);--cr-toast-max-width:var(--downloads-card-width)}#downloadsList{min-width:calc(var(--downloads-card-width) + 2 * var(--downloads-card-margin))}#no-downloads,#downloadsList{flex:1}:host([loading]) #no-downloads,:host([loading]) #downloadsList{display:none}#no-downloads{align-items:center;color:#6e6e6e;display:flex;font-size:123.1%;font-weight:500;justify-content:center;min-height:min-content}@media (prefers-color-scheme:dark){#no-downloads{color:var(--cr-secondary-text-color)}}#no-downloads .illustration{background:url(images/no_downloads.svg) no-repeat center center;background-size:contain;height:144px;margin-bottom:32px}#mainContainer{display:flex;flex:1;flex-direction:column;height:100%;overflow-y:overlay}managed-footnote{border-top:none;margin-bottom:calc(-21px - 8px);min-width:calc(var(--downloads-card-width) + 2 * var(--downloads-card-margin));padding-bottom:12px;padding-top:12px;z-index:1}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml() {
    // clang-format off
    return html `<!--_html_template_start_-->
<downloads-toolbar id="toolbar" role="none" .items="${this.items_}"
    .spinnerActive="${this.spinnerActive_}"
    .hasClearableDownloads="${this.hasClearableDownloads_()}"
    @spinner-active-changed="${this.onSpinnerActiveChanged_}"
    @search-changed="${this.onSearchChanged_}">
</downloads-toolbar>
<div id="drop-shadow" class="cr-container-shadow"></div>
<div id="mainContainer" @scroll="${this.onScroll_}"
    @save-dangerous-click="${this.onSaveDangerousClick_}">
  <managed-footnote ?hidden="${this.inSearchMode_}"></managed-footnote>
  <cr-infinite-list id="downloadsList" .items="${this.items_}"
      role="grid" aria-rowcount="${this.items_.length}"
      ?hidden="${!this.hasDownloads_}" .scrollTarget="${this.listScrollTarget_}"
      .template=${(item, index, tabindex) => html `
  
  
        <downloads-item .data="${item}" tabindex="${tabindex}"
            .listTabIndex="${tabindex}" .lastFocused="${this.lastFocused_}"
            @last-focused-changed="${this.onLastFocusedChanged_}"
            .listBlurred="${this.listBlurred_}"
            @list-blurred-changed="${this.onListBlurredChanged_}"
            .focusRowIndex="${index}"
        >
  
        </downloads-item>`}>
  </cr-infinite-list>
  <div id="no-downloads" ?hidden="${this.hasDownloads_}">
    <div>
      <div class="illustration"></div>
      <span>${this.noDownloadsText_()}</span>
    </div>
  </div>
</div>
<cr-toast-manager duration="0">
  <cr-button aria-label="$i18n{undoDescription}" @click="${this.onUndoClick_}">
    $i18n{undo}
  </cr-button>
</cr-toast-manager>
${this.shouldShowBypassWarningPrompt_() ?
        html `<downloads-bypass-warning-confirmation-dialog
        .fileName="${this.computeBypassWarningDialogFileName_()}"
        @close="${this.onBypassWarningConfirmationDialogClose_}">
    </downloads-bypass-warning-confirmation-dialog>`
        : ''}
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const DownloadsManagerElementBase = FindShortcutMixinLit(CrLitElement);
class DownloadsManagerElement extends DownloadsManagerElementBase {
    static get is() {
        return 'downloads-manager';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            hasDownloads_: { type: Boolean },
            hasShadow_: {
                type: Boolean,
                reflect: true,
            },
            inSearchMode_: { type: Boolean },
            items_: { type: Array },
            spinnerActive_: { type: Boolean },
            bypassPromptItemId_: { type: String },
            // 
            lastFocused_: { type: Object },
            listBlurred_: { type: Boolean },
            listScrollTarget_: { type: Object },
        };
    }
    static get observers() {
        return ['itemsChanged_(items_.*)'];
    }
    #items__accessor_storage = [];
    get items_() { return this.#items__accessor_storage; }
    set items_(value) { this.#items__accessor_storage = value; }
    #hasDownloads__accessor_storage = false;
    get hasDownloads_() { return this.#hasDownloads__accessor_storage; }
    set hasDownloads_(value) { this.#hasDownloads__accessor_storage = value; }
    #hasShadow__accessor_storage = false;
    // Used for CSS styling.
    get hasShadow_() { return this.#hasShadow__accessor_storage; }
    set hasShadow_(value) { this.#hasShadow__accessor_storage = value; }
    #inSearchMode__accessor_storage = false;
    get inSearchMode_() { return this.#inSearchMode__accessor_storage; }
    set inSearchMode_(value) { this.#inSearchMode__accessor_storage = value; }
    #spinnerActive__accessor_storage = false;
    get spinnerActive_() { return this.#spinnerActive__accessor_storage; }
    set spinnerActive_(value) { this.#spinnerActive__accessor_storage = value; }
    #bypassPromptItemId__accessor_storage = '';
    get bypassPromptItemId_() { return this.#bypassPromptItemId__accessor_storage; }
    set bypassPromptItemId_(value) { this.#bypassPromptItemId__accessor_storage = value; }
    #lastFocused__accessor_storage = null;
    // 
    get lastFocused_() { return this.#lastFocused__accessor_storage; }
    set lastFocused_(value) { this.#lastFocused__accessor_storage = value; }
    #listBlurred__accessor_storage = false;
    get listBlurred_() { return this.#listBlurred__accessor_storage; }
    set listBlurred_(value) { this.#listBlurred__accessor_storage = value; }
    #listScrollTarget__accessor_storage = null;
    get listScrollTarget_() { return this.#listScrollTarget__accessor_storage; }
    set listScrollTarget_(value) { this.#listScrollTarget__accessor_storage = value; }
    announcerTimeout_ = null;
    mojoHandler_;
    mojoEventTarget_;
    searchService_ = SearchService.getInstance();
    loaded_ = new PromiseResolver();
    listenerIds_ = [];
    eventTracker_ = new EventTracker();
    constructor() {
        super();
        const browserProxy = BrowserProxy.getInstance();
        this.mojoEventTarget_ = browserProxy.callbackRouter;
        this.mojoHandler_ = browserProxy.handler;
        // Regular expression that captures the leading slash, the content and the
        // trailing slash in three different groups.
        const CANONICAL_PATH_REGEX = /(^\/)([\/-\w]+)(\/$)/;
        const path = location.pathname.replace(CANONICAL_PATH_REGEX, '$1$2');
        if (path !== '/') { // There are no subpages in chrome://downloads.
            window.history.replaceState(undefined /* stateObject */, '', '/');
        }
    }
    connectedCallback() {
        super.connectedCallback();
        // TODO(dbeam): this should use a class instead.
        this.toggleAttribute('loading', true);
        document.documentElement.classList.remove('loading');
        this.listenerIds_ = [
            this.mojoEventTarget_.clearAll.addListener(this.clearAll_.bind(this)),
            this.mojoEventTarget_.insertItems.addListener(this.insertItems_.bind(this)),
            this.mojoEventTarget_.removeItem.addListener(this.removeItem_.bind(this)),
            this.mojoEventTarget_.updateItem.addListener(this.updateItem_.bind(this)),
        ];
        this.eventTracker_.add(document, 'keydown', (e) => this.onKeyDown_(e));
        this.eventTracker_.add(document, 'click', () => this.onClick_());
        this.loaded_.promise.then(() => {
            requestIdleCallback(function () {
                // https://github.com/microsoft/TypeScript/issues/13569
                document.fonts.load('bold 12px Roboto');
            });
        });
        this.searchService_.loadMore();
        // Intercepts clicks on toast.
        const toastManager = getToastManager();
        toastManager.shadowRoot.querySelector('#toast').onclick = e => this.onToastClicked_(e);
        // 
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.listenerIds_.forEach(id => assert(this.mojoEventTarget_.removeListener(id)));
        this.eventTracker_.removeAll();
    }
    firstUpdated(changedProperties) {
        super.firstUpdated(changedProperties);
        this.listScrollTarget_ = this.$.mainContainer;
    }
    onSaveDangerousClick_(e) {
        const bypassItem = this.items_.find(item => item.id === e.detail.id);
        if (bypassItem) {
            this.bypassPromptItemId_ = bypassItem.id;
            assert(!!this.mojoHandler_);
            this.mojoHandler_.recordOpenBypassWarningDialog(this.bypassPromptItemId_);
        }
    }
    // 
    shouldShowBypassWarningPrompt_() {
        return this.bypassPromptItemId_ !== '';
    }
    computeBypassWarningDialogFileName_() {
        const bypassItem = this.items_.find(item => item.id === this.bypassPromptItemId_);
        return bypassItem?.fileName || '';
    }
    hideBypassWarningPrompt_() {
        this.bypassPromptItemId_ = '';
    }
    onBypassWarningConfirmationDialogClose_() {
        const dialog = this.shadowRoot.querySelector('downloads-bypass-warning-confirmation-dialog');
        assert(dialog);
        assert(this.bypassPromptItemId_ !== '');
        assert(!!this.mojoHandler_);
        if (dialog.wasConfirmed()) {
            this.mojoHandler_.saveDangerousFromDialogRequiringGesture(this.bypassPromptItemId_);
        }
        else {
            // Closing the dialog by clicking cancel is treated the same as closing
            // the dialog by pressing Esc. Both are treated as CANCEL, not CLOSE.
            this.mojoHandler_.recordCancelBypassWarningDialog(this.bypassPromptItemId_);
        }
        this.hideBypassWarningPrompt_();
    }
    clearAll_() {
        this.items_ = [];
        this.itemsChanged_();
    }
    insertItems_(index, items) {
        // Insert |items| at the given |index|.
        if (items.length > 0) {
            this.updateItems_(index, 0, items);
        }
        if (this.hasAttribute('loading')) {
            this.removeAttribute('loading');
            this.loaded_.resolve();
        }
        this.spinnerActive_ = false;
    }
    hasClearableDownloads_() {
        return loadTimeData.getBoolean('allowDeletingHistory') &&
            this.hasDownloads_;
    }
    itemsChanged_() {
        this.hasDownloads_ = this.items_.length > 0;
        if (!this.inSearchMode_) {
            return;
        }
        if (this.announcerTimeout_) {
            clearTimeout(this.announcerTimeout_);
        }
        this.announcerTimeout_ = setTimeout(() => {
            const searchText = this.$.toolbar.getSearchText();
            const announcement = this.items_.length === 0 ?
                this.noDownloadsText_() :
                (this.items_.length === 1 ?
                    loadTimeData.getStringF('searchResultsSingular', searchText) :
                    loadTimeData.getStringF('searchResultsPlural', this.items_.length, searchText));
            getInstance().announce(announcement);
            this.announcerTimeout_ = null;
        }, 500);
    }
    /**
     * @return The text to show when no download items are showing.
     */
    noDownloadsText_() {
        return loadTimeData.getString(this.inSearchMode_ ? 'noSearchResults' : 'noDownloads');
    }
    onKeyDown_(e) {
        let clearAllKey = 'c';
        // 
        // On Mac, pressing alt+c produces 'ç' as |event.key|.
        clearAllKey = 'ç';
        // 
        if (e.key === clearAllKey && e.altKey && !e.ctrlKey && !e.shiftKey &&
            !e.metaKey) {
            this.onClearAllCommand_();
            e.preventDefault();
            return;
        }
        if (e.key === 'z' && !e.altKey && !e.shiftKey) {
            let hasTriggerModifier = e.ctrlKey && !e.metaKey;
            // 
            hasTriggerModifier = !e.ctrlKey && e.metaKey;
            // 
            if (hasTriggerModifier) {
                this.onUndoCommand_();
                e.preventDefault();
            }
        }
    }
    onClick_() {
        const toastManager = getToastManager();
        if (toastManager.isToastOpen) {
            toastManager.hide();
        }
    }
    onClearAllCommand_() {
        if (!this.$.toolbar.canClearAll()) {
            return;
        }
        this.mojoHandler_.clearAll();
        const canUndo = this.items_.some(data => !data.isDangerous && !data.isInsecure);
        getToastManager().show(loadTimeData.getString('toastClearedAll'), 
        /* hideSlotted= */ !canUndo);
    }
    onUndoCommand_() {
        if (!this.$.toolbar.canUndo()) {
            return;
        }
        getToastManager().hide();
        this.mojoHandler_.undo();
    }
    onToastClicked_(e) {
        e.stopPropagation();
        e.preventDefault();
    }
    onScroll_() {
        const container = this.listScrollTarget_;
        assert(!!container);
        const distanceToBottom = container.scrollHeight - container.scrollTop - container.offsetHeight;
        if (distanceToBottom <= 100) {
            // Approaching the end of the scrollback. Attempt to load more items.
            this.searchService_.loadMore();
        }
        this.hasShadow_ = container.scrollTop > 0;
    }
    onSearchChanged_() {
        this.inSearchMode_ = this.searchService_.isSearching();
    }
    onSpinnerActiveChanged_(event) {
        this.spinnerActive_ = event.detail.value;
    }
    removeItem_(index) {
        const removed = this.items_[index];
        if (removed.id === this.bypassPromptItemId_) {
            this.hideBypassWarningPrompt_();
        }
        this.updateItems_(index, 1, []);
        this.updateComplete.then(() => this.onScroll_());
    }
    updateItems_(index, toRemove, newItems = []) {
        const items = [
            ...this.items_.slice(0, index),
            ...newItems,
            ...this.items_.slice(index + toRemove),
        ];
        // Update whether dates should show.
        for (let i = index; i <= index + newItems.length; ++i) {
            const current = items[i];
            if (!current) {
                continue;
            }
            const prev = items[i - 1];
            current.hideDate = !!prev && prev.dateString === current.dateString;
        }
        const lengthChanged = this.items_.length !== items.length;
        this.items_ = items;
        if (lengthChanged) {
            this.itemsChanged_();
        }
    }
    onUndoClick_() {
        getToastManager().hide();
        this.mojoHandler_.undo();
    }
    updateItem_(index, data) {
        this.updateItems_(index, 1, [data]);
    }
    onLastFocusedChanged_(e) {
        this.lastFocused_ = e.detail.value;
    }
    onListBlurredChanged_(e) {
        this.listBlurred_ = e.detail.value;
    }
    // Override FindShortcutMixin methods.
    handleFindShortcut(modalContextOpen) {
        if (modalContextOpen) {
            return false;
        }
        this.$.toolbar.focusOnSearchInput();
        return true;
    }
    // Override FindShortcutMixin methods.
    searchInputHasFocus() {
        return this.$.toolbar.isSearchFocused();
    }
}
customElements.define(DownloadsManagerElement.is, DownloadsManagerElement);

export { BrowserProxy, CrIconElement, CrToastManagerElement, DangerType, DownloadsItemElement, DownloadsManagerElement, DownloadsToolbarElement, IconLoaderImpl, PageCallbackRouter, PageRemote, SafeBrowsingState, SearchService, State, TailoredWarningType };
//# sourceMappingURL=downloads.rollup.js.map
