// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '//resources/cr_elements/cr_action_menu/cr_action_menu.js';
import '//resources/cr_elements/cr_lazy_render/cr_lazy_render_lit.js';
import { WebUiListenerMixinLit } from '//resources/cr_elements/web_ui_listener_mixin_lit.js';
import { assert } from '//resources/js/assert.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { openMenu, ToolbarEvent } from '../common.js';
import { getCss } from './simple_action_menu.css.js';
import { getHtml } from './simple_action_menu.html.js';
const SimpleActionMenuElementBase = WebUiListenerMixinLit(CrLitElement);
// Represents a simple dropdown menu that contains a flat list of items with
// text and an optional icon. Selecting an item in this menu propagates that
// event, sets that item as selected with a visual checkmark, and then closes
// the menu.
export class SimpleActionMenuElement extends SimpleActionMenuElementBase {
    static get is() {
        return 'simple-action-menu';
    }
    render() {
        return getHtml.bind(this)();
    }
    static get styles() {
        return getCss();
    }
    static get properties() {
        return {
            currentSelectedIndex: { type: Number },
            menuItems: { type: Array },
            eventName: { type: String },
            label: { type: String },
        };
    }
    #currentSelectedIndex_accessor_storage = 0;
    get currentSelectedIndex() { return this.#currentSelectedIndex_accessor_storage; }
    set currentSelectedIndex(value) { this.#currentSelectedIndex_accessor_storage = value; }
    #menuItems_accessor_storage = [];
    get menuItems() { return this.#menuItems_accessor_storage; }
    set menuItems(value) { this.#menuItems_accessor_storage = value; }
    #eventName_accessor_storage = ToolbarEvent.THEME;
    // Initializing to random value, but this is set by the parent.
    get eventName() { return this.#eventName_accessor_storage; }
    set eventName(value) { this.#eventName_accessor_storage = value; }
    #label_accessor_storage = '';
    get label() { return this.#label_accessor_storage; }
    set label(value) { this.#label_accessor_storage = value; }
    open(anchor) {
        openMenu(this.$.lazyMenu.get(), anchor);
    }
    onClick_(e) {
        const currentTarget = e.currentTarget;
        this.currentSelectedIndex =
            Number.parseInt(currentTarget.dataset['index']);
        const menuItem = this.menuItems[this.currentSelectedIndex];
        assert(menuItem);
        this.fire(this.eventName, { data: menuItem.data });
        this.$.lazyMenu.get().close();
    }
    isItemSelected_(index) {
        return index === this.currentSelectedIndex;
    }
    doesItemHaveIcon_(item) {
        return item.icon !== undefined;
    }
    itemIcon_(item) {
        return item.icon === undefined ? '' : item.icon;
    }
}
customElements.define(SimpleActionMenuElement.is, SimpleActionMenuElement);
