"use strict";
import * as i18n from "../../../core/i18n/i18n.js";
import * as Helpers from "../helpers/helpers.js";
import { metricSavingsForWastedBytes } from "./Common.js";
import {
  InsightCategory,
  InsightKeys
} from "./types.js";
export const UIStrings = {
  /**
   * @description Title of an insight that recommends ways to reduce the size of images downloaded and used on the page.
   */
  title: "Improve image delivery",
  /**
   * @description Description of an insight that recommends ways to reduce the size of images downloaded and used on the page.
   */
  description: "Reducing the download time of images can improve the perceived load time of the page and LCP. [Learn more about optimizing image size](https://developer.chrome.com/docs/performance/insights/image-delivery)",
  /**
   * @description Message displayed in a chip explaining that an image file size is large for the # of pixels it has and recommends possible adjustments to improve the image size.
   */
  useCompression: "Increasing the image compression factor could improve this image's download size.",
  /**
   * @description Message displayed in a chip explaining that an image file size is large for the # of pixels it has and recommends possible adjustments to improve the image size.
   */
  useModernFormat: "Using a modern image format (WebP, AVIF) or increasing the image compression could improve this image's download size.",
  /**
   * @description Message displayed in a chip advising the user to use video formats instead of GIFs because videos generally have smaller file sizes.
   */
  useVideoFormat: "Using video formats instead of GIFs can improve the download size of animated content.",
  /**
   * @description Message displayed in a chip explaining that an image was displayed on the page with dimensions much smaller than the image file dimensions.
   * @example {1000x500} PH1
   * @example {100x50} PH2
   */
  useResponsiveSize: "This image file is larger than it needs to be ({PH1}) for its displayed dimensions ({PH2}). Use responsive images to reduce the image download size.",
  /**
   * @description Column header for a table column containing network requests for images which can improve their file size (e.g. use a different format, increase compression, etc).
   */
  optimizeFile: "Optimize file size",
  /**
   * @description Table row value representing the remaining items not shown in the table due to size constraints. This row will always represent at least 2 items.
   * @example {5} PH1
   */
  others: "{PH1} others",
  /**
   * @description Text status indicating that no potential optimizations were found for any image file
   */
  noOptimizableImages: "No optimizable images",
  /**
   * @description Text describing the estimated number of bytes that an image file optimization can save. This text is appended to another block of text describing the image optimization in more detail. "Est" means "Estimated".
   * @example {Use the correct image dimensions to reduce the image file size.} PH1
   * @example {50 MB} PH2
   */
  estimatedSavings: "{PH1} (Est {PH2})"
};
const str_ = i18n.i18n.registerUIStrings("models/trace/insights/ImageDelivery.ts", UIStrings);
export const i18nString = i18n.i18n.getLocalizedString.bind(void 0, str_);
const TARGET_BYTES_PER_PIXEL_AVIF = 2 * 1 / 12;
const GIF_SIZE_THRESHOLD = 100 * 1024;
const BYTE_SAVINGS_THRESHOLD = 4096;
const BYTE_SAVINGS_THRESHOLD_RESPONSIVE_BREAKPOINTS = 12288;
export var ImageOptimizationType = /* @__PURE__ */ ((ImageOptimizationType2) => {
  ImageOptimizationType2["ADJUST_COMPRESSION"] = "ADJUST_COMPRESSION";
  ImageOptimizationType2["MODERN_FORMAT_OR_COMPRESSION"] = "MODERN_FORMAT_OR_COMPRESSION";
  ImageOptimizationType2["VIDEO_FORMAT"] = "VIDEO_FORMAT";
  ImageOptimizationType2["RESPONSIVE_SIZE"] = "RESPONSIVE_SIZE";
  return ImageOptimizationType2;
})(ImageOptimizationType || {});
export function isImageDeliveryInsight(model) {
  return model.insightKey === "ImageDelivery";
}
export function getOptimizationMessage(optimization) {
  switch (optimization.type) {
    case "ADJUST_COMPRESSION" /* ADJUST_COMPRESSION */:
      return i18nString(UIStrings.useCompression);
    case "MODERN_FORMAT_OR_COMPRESSION" /* MODERN_FORMAT_OR_COMPRESSION */:
      return i18nString(UIStrings.useModernFormat);
    case "VIDEO_FORMAT" /* VIDEO_FORMAT */:
      return i18nString(UIStrings.useVideoFormat);
    case "RESPONSIVE_SIZE" /* RESPONSIVE_SIZE */:
      return i18nString(UIStrings.useResponsiveSize, {
        PH1: `${optimization.fileDimensions.width}x${optimization.fileDimensions.height}`,
        PH2: `${optimization.displayDimensions.width}x${optimization.displayDimensions.height}`
      });
  }
}
export function getOptimizationMessageWithBytes(optimization) {
  const byteSavingsText = i18n.ByteUtilities.bytesToString(optimization.byteSavings);
  const optimizationMessage = getOptimizationMessage(optimization);
  return i18nString(UIStrings.estimatedSavings, { PH1: optimizationMessage, PH2: byteSavingsText });
}
function finalize(partialModel) {
  return {
    insightKey: InsightKeys.IMAGE_DELIVERY,
    strings: UIStrings,
    title: i18nString(UIStrings.title),
    description: i18nString(UIStrings.description),
    docs: "https://developer.chrome.com/docs/performance/insights/image-delivery",
    category: InsightCategory.LCP,
    state: partialModel.optimizableImages.length > 0 ? "fail" : "pass",
    ...partialModel,
    relatedEvents: new Map(partialModel.optimizableImages.map(
      (image) => [image.request, image.optimizations.map(getOptimizationMessageWithBytes)]
    ))
  };
}
function estimateGIFPercentSavings(request) {
  return Math.round(29.1 * Math.log10(request.args.data.decodedBodyLength) - 100.7) / 100;
}
function getDisplayedSize(data, paintImage) {
  return data.ImagePainting.paintEventToCorrectedDisplaySize.get(paintImage) ?? {
    width: paintImage.args.data.width,
    height: paintImage.args.data.height
  };
}
function getPixelCounts(data, paintImage) {
  const { width, height } = getDisplayedSize(data, paintImage);
  return {
    filePixels: paintImage.args.data.srcWidth * paintImage.args.data.srcHeight,
    displayedPixels: width * height
  };
}
export function generateInsight(data, context) {
  const isWithinContext = (event) => Helpers.Timing.eventIsInBounds(event, context.bounds);
  const contextRequests = data.NetworkRequests.byTime.filter(isWithinContext);
  const optimizableImages = [];
  for (const request of contextRequests) {
    if (request.args.data.resourceType !== "Image") {
      continue;
    }
    if (request.args.data.mimeType === "image/svg+xml") {
      continue;
    }
    const url = request.args.data.redirects[0]?.url ?? request.args.data.url;
    const imagePaints = data.ImagePainting.paintImageEventForUrl.get(url)?.filter(isWithinContext);
    if (!imagePaints?.length) {
      continue;
    }
    const largestImagePaint = imagePaints.reduce((prev, curr) => {
      const prevPixels = getPixelCounts(data, prev).displayedPixels;
      const currPixels = getPixelCounts(data, curr).displayedPixels;
      return prevPixels > currPixels ? prev : curr;
    });
    const {
      filePixels: imageFilePixels,
      displayedPixels: largestImageDisplayPixels
    } = getPixelCounts(data, largestImagePaint);
    const imageBytes = Math.min(request.args.data.decodedBodyLength, request.args.data.encodedDataLength);
    const bytesPerPixel = imageBytes / imageFilePixels;
    let optimizations = [];
    if (request.args.data.mimeType === "image/gif") {
      if (imageBytes > GIF_SIZE_THRESHOLD) {
        const percentSavings = estimateGIFPercentSavings(request);
        const byteSavings = Math.round(imageBytes * percentSavings);
        optimizations.push({ type: "VIDEO_FORMAT" /* VIDEO_FORMAT */, byteSavings });
      }
    } else if (bytesPerPixel > TARGET_BYTES_PER_PIXEL_AVIF) {
      const idealAvifImageSize = Math.round(TARGET_BYTES_PER_PIXEL_AVIF * imageFilePixels);
      const byteSavings = imageBytes - idealAvifImageSize;
      if (request.args.data.mimeType !== "image/webp" && request.args.data.mimeType !== "image/avif") {
        optimizations.push({ type: "MODERN_FORMAT_OR_COMPRESSION" /* MODERN_FORMAT_OR_COMPRESSION */, byteSavings });
      } else {
        optimizations.push({ type: "ADJUST_COMPRESSION" /* ADJUST_COMPRESSION */, byteSavings });
      }
    }
    const imageByteSavingsFromFormat = Math.max(0, ...optimizations.map((o) => o.byteSavings));
    let imageByteSavings = imageByteSavingsFromFormat;
    const wastedPixelRatio = 1 - largestImageDisplayPixels / imageFilePixels;
    if (wastedPixelRatio > 0 && !largestImagePaint.args.data.isCSS) {
      const byteSavings = Math.round(wastedPixelRatio * imageBytes);
      const hadBreakpoints = largestImagePaint.args.data.isPicture || largestImagePaint.args.data.srcsetAttribute;
      if (!hadBreakpoints || byteSavings > BYTE_SAVINGS_THRESHOLD_RESPONSIVE_BREAKPOINTS) {
        imageByteSavings += Math.round(wastedPixelRatio * (imageBytes - imageByteSavingsFromFormat));
        const { width, height } = getDisplayedSize(data, largestImagePaint);
        optimizations.push({
          type: "RESPONSIVE_SIZE" /* RESPONSIVE_SIZE */,
          byteSavings,
          fileDimensions: {
            width: Math.round(largestImagePaint.args.data.srcWidth),
            height: Math.round(largestImagePaint.args.data.srcHeight)
          },
          displayDimensions: {
            width: Math.round(width),
            height: Math.round(height)
          }
        });
      }
    }
    optimizations = optimizations.filter((optimization) => optimization.byteSavings > BYTE_SAVINGS_THRESHOLD);
    if (optimizations.length > 0) {
      optimizableImages.push({
        request,
        largestImagePaint,
        optimizations,
        byteSavings: imageByteSavings
      });
    }
  }
  const wastedBytesByRequestId = /* @__PURE__ */ new Map();
  for (const image of optimizableImages) {
    wastedBytesByRequestId.set(image.request.args.data.requestId, image.byteSavings);
  }
  optimizableImages.sort((a, b) => {
    if (b.byteSavings !== a.byteSavings) {
      return b.byteSavings - a.byteSavings;
    }
    return b.request.args.data.decodedBodyLength - a.request.args.data.decodedBodyLength;
  });
  return finalize({
    optimizableImages,
    metricSavings: metricSavingsForWastedBytes(wastedBytesByRequestId, context),
    wastedBytes: optimizableImages.reduce((total, img) => total + img.byteSavings, 0)
  });
}
export function createOverlayForRequest(request) {
  return {
    type: "ENTRY_OUTLINE",
    entry: request,
    outlineReason: "ERROR"
  };
}
export function createOverlays(model) {
  return model.optimizableImages.map((image) => createOverlayForRequest(image.request));
}
//# sourceMappingURL=ImageDelivery.js.map
