"use strict";
import { luminance } from "../front_end/core/common/ColorUtils.js";
import { createChild } from "./common.js";
import { applyMatrixToPoint, parseHexa } from "./highlight_common.js";
var GridArrowType = /* @__PURE__ */ ((GridArrowType2) => {
  GridArrowType2["LEFT_TOP"] = "left-top";
  GridArrowType2["LEFT_MID"] = "left-mid";
  GridArrowType2["LEFT_BOTTOM"] = "left-bottom";
  GridArrowType2["TOP_LEFT"] = "top-left";
  GridArrowType2["TOP_MID"] = "top-mid";
  GridArrowType2["TOP_RIGHT"] = "top-right";
  GridArrowType2["RIGHT_TOP"] = "right-top";
  GridArrowType2["RIGHT_MID"] = "right-mid";
  GridArrowType2["RIGHT_BOTTOM"] = "right-bottom";
  GridArrowType2["BOTTOM_LEFT"] = "bottom-left";
  GridArrowType2["BOTTOM_MID"] = "bottom-mid";
  GridArrowType2["BOTTOM_RIGHT"] = "bottom-right";
  return GridArrowType2;
})(GridArrowType || {});
const gridArrowWidth = 3;
const gridPageMargin = 20;
const gridLabelDistance = 20;
const maxLineNamesCount = 3;
const defaultLabelColor = "#1A73E8";
const defaultLabelTextColor = "#121212";
export function drawGridLabels(config, gridBounds, areaBounds, canvasSize, labelState, emulationScaleFactor, writingModeMatrix = new DOMMatrix()) {
  const labelContainerId = `grid-${labelState.gridLayerCounter++}-labels`;
  let labelContainerForNode = document.getElementById(labelContainerId);
  if (!labelContainerForNode) {
    const mainLabelLayerContainer = document.getElementById("grid-label-container");
    if (!mainLabelLayerContainer) {
      throw new Error("#grid-label-container is not found");
    }
    labelContainerForNode = createChild(mainLabelLayerContainer, "div");
    labelContainerForNode.id = labelContainerId;
  }
  const rowColor = config.gridHighlightConfig?.rowLineColor ? config.gridHighlightConfig.rowLineColor : defaultLabelColor;
  const rowTextColor = generateLegibleTextColor(rowColor);
  labelContainerForNode.style.setProperty("--row-label-color", rowColor);
  labelContainerForNode.style.setProperty("--row-label-text-color", rowTextColor);
  const columnColor = config.gridHighlightConfig?.columnLineColor ? config.gridHighlightConfig.columnLineColor : defaultLabelColor;
  const columnTextColor = generateLegibleTextColor(columnColor);
  labelContainerForNode.style.setProperty("--column-label-color", columnColor);
  labelContainerForNode.style.setProperty("--column-label-text-color", columnTextColor);
  labelContainerForNode.innerText = "";
  const areaNameContainer = createChild(labelContainerForNode, "div", "area-names");
  const lineNameContainer = createChild(labelContainerForNode, "div", "line-names");
  const lineNumberContainer = createChild(labelContainerForNode, "div", "line-numbers");
  const trackSizesContainer = createChild(labelContainerForNode, "div", "track-sizes");
  const normalizedData = normalizePositionData(config, gridBounds);
  if (config.gridHighlightConfig?.showLineNames) {
    drawGridLineNames(
      lineNameContainer,
      normalizedData,
      canvasSize,
      emulationScaleFactor,
      writingModeMatrix,
      config.writingMode
    );
  } else {
    drawGridLineNumbers(
      lineNumberContainer,
      normalizedData,
      canvasSize,
      emulationScaleFactor,
      writingModeMatrix,
      config.writingMode
    );
  }
  drawGridAreaNames(areaNameContainer, areaBounds, writingModeMatrix, config.writingMode);
  if (config.columnTrackSizes) {
    drawGridTrackSizes(
      trackSizesContainer,
      config.columnTrackSizes,
      "column",
      canvasSize,
      emulationScaleFactor,
      writingModeMatrix,
      config.writingMode
    );
  }
  if (config.rowTrackSizes) {
    drawGridTrackSizes(
      trackSizesContainer,
      config.rowTrackSizes,
      "row",
      canvasSize,
      emulationScaleFactor,
      writingModeMatrix,
      config.writingMode
    );
  }
}
function* positionIterator(positions, axis) {
  let lastEmittedPos = null;
  for (const [i, pos] of positions.entries()) {
    const isFirst = i === 0;
    const isLast = i === positions.length - 1;
    const isFarEnoughFromPrevious = Math.abs(pos[axis] - (lastEmittedPos ? lastEmittedPos[axis] : 0)) > gridLabelDistance;
    const isFarEnoughFromLast = !isLast && Math.abs(positions[positions.length - 1][axis] - pos[axis]) > gridLabelDistance;
    if (isFirst || isLast || isFarEnoughFromPrevious && isFarEnoughFromLast) {
      yield [i, pos];
      lastEmittedPos = pos;
    }
  }
}
const last = (array) => array[array.length - 1];
const first = (array) => array[0];
function normalizeNameData(namePositions) {
  const positions = [];
  const names = [];
  for (const { name, x, y } of namePositions) {
    const normalizedX = Math.round(x);
    const normalizedY = Math.round(y);
    const existingIndex = positions.findIndex(({ x: x2, y: y2 }) => x2 === normalizedX && y2 === normalizedY);
    if (existingIndex > -1) {
      names[existingIndex].push(name);
    } else {
      positions.push({ x: normalizedX, y: normalizedY });
      names.push([name]);
    }
  }
  return { positions, names };
}
export function normalizePositionData(config, bounds) {
  const width = Math.round(bounds.maxX - bounds.minX);
  const height = Math.round(bounds.maxY - bounds.minY);
  const data = {
    rows: {
      positive: { positions: [], hasFirst: false, hasLast: false },
      negative: { positions: [], hasFirst: false, hasLast: false }
    },
    columns: {
      positive: { positions: [], hasFirst: false, hasLast: false },
      negative: { positions: [], hasFirst: false, hasLast: false }
    },
    bounds: {
      minX: Math.round(bounds.minX),
      maxX: Math.round(bounds.maxX),
      minY: Math.round(bounds.minY),
      maxY: Math.round(bounds.maxY),
      allPoints: bounds.allPoints,
      width,
      height
    }
  };
  if (config.gridHighlightConfig?.showLineNames) {
    const rowData = normalizeNameData(config.rowLineNameOffsets || []);
    const positiveRows = {
      positions: rowData.positions,
      names: rowData.names,
      hasFirst: rowData.positions.length ? first(rowData.positions).y === data.bounds.minY : false,
      hasLast: rowData.positions.length ? last(rowData.positions).y === data.bounds.maxY : false
    };
    data.rows.positive = positiveRows;
    const columnData = normalizeNameData(config.columnLineNameOffsets || []);
    const positiveColumns = {
      positions: columnData.positions,
      names: columnData.names,
      hasFirst: columnData.positions.length ? first(columnData.positions).x === data.bounds.minX : false,
      hasLast: columnData.positions.length ? last(columnData.positions).x === data.bounds.maxX : false
    };
    data.columns.positive = positiveColumns;
  } else {
    const normalizeXY = ({ x, y }) => ({ x: Math.round(x), y: Math.round(y) });
    if (config.positiveRowLineNumberPositions) {
      data.rows.positive = {
        positions: config.positiveRowLineNumberPositions.map(normalizeXY),
        hasFirst: Math.round(first(config.positiveRowLineNumberPositions).y) === data.bounds.minY,
        hasLast: Math.round(last(config.positiveRowLineNumberPositions).y) === data.bounds.maxY
      };
    }
    if (config.negativeRowLineNumberPositions) {
      data.rows.negative = {
        positions: config.negativeRowLineNumberPositions.map(normalizeXY),
        hasFirst: Math.round(first(config.negativeRowLineNumberPositions).y) === data.bounds.minY,
        hasLast: Math.round(last(config.negativeRowLineNumberPositions).y) === data.bounds.maxY
      };
    }
    if (config.positiveColumnLineNumberPositions) {
      data.columns.positive = {
        positions: config.positiveColumnLineNumberPositions.map(normalizeXY),
        hasFirst: Math.round(first(config.positiveColumnLineNumberPositions).x) === data.bounds.minX,
        hasLast: Math.round(last(config.positiveColumnLineNumberPositions).x) === data.bounds.maxX
      };
    }
    if (config.negativeColumnLineNumberPositions) {
      data.columns.negative = {
        positions: config.negativeColumnLineNumberPositions.map(normalizeXY),
        hasFirst: Math.round(first(config.negativeColumnLineNumberPositions).x) === data.bounds.minX,
        hasLast: Math.round(last(config.negativeColumnLineNumberPositions).x) === data.bounds.maxX
      };
    }
  }
  return data;
}
export function drawGridLineNumbers(container, data, canvasSize, emulationScaleFactor, writingModeMatrix = new DOMMatrix(), writingMode = "horizontal-tb") {
  if (!data.columns.positive.names) {
    for (const [i, pos] of positionIterator(data.columns.positive.positions, "x")) {
      const element = createLabelElement(container, (i + 1).toString(), "column");
      placePositiveColumnLabel(
        element,
        applyMatrixToPoint(pos, writingModeMatrix),
        data,
        writingMode,
        canvasSize,
        emulationScaleFactor
      );
    }
  }
  if (!data.rows.positive.names) {
    for (const [i, pos] of positionIterator(data.rows.positive.positions, "y")) {
      const element = createLabelElement(container, (i + 1).toString(), "row");
      placePositiveRowLabel(
        element,
        applyMatrixToPoint(pos, writingModeMatrix),
        data,
        writingMode,
        canvasSize,
        emulationScaleFactor
      );
    }
  }
  for (const [i, pos] of positionIterator(data.columns.negative.positions, "x")) {
    const element = createLabelElement(container, (data.columns.negative.positions.length * -1 + i).toString(), "column");
    placeNegativeColumnLabel(
      element,
      applyMatrixToPoint(pos, writingModeMatrix),
      data,
      writingMode,
      canvasSize,
      emulationScaleFactor
    );
  }
  for (const [i, pos] of positionIterator(data.rows.negative.positions, "y")) {
    const element = createLabelElement(container, (data.rows.negative.positions.length * -1 + i).toString(), "row");
    placeNegativeRowLabel(
      element,
      applyMatrixToPoint(pos, writingModeMatrix),
      data,
      writingMode,
      canvasSize,
      emulationScaleFactor
    );
  }
}
export function drawGridTrackSizes(container, trackSizes, direction, canvasSize, emulationScaleFactor, writingModeMatrix = new DOMMatrix(), writingMode = "horizontal-tb") {
  const { main, cross } = getAxes(writingMode);
  const { crossSize } = getCanvasSizes(writingMode, canvasSize);
  for (const { x, y, computedSize, authoredSize } of trackSizes) {
    const point = applyMatrixToPoint({ x, y }, writingModeMatrix);
    const size = computedSize.toFixed(2);
    const formattedComputed = `${size.endsWith(".00") ? size.slice(0, -3) : size}px`;
    const element = createLabelElement(container, `${authoredSize ? authoredSize + "\xB7" : ""}${formattedComputed}`, direction);
    const labelSize = getLabelSize(element, writingMode);
    let flipIn = point[main] - labelSize.mainSize < gridPageMargin;
    if (direction === "column") {
      flipIn = writingMode === "vertical-rl" ? crossSize - point[cross] - labelSize.crossSize < gridPageMargin : point[cross] - labelSize.crossSize < gridPageMargin;
    }
    let arrowType = adaptArrowTypeForWritingMode(
      direction === "column" ? "bottom-mid" /* BOTTOM_MID */ : "right-mid" /* RIGHT_MID */,
      writingMode
    );
    arrowType = flipArrowTypeIfNeeded(arrowType, flipIn);
    placeLineLabel(element, arrowType, point.x, point.y, labelSize, emulationScaleFactor);
  }
}
export function drawGridLineNames(container, data, canvasSize, emulationScaleFactor, writingModeMatrix = new DOMMatrix(), writingMode = "horizontal-tb") {
  for (const [i, pos] of data.columns.positive.positions.entries()) {
    const names = data.columns.positive.names[i];
    const element = createLabelElement(container, makeLineNameLabelContent(names), "column");
    placePositiveColumnLabel(
      element,
      applyMatrixToPoint(pos, writingModeMatrix),
      data,
      writingMode,
      canvasSize,
      emulationScaleFactor
    );
  }
  for (const [i, pos] of data.rows.positive.positions.entries()) {
    const names = data.rows.positive.names[i];
    const element = createLabelElement(container, makeLineNameLabelContent(names), "row");
    placePositiveRowLabel(
      element,
      applyMatrixToPoint(pos, writingModeMatrix),
      data,
      writingMode,
      canvasSize,
      emulationScaleFactor
    );
  }
}
function makeLineNameLabelContent(names) {
  const content = document.createElement("ul");
  const namesToDisplay = names.slice(0, maxLineNamesCount);
  for (const name of namesToDisplay) {
    createChild(content, "li", "line-name").textContent = name;
  }
  return content;
}
export function drawGridAreaNames(container, areaBounds, writingModeMatrix = new DOMMatrix(), writingMode = "horizontal-tb") {
  for (const { name, bounds } of areaBounds) {
    const element = createLabelElement(container, name, "row");
    const { width, height } = getLabelSize(element, writingMode);
    const point = writingMode === "vertical-rl" || writingMode === "sideways-rl" ? bounds.allPoints[3] : writingMode === "sideways-lr" ? bounds.allPoints[1] : bounds.allPoints[0];
    const corner = applyMatrixToPoint(point, writingModeMatrix);
    const flipX = bounds.allPoints[1].x < bounds.allPoints[0].x;
    const flipY = bounds.allPoints[3].y < bounds.allPoints[0].y;
    element.style.left = corner.x - (flipX ? width : 0) + "px";
    element.style.top = corner.y - (flipY ? height : 0) + "px";
  }
}
function createLabelElement(container, textContent, direction) {
  const wrapper = createChild(container, "div");
  const element = createChild(wrapper, "div", "grid-label-content");
  element.dataset.direction = direction;
  if (typeof textContent === "string") {
    element.textContent = textContent;
  } else {
    element.appendChild(textContent);
  }
  return element;
}
function getLabelSideEdgePoints(gridBounds, direction, side) {
  const [p1, p2, p3, p4] = gridBounds.allPoints;
  if (direction === "row") {
    return side === "positive" ? { start: p1, end: p4 } : { start: p2, end: p3 };
  }
  return side === "positive" ? { start: p1, end: p2 } : { start: p4, end: p3 };
}
function getAxes(writingMode) {
  return isHorizontalWritingMode(writingMode) ? { main: "x", cross: "y" } : { main: "y", cross: "x" };
}
function getCanvasSizes(writingMode, canvasSize) {
  return isHorizontalWritingMode(writingMode) ? { mainSize: canvasSize.canvasWidth, crossSize: canvasSize.canvasHeight } : { mainSize: canvasSize.canvasHeight, crossSize: canvasSize.canvasWidth };
}
function placePositiveRowLabel(element, pos, data, writingMode, canvasSize, emulationScaleFactor) {
  const { start, end } = getLabelSideEdgePoints(data.bounds, "row", "positive");
  const { main, cross } = getAxes(writingMode);
  const { crossSize } = getCanvasSizes(writingMode, canvasSize);
  const labelSize = getLabelSize(element, writingMode);
  const isAtSharedStartCorner = pos[cross] === start[cross] && data.columns?.positive.hasFirst;
  const isAtSharedEndCorner = pos[cross] === end[cross] && data.columns?.negative.hasFirst;
  const isTooCloseToViewportStart = pos[cross] < gridPageMargin;
  const isTooCloseToViewportEnd = crossSize - pos[cross] < gridPageMargin;
  const flipIn = pos[main] - labelSize.mainSize < gridPageMargin;
  if (flipIn && (isAtSharedStartCorner || isAtSharedEndCorner)) {
    element.classList.add("inner-shared-corner");
  }
  let arrowType = adaptArrowTypeForWritingMode("right-mid" /* RIGHT_MID */, writingMode);
  if (isTooCloseToViewportStart || isAtSharedStartCorner) {
    arrowType = adaptArrowTypeForWritingMode("right-top" /* RIGHT_TOP */, writingMode);
  } else if (isTooCloseToViewportEnd || isAtSharedEndCorner) {
    arrowType = adaptArrowTypeForWritingMode("right-bottom" /* RIGHT_BOTTOM */, writingMode);
  }
  arrowType = flipArrowTypeIfNeeded(arrowType, flipIn);
  placeLineLabel(element, arrowType, pos.x, pos.y, labelSize, emulationScaleFactor);
}
function placeNegativeRowLabel(element, pos, data, writingMode, canvasSize, emulationScaleFactor) {
  const { start, end } = getLabelSideEdgePoints(data.bounds, "row", "negative");
  const { main, cross } = getAxes(writingMode);
  const { mainSize, crossSize } = getCanvasSizes(writingMode, canvasSize);
  const labelSize = getLabelSize(element, writingMode);
  const isAtSharedStartCorner = pos[cross] === start[cross] && data.columns?.positive.hasLast;
  const isAtSharedEndCorner = pos[cross] === end[cross] && data.columns?.negative.hasLast;
  const isTooCloseToViewportStart = pos[cross] < gridPageMargin;
  const isTooCloseToViewportEnd = crossSize - pos[cross] < gridPageMargin;
  const flipIn = mainSize - pos[main] - labelSize.mainSize < gridPageMargin;
  if (flipIn && (isAtSharedStartCorner || isAtSharedEndCorner)) {
    element.classList.add("inner-shared-corner");
  }
  let arrowType = adaptArrowTypeForWritingMode("left-mid" /* LEFT_MID */, writingMode);
  if (isTooCloseToViewportStart || isAtSharedStartCorner) {
    arrowType = adaptArrowTypeForWritingMode("left-top" /* LEFT_TOP */, writingMode);
  } else if (isTooCloseToViewportEnd || isAtSharedEndCorner) {
    arrowType = adaptArrowTypeForWritingMode("left-bottom" /* LEFT_BOTTOM */, writingMode);
  }
  arrowType = flipArrowTypeIfNeeded(arrowType, flipIn);
  placeLineLabel(element, arrowType, pos.x, pos.y, labelSize, emulationScaleFactor);
}
function placePositiveColumnLabel(element, pos, data, writingMode, canvasSize, emulationScaleFactor) {
  const { start, end } = getLabelSideEdgePoints(data.bounds, "column", "positive");
  const { main, cross } = getAxes(writingMode);
  const { mainSize, crossSize } = getCanvasSizes(writingMode, canvasSize);
  const labelSize = getLabelSize(element, writingMode);
  const isAtSharedStartCorner = pos[main] === start[main] && data.rows?.positive.hasFirst;
  const isAtSharedEndCorner = pos[main] === end[main] && data.rows?.negative.hasFirst;
  const isTooCloseToViewportStart = pos[main] < gridPageMargin;
  const isTooCloseToViewportEnd = mainSize - pos[main] < gridPageMargin;
  const flipIn = isFlippedBlocksWritingMode(writingMode) ? crossSize - pos[cross] - labelSize.crossSize < gridPageMargin : pos[cross] - labelSize.crossSize < gridPageMargin;
  if (flipIn && (isAtSharedStartCorner || isAtSharedEndCorner)) {
    element.classList.add("inner-shared-corner");
  }
  let arrowType = adaptArrowTypeForWritingMode("bottom-mid" /* BOTTOM_MID */, writingMode);
  if (isTooCloseToViewportStart) {
    arrowType = adaptArrowTypeForWritingMode("bottom-left" /* BOTTOM_LEFT */, writingMode);
  } else if (isTooCloseToViewportEnd) {
    arrowType = adaptArrowTypeForWritingMode("bottom-right" /* BOTTOM_RIGHT */, writingMode);
  }
  arrowType = flipArrowTypeIfNeeded(arrowType, flipIn);
  placeLineLabel(element, arrowType, pos.x, pos.y, labelSize, emulationScaleFactor);
}
function placeNegativeColumnLabel(element, pos, data, writingMode, canvasSize, emulationScaleFactor) {
  const { start, end } = getLabelSideEdgePoints(data.bounds, "column", "negative");
  const { main, cross } = getAxes(writingMode);
  const { mainSize, crossSize } = getCanvasSizes(writingMode, canvasSize);
  const labelSize = getLabelSize(element, writingMode);
  const isAtSharedStartCorner = pos[main] === start[main] && data.rows?.positive.hasLast;
  const isAtSharedEndCorner = pos[main] === end[main] && data.rows?.negative.hasLast;
  const isTooCloseToViewportStart = pos[main] < gridPageMargin;
  const isTooCloseToViewportEnd = mainSize - pos[main] < gridPageMargin;
  const flipIn = isFlippedBlocksWritingMode(writingMode) ? pos[cross] - labelSize.crossSize < gridPageMargin : crossSize - pos[cross] - labelSize.crossSize < gridPageMargin;
  if (flipIn && (isAtSharedStartCorner || isAtSharedEndCorner)) {
    element.classList.add("inner-shared-corner");
  }
  let arrowType = adaptArrowTypeForWritingMode("top-mid" /* TOP_MID */, writingMode);
  if (isTooCloseToViewportStart) {
    arrowType = adaptArrowTypeForWritingMode("top-left" /* TOP_LEFT */, writingMode);
  } else if (isTooCloseToViewportEnd) {
    arrowType = adaptArrowTypeForWritingMode("top-right" /* TOP_RIGHT */, writingMode);
  }
  arrowType = flipArrowTypeIfNeeded(arrowType, flipIn);
  placeLineLabel(element, arrowType, pos.x, pos.y, labelSize, emulationScaleFactor);
}
function placeLineLabel(element, arrowType, x, y, labelSize, emulationScaleFactor) {
  const { contentLeft, contentTop } = getLabelPositionByArrowType(arrowType, x, y, labelSize.width, labelSize.height, emulationScaleFactor);
  element.classList.add(arrowType);
  element.style.left = contentLeft + "px";
  element.style.top = contentTop + "px";
}
function getLabelSize(element, writingMode) {
  const width = getAdjustedLabelWidth(element);
  const height = element.getBoundingClientRect().height;
  const isHorizontal = isHorizontalWritingMode(writingMode);
  const mainSize = isHorizontal ? width : height;
  const crossSize = isHorizontal ? height : width;
  return { width, height, mainSize, crossSize };
}
function getAdjustedLabelWidth(element) {
  let labelWidth = element.getBoundingClientRect().width;
  if (labelWidth % 2 === 1) {
    labelWidth += 1;
    element.style.width = labelWidth + "px";
  }
  return labelWidth;
}
function flipArrowTypeIfNeeded(arrowType, flipIn) {
  if (!flipIn) {
    return arrowType;
  }
  switch (arrowType) {
    case "left-top" /* LEFT_TOP */:
      return "right-top" /* RIGHT_TOP */;
    case "left-mid" /* LEFT_MID */:
      return "right-mid" /* RIGHT_MID */;
    case "left-bottom" /* LEFT_BOTTOM */:
      return "right-bottom" /* RIGHT_BOTTOM */;
    case "right-top" /* RIGHT_TOP */:
      return "left-top" /* LEFT_TOP */;
    case "right-mid" /* RIGHT_MID */:
      return "left-mid" /* LEFT_MID */;
    case "right-bottom" /* RIGHT_BOTTOM */:
      return "left-bottom" /* LEFT_BOTTOM */;
    case "top-left" /* TOP_LEFT */:
      return "bottom-left" /* BOTTOM_LEFT */;
    case "top-mid" /* TOP_MID */:
      return "bottom-mid" /* BOTTOM_MID */;
    case "top-right" /* TOP_RIGHT */:
      return "bottom-right" /* BOTTOM_RIGHT */;
    case "bottom-left" /* BOTTOM_LEFT */:
      return "top-left" /* TOP_LEFT */;
    case "bottom-mid" /* BOTTOM_MID */:
      return "top-mid" /* TOP_MID */;
    case "bottom-right" /* BOTTOM_RIGHT */:
      return "top-right" /* TOP_RIGHT */;
  }
  return arrowType;
}
function adaptArrowTypeForWritingMode(arrowType, writingMode) {
  if (writingMode === "vertical-lr") {
    switch (arrowType) {
      case "left-top" /* LEFT_TOP */:
        return "top-left" /* TOP_LEFT */;
      case "left-mid" /* LEFT_MID */:
        return "top-mid" /* TOP_MID */;
      case "left-bottom" /* LEFT_BOTTOM */:
        return "top-right" /* TOP_RIGHT */;
      case "top-left" /* TOP_LEFT */:
        return "left-top" /* LEFT_TOP */;
      case "top-mid" /* TOP_MID */:
        return "left-mid" /* LEFT_MID */;
      case "top-right" /* TOP_RIGHT */:
        return "left-bottom" /* LEFT_BOTTOM */;
      case "right-top" /* RIGHT_TOP */:
        return "bottom-right" /* BOTTOM_RIGHT */;
      case "right-mid" /* RIGHT_MID */:
        return "bottom-mid" /* BOTTOM_MID */;
      case "right-bottom" /* RIGHT_BOTTOM */:
        return "bottom-left" /* BOTTOM_LEFT */;
      case "bottom-left" /* BOTTOM_LEFT */:
        return "right-top" /* RIGHT_TOP */;
      case "bottom-mid" /* BOTTOM_MID */:
        return "right-mid" /* RIGHT_MID */;
      case "bottom-right" /* BOTTOM_RIGHT */:
        return "right-bottom" /* RIGHT_BOTTOM */;
    }
  }
  if (writingMode === "vertical-rl" || writingMode === "sideways-rl") {
    switch (arrowType) {
      case "left-top" /* LEFT_TOP */:
        return "top-right" /* TOP_RIGHT */;
      case "left-mid" /* LEFT_MID */:
        return "top-mid" /* TOP_MID */;
      case "left-bottom" /* LEFT_BOTTOM */:
        return "top-left" /* TOP_LEFT */;
      case "top-left" /* TOP_LEFT */:
        return "right-top" /* RIGHT_TOP */;
      case "top-mid" /* TOP_MID */:
        return "right-mid" /* RIGHT_MID */;
      case "top-right" /* TOP_RIGHT */:
        return "right-bottom" /* RIGHT_BOTTOM */;
      case "right-top" /* RIGHT_TOP */:
        return "bottom-right" /* BOTTOM_RIGHT */;
      case "right-mid" /* RIGHT_MID */:
        return "bottom-mid" /* BOTTOM_MID */;
      case "right-bottom" /* RIGHT_BOTTOM */:
        return "bottom-left" /* BOTTOM_LEFT */;
      case "bottom-left" /* BOTTOM_LEFT */:
        return "left-top" /* LEFT_TOP */;
      case "bottom-mid" /* BOTTOM_MID */:
        return "left-mid" /* LEFT_MID */;
      case "bottom-right" /* BOTTOM_RIGHT */:
        return "left-bottom" /* LEFT_BOTTOM */;
    }
  }
  if (writingMode === "sideways-lr") {
    switch (arrowType) {
      case "left-top" /* LEFT_TOP */:
        return "bottom-left" /* BOTTOM_LEFT */;
      case "left-mid" /* LEFT_MID */:
        return "bottom-mid" /* BOTTOM_MID */;
      case "left-bottom" /* LEFT_BOTTOM */:
        return "bottom-right" /* BOTTOM_RIGHT */;
      case "top-left" /* TOP_LEFT */:
        return "left-bottom" /* LEFT_BOTTOM */;
      case "top-mid" /* TOP_MID */:
        return "left-mid" /* LEFT_MID */;
      case "top-right" /* TOP_RIGHT */:
        return "left-top" /* LEFT_TOP */;
      case "right-top" /* RIGHT_TOP */:
        return "top-left" /* TOP_LEFT */;
      case "right-mid" /* RIGHT_MID */:
        return "top-mid" /* TOP_MID */;
      case "right-bottom" /* RIGHT_BOTTOM */:
        return "top-right" /* TOP_RIGHT */;
      case "bottom-left" /* BOTTOM_LEFT */:
        return "right-bottom" /* RIGHT_BOTTOM */;
      case "bottom-mid" /* BOTTOM_MID */:
        return "right-mid" /* RIGHT_MID */;
      case "bottom-right" /* BOTTOM_RIGHT */:
        return "right-top" /* RIGHT_TOP */;
    }
  }
  return arrowType;
}
function getLabelPositionByArrowType(arrowType, x, y, labelWidth, labelHeight, emulationScaleFactor) {
  let contentTop = 0;
  let contentLeft = 0;
  x *= emulationScaleFactor;
  y *= emulationScaleFactor;
  switch (arrowType) {
    case "left-top" /* LEFT_TOP */:
      contentTop = y;
      contentLeft = x + gridArrowWidth;
      break;
    case "left-mid" /* LEFT_MID */:
      contentTop = y - labelHeight / 2;
      contentLeft = x + gridArrowWidth;
      break;
    case "left-bottom" /* LEFT_BOTTOM */:
      contentTop = y - labelHeight;
      contentLeft = x + gridArrowWidth;
      break;
    case "right-top" /* RIGHT_TOP */:
      contentTop = y;
      contentLeft = x - gridArrowWidth - labelWidth;
      break;
    case "right-mid" /* RIGHT_MID */:
      contentTop = y - labelHeight / 2;
      contentLeft = x - gridArrowWidth - labelWidth;
      break;
    case "right-bottom" /* RIGHT_BOTTOM */:
      contentTop = y - labelHeight;
      contentLeft = x - labelWidth - gridArrowWidth;
      break;
    case "top-left" /* TOP_LEFT */:
      contentTop = y + gridArrowWidth;
      contentLeft = x;
      break;
    case "top-mid" /* TOP_MID */:
      contentTop = y + gridArrowWidth;
      contentLeft = x - labelWidth / 2;
      break;
    case "top-right" /* TOP_RIGHT */:
      contentTop = y + gridArrowWidth;
      contentLeft = x - labelWidth;
      break;
    case "bottom-left" /* BOTTOM_LEFT */:
      contentTop = y - gridArrowWidth - labelHeight;
      contentLeft = x;
      break;
    case "bottom-mid" /* BOTTOM_MID */:
      contentTop = y - gridArrowWidth - labelHeight;
      contentLeft = x - labelWidth / 2;
      break;
    case "bottom-right" /* BOTTOM_RIGHT */:
      contentTop = y - gridArrowWidth - labelHeight;
      contentLeft = x - labelWidth;
      break;
  }
  return {
    contentTop,
    contentLeft
  };
}
export function generateLegibleTextColor(backgroundColor) {
  let rgb = [];
  const rgba = parseHexa(backgroundColor + "00");
  if (rgba.length === 4) {
    rgb = rgba.slice(0, 3).map((c) => c);
  } else {
    const parsed = backgroundColor.match(/[0-9.]+/g);
    if (!parsed) {
      return null;
    }
    rgb = parsed.slice(0, 3).map((s) => parseInt(s, 10) / 255);
  }
  if (!rgb.length) {
    return null;
  }
  return luminance(rgb) > 0.2 ? defaultLabelTextColor : "white";
}
export function isHorizontalWritingMode(writingMode) {
  return writingMode.startsWith("horizontal");
}
function isFlippedBlocksWritingMode(writingMode) {
  return writingMode === "vertical-rl" || writingMode === "sideways-rl";
}
//# sourceMappingURL=css_grid_label_helpers.js.map
