// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import './icons.html.js';
import './composebox_tab_favicon.js';
import '//resources/cr_elements/icons.html.js';
import '//resources/cr_elements/cr_icon/cr_icon.js';
import '//resources/cr_elements/cr_action_menu/cr_action_menu.js';
import '//resources/cr_elements/cr_button/cr_button.js';
import { AnchorAlignment } from '//resources/cr_elements/cr_action_menu/cr_action_menu.js';
import { I18nMixinLit } from '//resources/cr_elements/i18n_mixin_lit.js';
import { assert } from '//resources/js/assert.js';
import { loadTimeData } from '//resources/js/load_time_data.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { getCss } from './context_menu_entrypoint.css.js';
import { getHtml } from './context_menu_entrypoint.html.js';
/** The width of the dropdown menu in pixels. */
const MENU_WIDTH_PX = 190;
const ContextMenuEntrypointElementBase = I18nMixinLit(CrLitElement);
export class ContextMenuEntrypointElement extends ContextMenuEntrypointElementBase {
    static get is() {
        return 'composebox-context-menu-entrypoint';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            // =========================================================================
            // Public properties
            // =========================================================================
            inputsDisabled: { type: Boolean },
            fileNum: { type: Number },
            showContextMenuDescription: { type: Boolean },
            inCreateImageMode: {
                reflect: true,
                type: Boolean,
            },
            hasImageFiles: {
                reflect: true,
                type: Boolean,
            },
            disabledTabIds: { type: Object },
            tabSuggestions: { type: Array },
            entrypointName: { type: String },
            // =========================================================================
            // Protected properties
            // =========================================================================
            tabPreviewUrl_: { type: String },
            tabPreviewsEnabled_: { type: Boolean },
            showDeepSearch_: {
                reflect: true,
                type: Boolean,
            },
            showCreateImage_: {
                reflect: true,
                type: Boolean,
            },
        };
    }
    #inputsDisabled_accessor_storage = false;
    get inputsDisabled() { return this.#inputsDisabled_accessor_storage; }
    set inputsDisabled(value) { this.#inputsDisabled_accessor_storage = value; }
    #fileNum_accessor_storage = 0;
    get fileNum() { return this.#fileNum_accessor_storage; }
    set fileNum(value) { this.#fileNum_accessor_storage = value; }
    #showContextMenuDescription_accessor_storage = false;
    get showContextMenuDescription() { return this.#showContextMenuDescription_accessor_storage; }
    set showContextMenuDescription(value) { this.#showContextMenuDescription_accessor_storage = value; }
    #inCreateImageMode_accessor_storage = false;
    get inCreateImageMode() { return this.#inCreateImageMode_accessor_storage; }
    set inCreateImageMode(value) { this.#inCreateImageMode_accessor_storage = value; }
    #hasImageFiles_accessor_storage = false;
    get hasImageFiles() { return this.#hasImageFiles_accessor_storage; }
    set hasImageFiles(value) { this.#hasImageFiles_accessor_storage = value; }
    #disabledTabIds_accessor_storage = new Set();
    get disabledTabIds() { return this.#disabledTabIds_accessor_storage; }
    set disabledTabIds(value) { this.#disabledTabIds_accessor_storage = value; }
    #tabSuggestions_accessor_storage = [];
    get tabSuggestions() { return this.#tabSuggestions_accessor_storage; }
    set tabSuggestions(value) { this.#tabSuggestions_accessor_storage = value; }
    #entrypointName_accessor_storage = '';
    get entrypointName() { return this.#entrypointName_accessor_storage; }
    set entrypointName(value) { this.#entrypointName_accessor_storage = value; }
    #tabPreviewUrl__accessor_storage = '';
    get tabPreviewUrl_() { return this.#tabPreviewUrl__accessor_storage; }
    set tabPreviewUrl_(value) { this.#tabPreviewUrl__accessor_storage = value; }
    #tabPreviewsEnabled__accessor_storage = loadTimeData.getBoolean('composeboxShowContextMenuTabPreviews');
    get tabPreviewsEnabled_() { return this.#tabPreviewsEnabled__accessor_storage; }
    set tabPreviewsEnabled_(value) { this.#tabPreviewsEnabled__accessor_storage = value; }
    #showDeepSearch__accessor_storage = loadTimeData.getBoolean('composeboxShowDeepSearchButton');
    get showDeepSearch_() { return this.#showDeepSearch__accessor_storage; }
    set showDeepSearch_(value) { this.#showDeepSearch__accessor_storage = value; }
    #showCreateImage__accessor_storage = loadTimeData.getBoolean('composeboxShowCreateImageButton');
    get showCreateImage_() { return this.#showCreateImage__accessor_storage; }
    set showCreateImage_(value) { this.#showCreateImage__accessor_storage = value; }
    maxFileCount_ = loadTimeData.getInteger('composeboxFileMaxCount');
    constructor() {
        super();
    }
    // Checks if the image upload item in the context menu should be disabled.
    get imageUploadDisabled_() {
        return this.fileNum >= this.maxFileCount_ ||
            (this.inCreateImageMode && this.hasImageFiles);
    }
    // Checks if the file upload item in the context menu should be disabled.
    get fileUploadDisabled_() {
        return this.inCreateImageMode || this.fileNum >= this.maxFileCount_;
    }
    // Checks if the deep search item in the context menu should be disabled.
    get deepSearchDisabled_() {
        return this.inCreateImageMode || this.fileNum === 1 || this.fileNum > 1;
    }
    // Checks if the create image item in the context menu should be disabled.
    get createImageDisabled_() {
        return this.inCreateImageMode || this.fileNum > 1 ||
            ((this.fileNum === 1) && !this.hasImageFiles);
    }
    // Checks if a tab item in the context menu should be disabled.
    isTabDisabled_(tab) {
        return this.inCreateImageMode || this.fileNum >= this.maxFileCount_ ||
            this.disabledTabIds.has(tab.tabId);
    }
    onEntrypointClick_() {
        const metricName = 'NewTabPage.' + this.entrypointName + '.ContextMenuEntry.Clicked';
        chrome.metricsPrivate.recordBoolean(metricName, true);
        const entrypoint = this.shadowRoot.querySelector('#entrypoint');
        assert(entrypoint);
        this.$.menu.showAt(entrypoint, {
            top: entrypoint.getBoundingClientRect().bottom,
            width: MENU_WIDTH_PX,
            anchorAlignmentX: AnchorAlignment['AFTER_START'],
        });
    }
    addTabContext_(e) {
        e.stopPropagation();
        const tabElement = e.currentTarget;
        const tabInfo = this.tabSuggestions[Number(tabElement.dataset['index'])];
        assert(tabInfo);
        this.fire('add-tab-context', {
            id: tabInfo.tabId,
            title: tabInfo.title,
            url: tabInfo.url,
        });
        this.$.menu.close();
    }
    onTabPointerenter_(e) {
        if (!this.tabPreviewsEnabled_) {
            return;
        }
        const tabElement = e.currentTarget;
        const tabInfo = this.tabSuggestions[Number(tabElement.dataset['index'])];
        assert(tabInfo);
        // Clear the preview URL before fetching the new one to make sure an old
        // or incorrect preview doesn't show while the new one is loading.
        this.tabPreviewUrl_ = '';
        this.fire('get-tab-preview', {
            tabId: tabInfo.tabId,
            onPreviewFetched: (previewDataUrl) => {
                this.tabPreviewUrl_ = previewDataUrl;
            },
        });
    }
    shouldShowTabPreview_() {
        return this.tabPreviewsEnabled_ && this.tabPreviewUrl_ !== '';
    }
    openImageUpload_() {
        this.fire('open-image-upload');
        this.$.menu.close();
    }
    openFileUpload_() {
        this.fire('open-file-upload');
        this.$.menu.close();
    }
    onDeepSearchClick_() {
        this.fire('deep-search-click');
        this.$.menu.close();
    }
    onCreateImageClick_() {
        this.fire('create-image-click');
        this.$.menu.close();
    }
}
customElements.define(ContextMenuEntrypointElement.is, ContextMenuEntrypointElement);
